/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.index.store.remote.utils;

import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.AccessController;
import java.security.PrivilegedActionException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.opensearch.index.store.remote.filecache.CachedIndexInput;
import org.opensearch.index.store.remote.filecache.FileCache;
import org.opensearch.index.store.remote.filecache.FileCachedIndexInput;
import org.opensearch.index.store.remote.utils.BlobFetchRequest;

public class TransferManager {
    private static final Logger logger = LogManager.getLogger(TransferManager.class);
    private final StreamReader streamReader;
    private final FileCache fileCache;

    public TransferManager(StreamReader streamReader, FileCache fileCache) {
        this.streamReader = streamReader;
        this.fileCache = fileCache;
    }

    public IndexInput fetchBlob(BlobFetchRequest blobFetchRequest) throws IOException {
        Path key = blobFetchRequest.getFilePath();
        logger.trace("fetchBlob called for {}", (Object)key.toString());
        try {
            return AccessController.doPrivileged(() -> {
                CachedIndexInput cacheEntry = this.fileCache.compute(key, (path, cachedIndexInput) -> {
                    if (cachedIndexInput == null || cachedIndexInput.isClosed()) {
                        logger.trace("Transfer Manager - IndexInput closed or not in cache");
                        return new DelayedCreationCachedIndexInput(this.fileCache, this.streamReader, blobFetchRequest);
                    }
                    logger.trace("Transfer Manager - Already in cache");
                    return cachedIndexInput;
                });
                try {
                    IndexInput indexInput = cacheEntry.getIndexInput().clone();
                    return indexInput;
                }
                finally {
                    this.fileCache.decRef(key);
                }
            });
        }
        catch (PrivilegedActionException e) {
            Exception cause = e.getException();
            if (cause instanceof IOException) {
                throw (IOException)cause;
            }
            if (cause instanceof RuntimeException) {
                throw (RuntimeException)cause;
            }
            throw new IOException(cause);
        }
    }

    private static FileCachedIndexInput createIndexInput(FileCache fileCache, StreamReader streamReader, BlobFetchRequest request) {
        try {
            if (fileCache.capacity() < fileCache.usage().usage()) {
                fileCache.prune();
                throw new IOException("Local file cache capacity (" + fileCache.capacity() + ") exceeded (" + fileCache.usage().usage() + ") - BlobFetchRequest failed: " + String.valueOf(request.getFilePath()));
            }
            if (!Files.exists(request.getFilePath(), new LinkOption[0])) {
                logger.trace("Fetching from Remote in createIndexInput of Transfer Manager");
                try (OutputStream fileOutputStream = Files.newOutputStream(request.getFilePath(), new OpenOption[0]);
                     BufferedOutputStream localFileOutputStream = new BufferedOutputStream(fileOutputStream);){
                    for (BlobFetchRequest.BlobPart blobPart : request.blobParts()) {
                        InputStream snapshotFileInputStream = streamReader.read(blobPart.getBlobName(), blobPart.getPosition(), blobPart.getLength());
                        try {
                            snapshotFileInputStream.transferTo(localFileOutputStream);
                        }
                        finally {
                            if (snapshotFileInputStream == null) continue;
                            snapshotFileInputStream.close();
                        }
                    }
                }
            }
            IndexInput luceneIndexInput = request.getDirectory().openInput(request.getFileName(), IOContext.DEFAULT);
            return new FileCachedIndexInput(fileCache, request.getFilePath(), luceneIndexInput);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    @FunctionalInterface
    public static interface StreamReader {
        public InputStream read(String var1, long var2, long var4) throws IOException;
    }

    private static class DelayedCreationCachedIndexInput
    implements CachedIndexInput {
        private final FileCache fileCache;
        private final StreamReader streamReader;
        private final BlobFetchRequest request;
        private final CompletableFuture<IndexInput> result = new CompletableFuture();
        private final AtomicBoolean isStarted = new AtomicBoolean(false);
        private final AtomicBoolean isClosed = new AtomicBoolean(false);

        private DelayedCreationCachedIndexInput(FileCache fileCache, StreamReader streamReader, BlobFetchRequest request) {
            this.fileCache = fileCache;
            this.streamReader = streamReader;
            this.request = request;
        }

        @Override
        public IndexInput getIndexInput() throws IOException {
            if (this.isClosed.get()) {
                throw new IllegalStateException("Already closed");
            }
            if (!this.isStarted.getAndSet(true)) {
                try {
                    this.result.complete(TransferManager.createIndexInput(this.fileCache, this.streamReader, this.request));
                }
                catch (Exception e) {
                    this.result.completeExceptionally(e);
                    this.fileCache.remove(this.request.getFilePath());
                }
            }
            try {
                return this.result.join();
            }
            catch (CompletionException e) {
                if (e.getCause() instanceof UncheckedIOException) {
                    throw ((UncheckedIOException)e.getCause()).getCause();
                }
                if (e.getCause() instanceof RuntimeException) {
                    throw (RuntimeException)e.getCause();
                }
                throw e;
            }
        }

        @Override
        public long length() {
            return this.request.getBlobLength();
        }

        @Override
        public boolean isClosed() {
            return this.isClosed.get();
        }

        @Override
        public void close() throws Exception {
            if (!this.isClosed.getAndSet(true)) {
                this.result.whenComplete((indexInput, error) -> {
                    if (indexInput != null) {
                        try {
                            indexInput.close();
                        }
                        catch (IOException e) {
                            logger.warn("Error closing IndexInput", (Throwable)e);
                        }
                    }
                });
            }
        }
    }
}

