"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.configRoutes = configRoutes;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helper = require("../utils/helper");
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../common/constants");
var _MDSEnabledClientService = require("../../common/MDSEnabledClientService");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function configRoutes(router, dataSourceEnabled) {
  const getConfigsQuerySchema = {
    from_index: _configSchema.schema.number(),
    max_items: _configSchema.schema.number(),
    query: _configSchema.schema.maybe(_configSchema.schema.string()),
    config_type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]),
    is_enabled: _configSchema.schema.maybe(_configSchema.schema.boolean()),
    sort_field: _configSchema.schema.string(),
    sort_order: _configSchema.schema.string(),
    config_id_list: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])),
    'smtp_account.method': _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()]))
  };
  if (dataSourceEnabled) {
    getConfigsQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  const genericBodyAndDataSourceIdQuery = {
    body: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericBodyAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const genericParamsAndDataSourceIdQuery = {
    params: _configSchema.schema.any()
  };
  if (dataSourceEnabled) {
    genericParamsAndDataSourceIdQuery.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const updateQuerySchema = {
    body: _configSchema.schema.any(),
    params: _configSchema.schema.object({
      configId: _configSchema.schema.string()
    })
  };
  if (dataSourceEnabled) {
    updateQuerySchema.query = _configSchema.schema.object({
      dataSourceId: _configSchema.schema.string()
    });
  }
  const deleteQuerySchema = {
    config_id_list: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string()), _configSchema.schema.string()])
  };
  if (dataSourceEnabled) {
    deleteQuerySchema.dataSourceId = _configSchema.schema.string();
  }
  router.get({
    path: _common.NODE_API.GET_CONFIGS,
    validate: {
      query: _configSchema.schema.object(getConfigsQuerySchema)
    }
  }, async (context, request, response) => {
    const config_type = (0, _helper.joinRequestParams)(request.query.config_type);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    const encryption_method = (0, _helper.joinRequestParams)(request.query['smtp_account.method']);
    const query = request.query.query;
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigs', {
        from_index: request.query.from_index,
        max_items: request.query.max_items,
        is_enabled: request.query.is_enabled,
        sort_field: request.query.sort_field,
        sort_order: request.query.sort_order,
        config_type,
        ...(query && {
          text_query: query
        }),
        // text_query will exclude keyword fields
        ...(config_id_list && {
          config_id_list
        }),
        ...(encryption_method && {
          'smtp_account.method': encryption_method
        })
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 404,
        body: error.message
      });
    }
  });
  router.get({
    path: `${_common.NODE_API.GET_CONFIG}/{configId}`,
    validate: genericParamsAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getConfigById', {
        configId: request.params.configId
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _common.NODE_API.CREATE_CONFIG,
    validate: genericBodyAndDataSourceIdQuery
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.createConfig', {
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_common.NODE_API.UPDATE_CONFIG}/{configId}`,
    validate: updateQuerySchema
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.updateConfigById', {
        configId: request.params.configId,
        body: request.body
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: _common.NODE_API.DELETE_CONFIGS,
    validate: {
      query: _configSchema.schema.object(deleteQuerySchema)
    }
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    const config_id_list = (0, _helper.joinRequestParams)(request.query.config_id_list);
    try {
      const resp = await client('notifications.deleteConfigs', {
        config_id_list
      });
      return response.ok({
        body: resp
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get({
    path: _common.NODE_API.GET_AVAILABLE_FEATURES,
    validate: dataSourceEnabled ? {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.string()
      })
    } : false
  }, async (context, request, response) => {
    const client = _MDSEnabledClientService.MDSEnabledClientService.getClient(request, context, dataSourceEnabled);
    try {
      const resp = await client('notifications.getServerFeatures');
      const config_type_list = resp.allowed_config_type_list;
      const channelTypes = {};
      for (let channel of config_type_list) {
        if (_constants.CHANNEL_TYPE[channel]) {
          channelTypes[channel] = _constants.CHANNEL_TYPE[channel];
        }
      }
      const availableFeature = {
        availableChannels: channelTypes,
        availableConfigTypes: config_type_list,
        tooltipSupport: _lodash.default.get(response, ['plugin_features', 'tooltip_support']) === 'true'
      };
      return response.ok({
        body: availableFeature
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 404,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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