"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerVisualizationSummaryRoute = registerVisualizationSummaryRoute;
var _configSchema = require("@osd/config-schema");
var _utils = require("../utils");
var _ml_commons = require("../../../common/constants/ml_commons");
var _notebooks = require("../../../common/constants/notebooks");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

const ML_CONFIG_NAME = 'os_visualization_summary';

/**
 * Route handler for generating visualization summaries using ML models
 *
 * This endpoint:
 * 1. Receives a base64-encoded visualization image
 * 2. Retrieves the ML model ID from the ML config API
 * 3. Calls the ML model predict API to generate a summary
 *
 * @param router - OpenSearch Dashboards router instance
 */
function registerVisualizationSummaryRoute(router) {
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/visualization/summary`,
    validate: {
      body: _configSchema.schema.object({
        visualization: _configSchema.schema.string({
          minLength: 1
        }),
        localTimeZoneOffset: _configSchema.schema.number()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        visualization,
        localTimeZoneOffset
      } = request.body;
      const {
        dataSourceId
      } = request.query;

      // Get transport client with data source support
      const transport = await (0, _utils.getOpenSearchClientTransport)({
        context,
        dataSourceId,
        request
      });

      // Step 1: Get model ID from ML config API
      let agentId;
      try {
        var _configBody$configura;
        const configResponse = await transport.request({
          method: 'GET',
          path: `${_ml_commons.OPENSEARCH_ML_COMMONS_API_PREFIX}/config/${ML_CONFIG_NAME}`
        });

        // Extract agent ID from config response
        const configBody = configResponse.body;
        agentId = configBody === null || configBody === void 0 || (_configBody$configura = configBody.configuration) === null || _configBody$configura === void 0 ? void 0 : _configBody$configura.agent_id;
        if (!agentId) {
          return response.notFound({
            body: {
              message: `Agent not found.`
            }
          });
        }
      } catch (configError) {
        return response.customError({
          statusCode: configError.statusCode || 500,
          body: {
            message: `Failed to retrieve ML config '${ML_CONFIG_NAME}': ${configError.message}`
          }
        });
      }

      // Step 2: Call ML model predict API with the visualization
      try {
        var _predictBody$inferenc, _resultJson;
        const predictResponse = await transport.request({
          method: 'POST',
          path: `${_ml_commons.OPENSEARCH_ML_COMMONS_API_PREFIX}/agents/${agentId}/_execute`,
          body: {
            parameters: {
              image_base64: visualization,
              local_time_offset: localTimeZoneOffset
            }
          }
        });
        const predictBody = predictResponse.body;

        // Extract summary from prediction response
        // The response structure may vary depending on the model
        const resultString = predictBody === null || predictBody === void 0 || (_predictBody$inferenc = predictBody.inference_results) === null || _predictBody$inferenc === void 0 || (_predictBody$inferenc = _predictBody$inferenc[0]) === null || _predictBody$inferenc === void 0 || (_predictBody$inferenc = _predictBody$inferenc.output) === null || _predictBody$inferenc === void 0 || (_predictBody$inferenc = _predictBody$inferenc[0]) === null || _predictBody$inferenc === void 0 ? void 0 : _predictBody$inferenc.result;
        if (!resultString) {
          return response.customError({
            statusCode: 500,
            body: {
              message: `Invalid ML response structure: missing result field`
            }
          });
        }
        let resultJson;
        try {
          resultJson = JSON.parse(resultString);
        } catch (parseError) {
          return response.customError({
            statusCode: 500,
            body: {
              message: `Failed to parse ML response: invalid JSON`
            }
          });
        }
        const summaryText = (_resultJson = resultJson) === null || _resultJson === void 0 || (_resultJson = _resultJson.output) === null || _resultJson === void 0 || (_resultJson = _resultJson.message) === null || _resultJson === void 0 || (_resultJson = _resultJson.content) === null || _resultJson === void 0 || (_resultJson = _resultJson[0]) === null || _resultJson === void 0 ? void 0 : _resultJson.text;
        if (!summaryText) {
          return response.customError({
            statusCode: 500,
            body: {
              message: `Invalid ML response structure: missing summary text`
            }
          });
        }
        return response.ok({
          body: {
            summary: summaryText
          }
        });
      } catch (predictError) {
        return response.customError({
          statusCode: predictError.statusCode || 500,
          body: {
            message: `Failed to generate visualization summary using agent '${agentId}': ${predictError.message}`
          }
        });
      }
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: {
          message: `Unexpected error generating visualization summary: ${error.message}`
        }
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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