"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerParaRoute = registerParaRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _saved_objects_paragraphs_router = require("../../adaptors/notebooks/saved_objects_paragraphs_router");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const paragraphInputValidation = _configSchema.schema.object({
  inputText: _configSchema.schema.string(),
  inputType: _configSchema.schema.string(),
  parameters: _configSchema.schema.maybe(_configSchema.schema.object({}, {
    unknowns: 'allow'
  }))
});
const paragraphOutputValidation = _configSchema.schema.arrayOf(_configSchema.schema.object({
  outputType: _configSchema.schema.string(),
  result: _configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.object({}, {
    unknowns: 'allow'
  })]),
  execution_time: _configSchema.schema.maybe(_configSchema.schema.string())
}));
function registerParaRoute(router) {
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIndex: _configSchema.schema.number(),
        input: paragraphInputValidation,
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string()),
        aiGenerated: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.createParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      query: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const params = {
      noteId: request.query.noteId,
      paragraphId: request.query.paragraphId
    };
    try {
      const deleteResponse = await (0, _saved_objects_paragraphs_router.deleteParagraphs)(params, context.core.savedObjects.client);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph/update/run`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        input: paragraphInputValidation,
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      const runResponse = await (0, _saved_objects_paragraphs_router.updateRunFetchParagraph)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraph`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphId: _configSchema.schema.string(),
        input: paragraphInputValidation,
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        })),
        output: _configSchema.schema.maybe(paragraphOutputValidation)
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        output,
        ...others
      } = request.body;
      const saveResponse = await (0, _saved_objects_paragraphs_router.updateFetchParagraph)({
        ...others,
        ...(output ? {
          output: [output[0]]
        } : {})
      }, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraphs`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      const deleteResponse = await (0, _saved_objects_paragraphs_router.batchDeleteParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: deleteResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraphs/batch`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        startIndex: _configSchema.schema.number(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          input: paragraphInputValidation,
          dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string()),
          aiGenerated: _configSchema.schema.maybe(_configSchema.schema.boolean())
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      const batchResponse = await (0, _saved_objects_paragraphs_router.batchCreateParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: batchResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraphs/batch/run`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          id: _configSchema.schema.string(),
          input: paragraphInputValidation,
          dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string())
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      const runResponse = await (0, _saved_objects_paragraphs_router.batchRunParagraphs)(request.body, context.core.savedObjects.client);
      return response.ok({
        body: runResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/paragraphs/batch`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          paragraphId: _configSchema.schema.string(),
          input: paragraphInputValidation,
          dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string()),
          output: _configSchema.schema.maybe(paragraphOutputValidation)
        }))
      })
    }
  }, async (context, request, response) => {
    try {
      const saveResponse = await (0, _saved_objects_paragraphs_router.batchSaveParagraphs)({
        ...request.body,
        paragraphs: request.body.paragraphs.map(p => ({
          ...p,
          output: p.output ? [p.output[0]] : undefined
        }))
      }, context.core.savedObjects.client);
      return response.ok({
        body: saveResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/set_paragraphs`,
    validate: {
      body: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        paragraphs: _configSchema.schema.arrayOf(_configSchema.schema.object({
          output: _configSchema.schema.maybe(paragraphOutputValidation),
          input: paragraphInputValidation,
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          id: _configSchema.schema.string(),
          dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
            defaultValue: ''
          }))
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const updateNotebook = {
        paragraphs: request.body.paragraphs,
        dateModified: new Date().toISOString()
      };
      const updateResponse = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId, {
        savedNotebook: updateNotebook
      });
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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