"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerNoteRoute = registerNoteRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _saved_objects_notebooks_router = require("../../adaptors/notebooks/saved_objects_notebooks_router");
var _get_set = require("../../../server/services/get_set");
var _saved_objects_paragraphs_router = require("../../adaptors/notebooks/saved_objects_paragraphs_router");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerNoteRoute(router, auth) {
  const getUserName = request => {
    var _authInfo$state;
    const authInfo = auth.get(request);
    return authInfo === null || authInfo === void 0 || (_authInfo$state = authInfo.state) === null || _authInfo$state === void 0 || (_authInfo$state = _authInfo$state.authInfo) === null || _authInfo$state === void 0 ? void 0 : _authInfo$state.user_name;
  };
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const notebooksData = await opensearchNotebooksClient.find({
        type: _observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT,
        perPage: 1000
      });
      const capabilities = await (0, _get_set.getCapabilities)().resolveCapabilities(request);
      const agenticFeaturesEnabled = capabilities.investigation.agenticFeaturesEnabled;
      const fetchedNotebooks = (0, _saved_objects_notebooks_router.fetchNotebooks)(notebooksData.saved_objects, agenticFeaturesEnabled);
      return response.ok({
        body: {
          data: fetchedNotebooks
        }
      });
    } catch (error) {
      console.log('Notebook:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        context: _configSchema.schema.maybe(_configSchema.schema.any())
      })
    }
  }, async (context, request, response) => {
    const userName = getUserName(request);
    const opensearchNotebooksClient = context.core.savedObjects.client;
    let notebooksData;
    try {
      const newNotebookObject = (0, _saved_objects_notebooks_router.createNotebook)(request.body, userName);
      notebooksData = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, newNotebookObject);
      return response.ok({
        body: `${notebooksData.id}`
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/updateNotebookContext`,
    validate: {
      body: _configSchema.schema.object({
        notebookId: _configSchema.schema.string(),
        context: _configSchema.schema.maybe(_configSchema.schema.any())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const {
        notebookId,
        context: newContext
      } = request.body;
      const noteObject = {
        context: newContext,
        dateModified: new Date().toISOString()
      };
      const updatedNotebook = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, notebookId, {
        savedNotebook: noteObject
      });
      return response.ok({
        body: {
          id: updatedNotebook.id
        }
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to update notebook context: ${error.message}`,
          error: error.name
        }
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/updateHypotheses`,
    validate: {
      body: _configSchema.schema.object({
        notebookId: _configSchema.schema.string(),
        hypotheses: _configSchema.schema.arrayOf(_configSchema.schema.object({
          id: _configSchema.schema.string(),
          title: _configSchema.schema.string(),
          description: _configSchema.schema.string(),
          likelihood: _configSchema.schema.number(),
          supportingFindingParagraphIds: _configSchema.schema.arrayOf(_configSchema.schema.string()),
          irrelevantFindingParagraphIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          userSelectedFindingParagraphIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          newAddedFindingIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          dateCreated: _configSchema.schema.string(),
          dateModified: _configSchema.schema.string(),
          status: _configSchema.schema.maybe(_configSchema.schema.string())
        })),
        runningMemory: _configSchema.schema.nullable(_configSchema.schema.object({
          executorMemoryId: _configSchema.schema.maybe(_configSchema.schema.string()),
          parentInteractionId: _configSchema.schema.maybe(_configSchema.schema.string()),
          memoryContainerId: _configSchema.schema.maybe(_configSchema.schema.string()),
          owner: _configSchema.schema.maybe(_configSchema.schema.string())
        })),
        historyMemory: _configSchema.schema.nullable(_configSchema.schema.object({
          executorMemoryId: _configSchema.schema.maybe(_configSchema.schema.string()),
          parentInteractionId: _configSchema.schema.maybe(_configSchema.schema.string()),
          memoryContainerId: _configSchema.schema.maybe(_configSchema.schema.string()),
          owner: _configSchema.schema.maybe(_configSchema.schema.string())
        })),
        topologies: _configSchema.schema.nullable(_configSchema.schema.arrayOf(_configSchema.schema.object({
          id: _configSchema.schema.string(),
          description: _configSchema.schema.string(),
          traceId: _configSchema.schema.string(),
          hypothesisIds: _configSchema.schema.arrayOf(_configSchema.schema.string()),
          nodes: _configSchema.schema.arrayOf(_configSchema.schema.object({
            id: _configSchema.schema.string(),
            name: _configSchema.schema.string(),
            startTime: _configSchema.schema.string(),
            duration: _configSchema.schema.string(),
            status: _configSchema.schema.string(),
            parentId: _configSchema.schema.nullable(_configSchema.schema.string())
          }))
        }))),
        failedInvestigation: _configSchema.schema.maybe(_configSchema.schema.nullable(_configSchema.schema.object({
          error: _configSchema.schema.object({
            message: _configSchema.schema.string(),
            name: _configSchema.schema.maybe(_configSchema.schema.string()),
            cause: _configSchema.schema.maybe(_configSchema.schema.any())
          }),
          memory: _configSchema.schema.object({
            executorMemoryId: _configSchema.schema.maybe(_configSchema.schema.string()),
            parentInteractionId: _configSchema.schema.maybe(_configSchema.schema.string()),
            memoryContainerId: _configSchema.schema.maybe(_configSchema.schema.string()),
            owner: _configSchema.schema.maybe(_configSchema.schema.string())
          }),
          timestamp: _configSchema.schema.string()
        })))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const noteObject = {
        hypotheses: request.body.hypotheses,
        ...(request.body.topologies !== null && request.body.topologies !== undefined ? {
          topologies: request.body.topologies
        } : {}),
        dateModified: new Date().toISOString(),
        ...(request.body.runningMemory ? {
          runningMemory: request.body.runningMemory
        } : {
          runningMemory: null
        }),
        ...(request.body.historyMemory ? {
          historyMemory: request.body.historyMemory
        } : {
          historyMemory: null
        }),
        ...(request.body.failedInvestigation !== undefined ? {
          failedInvestigation: request.body.failedInvestigation
        } : {})
      };
      const noteBookInfo = await (0, _saved_objects_paragraphs_router.fetchNotebook)(request.body.notebookId, opensearchNotebooksClient);
      noteBookInfo.attributes.savedNotebook = {
        ...noteBookInfo.attributes.savedNotebook,
        ...noteObject
      };
      const updateResponse = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteBookInfo.attributes, {
        id: request.body.notebookId,
        overwrite: true,
        version: noteBookInfo.version
      });
      return response.ok({
        body: updateResponse
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to hypotheses: ${error.message}`,
          error: error.name
        }
      });
    }
  });
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const userName = getUserName(request);
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const notebookinfo = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.noteId);
      const savedNotebook = notebookinfo.attributes.savedNotebook;
      return response.ok({
        body: {
          ...savedNotebook,
          currentUser: userName
        }
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/clone`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const getNotebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId);
      const createCloneNotebook = (0, _saved_objects_notebooks_router.cloneNotebook)(getNotebook.attributes.savedNotebook, request.body.name);
      const createdNotebook = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, createCloneNotebook);
      return response.ok({
        body: createdNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/rename`,
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const renamedNotebook = (0, _saved_objects_notebooks_router.renameNotebook)(request.body);
      const updatedNotebook = await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.body.noteId, renamedNotebook);
      return response.ok({
        body: updatedNotebook
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/{noteId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const deletedNotebooks = await opensearchNotebooksClient.delete(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.noteId);
      return response.ok({
        body: deletedNotebooks
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/note/savedNotebook/addSampleNotebooks`,
    validate: {
      body: _configSchema.schema.object({
        visIds: _configSchema.schema.arrayOf(_configSchema.schema.string()),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const sampleNotebooks = await (0, _saved_objects_notebooks_router.addSampleNotes)(opensearchNotebooksClient, request.body.visIds, request.body.dataSourceId);
      return response.ok({
        body: sampleNotebooks
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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