"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerMLConnectorRoute = registerMLConnectorRoute;
var _configSchema = require("@osd/config-schema");
var _lruCache = _interopRequireDefault(require("lru-cache"));
var _notebooks = require("../../../common/constants/notebooks");
var _utils = require("../utils");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// Initialize LRU cache with max 100 entries and 1 hour TTL
const mlProxyCache = new _lruCache.default({
  max: 100,
  maxAge: 1000 * 60 * 60 // 1 hour
});

/**
 * Removes specified characters from the beginning of a string.
 * @param str The string to trim
 * @param chars The characters to trim from the beginning (defaults to whitespace)
 * @returns A new string with specified characters removed from the beginning
 */
function trimStart(str, chars) {
  // Handle empty inputs
  if (!str) return str;

  // If no chars provided, use whitespace as default
  if (chars === undefined || chars === '') {
    return str.replace(/^\s+/, '');
  }

  // Escape special regex characters in the chars parameter
  const escapedChars = chars.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

  // Create a regex that matches any of the specified characters at the beginning
  const regex = new RegExp(`^[${escapedChars}]+`);

  // Replace matches with empty string
  return str.replace(regex, '');
}
function toUrlPath(path) {
  const FAKE_BASE = 'http://localhost';
  const urlWithFakeBase = new URL(`${FAKE_BASE}/${trimStart(path, '/')}`);
  const urlPath = urlWithFakeBase.href.replace(urlWithFakeBase.origin, '');
  return urlPath;
}
const acceptedHttpVerb = _configSchema.schema.string({
  validate: method => {
    return ['GET', 'POST', 'PUT', 'DELETE'].some(verb => verb.toLowerCase() === method.toLowerCase()) ? undefined : `Method must be one of, case insensitive ['GET', 'POST', 'PUT', 'DELETE']. Received '${method}'.`;
  }
});

/**
 * Validates if the ML API path is allowed for proxy requests.
 * Only Memory search and ML config related APIs are permitted.
 * @param path The API path to validate
 * @returns true if the path is allowed, false otherwise
 */
function isAllowedMLPath(path) {
  // Define allowed ML API path patterns
  const allowedPatterns = [
  // Agentic Memory related APIs
  /^\/_plugins\/_ml\/memory_containers\/[^/]+\/memories\/working\/_search$/, /^\/_plugins\/_ml\/memory_containers\/[^/]+\/memories\/sessions$/,
  // ML Config API
  /^\/_plugins\/_ml\/config\/[^/]+$/,
  // Index Insight API
  /^\/_plugins\/_ml\/insights\/[^/]+\/LOG_RELATED_INDEX_CHECK$/,
  // Agent Detail API
  /^\/_plugins\/_ml\/agents\/[^/]+$/];

  // Check if path matches any allowed pattern
  return allowedPatterns.some(pattern => pattern.test(path));
}
function registerMLConnectorRoute(router) {
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/ml/proxy`,
    validate: {
      body: _configSchema.schema.maybe(_configSchema.schema.any()),
      query: _configSchema.schema.object({
        method: acceptedHttpVerb,
        path: _configSchema.schema.string(),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const {
      method,
      path
    } = request.query;
    const matchIndexInsight = /\/_plugins\/_ml\/insights\/[^/]+\/LOG_RELATED_INDEX_CHECK$/.test(path);
    let cacheKey = '';
    if (matchIndexInsight) {
      var _path$match;
      const indexName = ((_path$match = path.match(/\/insights\/([^/]+)\//)) === null || _path$match === void 0 ? void 0 : _path$match[1]) || '';
      cacheKey = `${request.query.dataSourceId || 'local'}:${indexName}`;
      if (mlProxyCache.has(cacheKey)) {
        return response.ok({
          body: mlProxyCache.get(cacheKey)
        });
      }
    }

    // Validate if the specific ML API is allowed
    if (!isAllowedMLPath(path)) {
      return response.forbidden({
        body: `Error connecting to '${path}':\n\nUnable to send requests to that path.`,
        headers: {
          'Content-Type': 'text/plain'
        }
      });
    }
    try {
      var _result$headers;
      const transport = await (0, _utils.getOpenSearchClientTransport)({
        context,
        request,
        dataSourceId: request.query.dataSourceId
      });
      const result = await transport.request({
        path: toUrlPath(path),
        method,
        body: request.body
      });
      if (matchIndexInsight && (result.statusCode || 200) >= 200 && (result.statusCode || 200) < 300) {
        mlProxyCache.set(cacheKey, result.body);
      }
      const contentType = (_result$headers = result.headers) === null || _result$headers === void 0 ? void 0 : _result$headers['Content-Type'];
      return response.custom({
        body: result.body,
        statusCode: result.statusCode || 200,
        headers: {
          'Content-Type': contentType
        }
      });
    } catch (error) {
      return (0, _utils.handleError)(error, response);
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9scnVDYWNoZSIsIl9pbnRlcm9wUmVxdWlyZURlZmF1bHQiLCJfbm90ZWJvb2tzIiwiX3V0aWxzIiwiZSIsIl9fZXNNb2R1bGUiLCJkZWZhdWx0IiwibWxQcm94eUNhY2hlIiwiTFJVQ2FjaGUiLCJtYXgiLCJtYXhBZ2UiLCJ0cmltU3RhcnQiLCJzdHIiLCJjaGFycyIsInVuZGVmaW5lZCIsInJlcGxhY2UiLCJlc2NhcGVkQ2hhcnMiLCJyZWdleCIsIlJlZ0V4cCIsInRvVXJsUGF0aCIsInBhdGgiLCJGQUtFX0JBU0UiLCJ1cmxXaXRoRmFrZUJhc2UiLCJVUkwiLCJ1cmxQYXRoIiwiaHJlZiIsIm9yaWdpbiIsImFjY2VwdGVkSHR0cFZlcmIiLCJzY2hlbWEiLCJzdHJpbmciLCJ2YWxpZGF0ZSIsIm1ldGhvZCIsInNvbWUiLCJ2ZXJiIiwidG9Mb3dlckNhc2UiLCJpc0FsbG93ZWRNTFBhdGgiLCJhbGxvd2VkUGF0dGVybnMiLCJwYXR0ZXJuIiwidGVzdCIsInJlZ2lzdGVyTUxDb25uZWN0b3JSb3V0ZSIsInJvdXRlciIsInBvc3QiLCJOT1RFQk9PS1NfQVBJX1BSRUZJWCIsImJvZHkiLCJtYXliZSIsImFueSIsInF1ZXJ5Iiwib2JqZWN0IiwiZGF0YVNvdXJjZUlkIiwiY29udGV4dCIsInJlcXVlc3QiLCJyZXNwb25zZSIsIm1hdGNoSW5kZXhJbnNpZ2h0IiwiY2FjaGVLZXkiLCJfcGF0aCRtYXRjaCIsImluZGV4TmFtZSIsIm1hdGNoIiwiaGFzIiwib2siLCJnZXQiLCJmb3JiaWRkZW4iLCJoZWFkZXJzIiwiX3Jlc3VsdCRoZWFkZXJzIiwidHJhbnNwb3J0IiwiZ2V0T3BlblNlYXJjaENsaWVudFRyYW5zcG9ydCIsInJlc3VsdCIsInN0YXR1c0NvZGUiLCJzZXQiLCJjb250ZW50VHlwZSIsImN1c3RvbSIsImVycm9yIiwiaGFuZGxlRXJyb3IiXSwic291cmNlcyI6WyJtbF9yb3V0ZXIudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcclxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXHJcbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXHJcbiAqL1xyXG5cclxuaW1wb3J0IHsgc2NoZW1hIH0gZnJvbSAnQG9zZC9jb25maWctc2NoZW1hJztcclxuaW1wb3J0IExSVUNhY2hlIGZyb20gJ2xydS1jYWNoZSc7XHJcbmltcG9ydCB7IElPcGVuU2VhcmNoRGFzaGJvYXJkc1Jlc3BvbnNlLCBJUm91dGVyIH0gZnJvbSAnLi4vLi4vLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcclxuaW1wb3J0IHsgTk9URUJPT0tTX0FQSV9QUkVGSVggfSBmcm9tICcuLi8uLi8uLi9jb21tb24vY29uc3RhbnRzL25vdGVib29rcyc7XHJcbmltcG9ydCB7IGdldE9wZW5TZWFyY2hDbGllbnRUcmFuc3BvcnQsIGhhbmRsZUVycm9yIH0gZnJvbSAnLi4vdXRpbHMnO1xyXG5cclxuLy8gSW5pdGlhbGl6ZSBMUlUgY2FjaGUgd2l0aCBtYXggMTAwIGVudHJpZXMgYW5kIDEgaG91ciBUVExcclxuY29uc3QgbWxQcm94eUNhY2hlID0gbmV3IExSVUNhY2hlPHN0cmluZywgc3RyaW5nPih7XHJcbiAgbWF4OiAxMDAsXHJcbiAgbWF4QWdlOiAxMDAwICogNjAgKiA2MCwgLy8gMSBob3VyXHJcbn0pO1xyXG5cclxuLyoqXHJcbiAqIFJlbW92ZXMgc3BlY2lmaWVkIGNoYXJhY3RlcnMgZnJvbSB0aGUgYmVnaW5uaW5nIG9mIGEgc3RyaW5nLlxyXG4gKiBAcGFyYW0gc3RyIFRoZSBzdHJpbmcgdG8gdHJpbVxyXG4gKiBAcGFyYW0gY2hhcnMgVGhlIGNoYXJhY3RlcnMgdG8gdHJpbSBmcm9tIHRoZSBiZWdpbm5pbmcgKGRlZmF1bHRzIHRvIHdoaXRlc3BhY2UpXHJcbiAqIEByZXR1cm5zIEEgbmV3IHN0cmluZyB3aXRoIHNwZWNpZmllZCBjaGFyYWN0ZXJzIHJlbW92ZWQgZnJvbSB0aGUgYmVnaW5uaW5nXHJcbiAqL1xyXG5mdW5jdGlvbiB0cmltU3RhcnQoc3RyOiBzdHJpbmcsIGNoYXJzPzogc3RyaW5nKTogc3RyaW5nIHtcclxuICAvLyBIYW5kbGUgZW1wdHkgaW5wdXRzXHJcbiAgaWYgKCFzdHIpIHJldHVybiBzdHI7XHJcblxyXG4gIC8vIElmIG5vIGNoYXJzIHByb3ZpZGVkLCB1c2Ugd2hpdGVzcGFjZSBhcyBkZWZhdWx0XHJcbiAgaWYgKGNoYXJzID09PSB1bmRlZmluZWQgfHwgY2hhcnMgPT09ICcnKSB7XHJcbiAgICByZXR1cm4gc3RyLnJlcGxhY2UoL15cXHMrLywgJycpO1xyXG4gIH1cclxuXHJcbiAgLy8gRXNjYXBlIHNwZWNpYWwgcmVnZXggY2hhcmFjdGVycyBpbiB0aGUgY2hhcnMgcGFyYW1ldGVyXHJcbiAgY29uc3QgZXNjYXBlZENoYXJzID0gY2hhcnMucmVwbGFjZSgvWy4qKz9eJHt9KCl8W1xcXVxcXFxdL2csICdcXFxcJCYnKTtcclxuXHJcbiAgLy8gQ3JlYXRlIGEgcmVnZXggdGhhdCBtYXRjaGVzIGFueSBvZiB0aGUgc3BlY2lmaWVkIGNoYXJhY3RlcnMgYXQgdGhlIGJlZ2lubmluZ1xyXG4gIGNvbnN0IHJlZ2V4ID0gbmV3IFJlZ0V4cChgXlske2VzY2FwZWRDaGFyc31dK2ApO1xyXG5cclxuICAvLyBSZXBsYWNlIG1hdGNoZXMgd2l0aCBlbXB0eSBzdHJpbmdcclxuICByZXR1cm4gc3RyLnJlcGxhY2UocmVnZXgsICcnKTtcclxufVxyXG5cclxuZnVuY3Rpb24gdG9VcmxQYXRoKHBhdGg6IHN0cmluZykge1xyXG4gIGNvbnN0IEZBS0VfQkFTRSA9ICdodHRwOi8vbG9jYWxob3N0JztcclxuICBjb25zdCB1cmxXaXRoRmFrZUJhc2UgPSBuZXcgVVJMKGAke0ZBS0VfQkFTRX0vJHt0cmltU3RhcnQocGF0aCwgJy8nKX1gKTtcclxuICBjb25zdCB1cmxQYXRoID0gdXJsV2l0aEZha2VCYXNlLmhyZWYucmVwbGFjZSh1cmxXaXRoRmFrZUJhc2Uub3JpZ2luLCAnJyk7XHJcbiAgcmV0dXJuIHVybFBhdGg7XHJcbn1cclxuXHJcbmNvbnN0IGFjY2VwdGVkSHR0cFZlcmIgPSBzY2hlbWEuc3RyaW5nKHtcclxuICB2YWxpZGF0ZTogKG1ldGhvZCkgPT4ge1xyXG4gICAgcmV0dXJuIFsnR0VUJywgJ1BPU1QnLCAnUFVUJywgJ0RFTEVURSddLnNvbWUoXHJcbiAgICAgICh2ZXJiKSA9PiB2ZXJiLnRvTG93ZXJDYXNlKCkgPT09IG1ldGhvZC50b0xvd2VyQ2FzZSgpXHJcbiAgICApXHJcbiAgICAgID8gdW5kZWZpbmVkXHJcbiAgICAgIDogYE1ldGhvZCBtdXN0IGJlIG9uZSBvZiwgY2FzZSBpbnNlbnNpdGl2ZSBbJ0dFVCcsICdQT1NUJywgJ1BVVCcsICdERUxFVEUnXS4gUmVjZWl2ZWQgJyR7bWV0aG9kfScuYDtcclxuICB9LFxyXG59KTtcclxuXHJcbi8qKlxyXG4gKiBWYWxpZGF0ZXMgaWYgdGhlIE1MIEFQSSBwYXRoIGlzIGFsbG93ZWQgZm9yIHByb3h5IHJlcXVlc3RzLlxyXG4gKiBPbmx5IE1lbW9yeSBzZWFyY2ggYW5kIE1MIGNvbmZpZyByZWxhdGVkIEFQSXMgYXJlIHBlcm1pdHRlZC5cclxuICogQHBhcmFtIHBhdGggVGhlIEFQSSBwYXRoIHRvIHZhbGlkYXRlXHJcbiAqIEByZXR1cm5zIHRydWUgaWYgdGhlIHBhdGggaXMgYWxsb3dlZCwgZmFsc2Ugb3RoZXJ3aXNlXHJcbiAqL1xyXG5mdW5jdGlvbiBpc0FsbG93ZWRNTFBhdGgocGF0aDogc3RyaW5nKTogYm9vbGVhbiB7XHJcbiAgLy8gRGVmaW5lIGFsbG93ZWQgTUwgQVBJIHBhdGggcGF0dGVybnNcclxuICBjb25zdCBhbGxvd2VkUGF0dGVybnMgPSBbXHJcbiAgICAvLyBBZ2VudGljIE1lbW9yeSByZWxhdGVkIEFQSXNcclxuICAgIC9eXFwvX3BsdWdpbnNcXC9fbWxcXC9tZW1vcnlfY29udGFpbmVyc1xcL1teL10rXFwvbWVtb3JpZXNcXC93b3JraW5nXFwvX3NlYXJjaCQvLFxyXG4gICAgL15cXC9fcGx1Z2luc1xcL19tbFxcL21lbW9yeV9jb250YWluZXJzXFwvW14vXStcXC9tZW1vcmllc1xcL3Nlc3Npb25zJC8sXHJcbiAgICAvLyBNTCBDb25maWcgQVBJXHJcbiAgICAvXlxcL19wbHVnaW5zXFwvX21sXFwvY29uZmlnXFwvW14vXSskLyxcclxuICAgIC8vIEluZGV4IEluc2lnaHQgQVBJXHJcbiAgICAvXlxcL19wbHVnaW5zXFwvX21sXFwvaW5zaWdodHNcXC9bXi9dK1xcL0xPR19SRUxBVEVEX0lOREVYX0NIRUNLJC8sXHJcbiAgICAvLyBBZ2VudCBEZXRhaWwgQVBJXHJcbiAgICAvXlxcL19wbHVnaW5zXFwvX21sXFwvYWdlbnRzXFwvW14vXSskLyxcclxuICBdO1xyXG5cclxuICAvLyBDaGVjayBpZiBwYXRoIG1hdGNoZXMgYW55IGFsbG93ZWQgcGF0dGVyblxyXG4gIHJldHVybiBhbGxvd2VkUGF0dGVybnMuc29tZSgocGF0dGVybikgPT4gcGF0dGVybi50ZXN0KHBhdGgpKTtcclxufVxyXG5cclxuZXhwb3J0IGZ1bmN0aW9uIHJlZ2lzdGVyTUxDb25uZWN0b3JSb3V0ZShyb3V0ZXI6IElSb3V0ZXIpIHtcclxuICByb3V0ZXIucG9zdChcclxuICAgIHtcclxuICAgICAgcGF0aDogYCR7Tk9URUJPT0tTX0FQSV9QUkVGSVh9L21sL3Byb3h5YCxcclxuICAgICAgdmFsaWRhdGU6IHtcclxuICAgICAgICBib2R5OiBzY2hlbWEubWF5YmUoc2NoZW1hLmFueSgpKSxcclxuICAgICAgICBxdWVyeTogc2NoZW1hLm9iamVjdCh7XHJcbiAgICAgICAgICBtZXRob2Q6IGFjY2VwdGVkSHR0cFZlcmIsXHJcbiAgICAgICAgICBwYXRoOiBzY2hlbWEuc3RyaW5nKCksXHJcbiAgICAgICAgICBkYXRhU291cmNlSWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpLFxyXG4gICAgICAgIH0pLFxyXG4gICAgICB9LFxyXG4gICAgfSxcclxuICAgIGFzeW5jIChjb250ZXh0LCByZXF1ZXN0LCByZXNwb25zZSk6IFByb21pc2U8SU9wZW5TZWFyY2hEYXNoYm9hcmRzUmVzcG9uc2U+ID0+IHtcclxuICAgICAgY29uc3QgeyBtZXRob2QsIHBhdGggfSA9IHJlcXVlc3QucXVlcnk7XHJcbiAgICAgIGNvbnN0IG1hdGNoSW5kZXhJbnNpZ2h0ID0gL1xcL19wbHVnaW5zXFwvX21sXFwvaW5zaWdodHNcXC9bXi9dK1xcL0xPR19SRUxBVEVEX0lOREVYX0NIRUNLJC8udGVzdChcclxuICAgICAgICBwYXRoXHJcbiAgICAgICk7XHJcbiAgICAgIGxldCBjYWNoZUtleSA9ICcnO1xyXG4gICAgICBpZiAobWF0Y2hJbmRleEluc2lnaHQpIHtcclxuICAgICAgICBjb25zdCBpbmRleE5hbWUgPSBwYXRoLm1hdGNoKC9cXC9pbnNpZ2h0c1xcLyhbXi9dKylcXC8vKT8uWzFdIHx8ICcnO1xyXG4gICAgICAgIGNhY2hlS2V5ID0gYCR7cmVxdWVzdC5xdWVyeS5kYXRhU291cmNlSWQgfHwgJ2xvY2FsJ306JHtpbmRleE5hbWV9YDtcclxuICAgICAgICBpZiAobWxQcm94eUNhY2hlLmhhcyhjYWNoZUtleSkpIHtcclxuICAgICAgICAgIHJldHVybiByZXNwb25zZS5vayh7XHJcbiAgICAgICAgICAgIGJvZHk6IG1sUHJveHlDYWNoZS5nZXQoY2FjaGVLZXkpLFxyXG4gICAgICAgICAgfSk7XHJcbiAgICAgICAgfVxyXG4gICAgICB9XHJcblxyXG4gICAgICAvLyBWYWxpZGF0ZSBpZiB0aGUgc3BlY2lmaWMgTUwgQVBJIGlzIGFsbG93ZWRcclxuICAgICAgaWYgKCFpc0FsbG93ZWRNTFBhdGgocGF0aCkpIHtcclxuICAgICAgICByZXR1cm4gcmVzcG9uc2UuZm9yYmlkZGVuKHtcclxuICAgICAgICAgIGJvZHk6IGBFcnJvciBjb25uZWN0aW5nIHRvICcke3BhdGh9JzpcXG5cXG5VbmFibGUgdG8gc2VuZCByZXF1ZXN0cyB0byB0aGF0IHBhdGguYCxcclxuICAgICAgICAgIGhlYWRlcnM6IHtcclxuICAgICAgICAgICAgJ0NvbnRlbnQtVHlwZSc6ICd0ZXh0L3BsYWluJyxcclxuICAgICAgICAgIH0sXHJcbiAgICAgICAgfSk7XHJcbiAgICAgIH1cclxuXHJcbiAgICAgIHRyeSB7XHJcbiAgICAgICAgY29uc3QgdHJhbnNwb3J0ID0gYXdhaXQgZ2V0T3BlblNlYXJjaENsaWVudFRyYW5zcG9ydCh7XHJcbiAgICAgICAgICBjb250ZXh0LFxyXG4gICAgICAgICAgcmVxdWVzdCxcclxuICAgICAgICAgIGRhdGFTb3VyY2VJZDogcmVxdWVzdC5xdWVyeS5kYXRhU291cmNlSWQsXHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIGNvbnN0IHJlc3VsdCA9IGF3YWl0IHRyYW5zcG9ydC5yZXF1ZXN0KHtcclxuICAgICAgICAgIHBhdGg6IHRvVXJsUGF0aChwYXRoKSxcclxuICAgICAgICAgIG1ldGhvZCxcclxuICAgICAgICAgIGJvZHk6IHJlcXVlc3QuYm9keSxcclxuICAgICAgICB9KTtcclxuICAgICAgICBpZiAoXHJcbiAgICAgICAgICBtYXRjaEluZGV4SW5zaWdodCAmJlxyXG4gICAgICAgICAgKHJlc3VsdC5zdGF0dXNDb2RlIHx8IDIwMCkgPj0gMjAwICYmXHJcbiAgICAgICAgICAocmVzdWx0LnN0YXR1c0NvZGUgfHwgMjAwKSA8IDMwMFxyXG4gICAgICAgICkge1xyXG4gICAgICAgICAgbWxQcm94eUNhY2hlLnNldChjYWNoZUtleSwgcmVzdWx0LmJvZHkpO1xyXG4gICAgICAgIH1cclxuICAgICAgICBjb25zdCBjb250ZW50VHlwZSA9IHJlc3VsdC5oZWFkZXJzPy5bJ0NvbnRlbnQtVHlwZSddO1xyXG5cclxuICAgICAgICByZXR1cm4gcmVzcG9uc2UuY3VzdG9tKHtcclxuICAgICAgICAgIGJvZHk6IHJlc3VsdC5ib2R5LFxyXG4gICAgICAgICAgc3RhdHVzQ29kZTogcmVzdWx0LnN0YXR1c0NvZGUgfHwgMjAwLFxyXG4gICAgICAgICAgaGVhZGVyczoge1xyXG4gICAgICAgICAgICAnQ29udGVudC1UeXBlJzogY29udGVudFR5cGUsXHJcbiAgICAgICAgICB9LFxyXG4gICAgICAgIH0pO1xyXG4gICAgICB9IGNhdGNoIChlcnJvcikge1xyXG4gICAgICAgIHJldHVybiBoYW5kbGVFcnJvcihlcnJvciwgcmVzcG9uc2UpO1xyXG4gICAgICB9XHJcbiAgICB9XHJcbiAgKTtcclxufVxyXG4iXSwibWFwcGluZ3MiOiI7Ozs7OztBQUtBLElBQUFBLGFBQUEsR0FBQUMsT0FBQTtBQUNBLElBQUFDLFNBQUEsR0FBQUMsc0JBQUEsQ0FBQUYsT0FBQTtBQUVBLElBQUFHLFVBQUEsR0FBQUgsT0FBQTtBQUNBLElBQUFJLE1BQUEsR0FBQUosT0FBQTtBQUFxRSxTQUFBRSx1QkFBQUcsQ0FBQSxXQUFBQSxDQUFBLElBQUFBLENBQUEsQ0FBQUMsVUFBQSxHQUFBRCxDQUFBLEtBQUFFLE9BQUEsRUFBQUYsQ0FBQTtBQVRyRTtBQUNBO0FBQ0E7QUFDQTs7QUFRQTtBQUNBLE1BQU1HLFlBQVksR0FBRyxJQUFJQyxpQkFBUSxDQUFpQjtFQUNoREMsR0FBRyxFQUFFLEdBQUc7RUFDUkMsTUFBTSxFQUFFLElBQUksR0FBRyxFQUFFLEdBQUcsRUFBRSxDQUFFO0FBQzFCLENBQUMsQ0FBQzs7QUFFRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxTQUFTQyxTQUFTQSxDQUFDQyxHQUFXLEVBQUVDLEtBQWMsRUFBVTtFQUN0RDtFQUNBLElBQUksQ0FBQ0QsR0FBRyxFQUFFLE9BQU9BLEdBQUc7O0VBRXBCO0VBQ0EsSUFBSUMsS0FBSyxLQUFLQyxTQUFTLElBQUlELEtBQUssS0FBSyxFQUFFLEVBQUU7SUFDdkMsT0FBT0QsR0FBRyxDQUFDRyxPQUFPLENBQUMsTUFBTSxFQUFFLEVBQUUsQ0FBQztFQUNoQzs7RUFFQTtFQUNBLE1BQU1DLFlBQVksR0FBR0gsS0FBSyxDQUFDRSxPQUFPLENBQUMscUJBQXFCLEVBQUUsTUFBTSxDQUFDOztFQUVqRTtFQUNBLE1BQU1FLEtBQUssR0FBRyxJQUFJQyxNQUFNLENBQUUsS0FBSUYsWUFBYSxJQUFHLENBQUM7O0VBRS9DO0VBQ0EsT0FBT0osR0FBRyxDQUFDRyxPQUFPLENBQUNFLEtBQUssRUFBRSxFQUFFLENBQUM7QUFDL0I7QUFFQSxTQUFTRSxTQUFTQSxDQUFDQyxJQUFZLEVBQUU7RUFDL0IsTUFBTUMsU0FBUyxHQUFHLGtCQUFrQjtFQUNwQyxNQUFNQyxlQUFlLEdBQUcsSUFBSUMsR0FBRyxDQUFFLEdBQUVGLFNBQVUsSUFBR1YsU0FBUyxDQUFDUyxJQUFJLEVBQUUsR0FBRyxDQUFFLEVBQUMsQ0FBQztFQUN2RSxNQUFNSSxPQUFPLEdBQUdGLGVBQWUsQ0FBQ0csSUFBSSxDQUFDVixPQUFPLENBQUNPLGVBQWUsQ0FBQ0ksTUFBTSxFQUFFLEVBQUUsQ0FBQztFQUN4RSxPQUFPRixPQUFPO0FBQ2hCO0FBRUEsTUFBTUcsZ0JBQWdCLEdBQUdDLG9CQUFNLENBQUNDLE1BQU0sQ0FBQztFQUNyQ0MsUUFBUSxFQUFHQyxNQUFNLElBQUs7SUFDcEIsT0FBTyxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUUsS0FBSyxFQUFFLFFBQVEsQ0FBQyxDQUFDQyxJQUFJLENBQ3pDQyxJQUFJLElBQUtBLElBQUksQ0FBQ0MsV0FBVyxDQUFDLENBQUMsS0FBS0gsTUFBTSxDQUFDRyxXQUFXLENBQUMsQ0FDdEQsQ0FBQyxHQUNHcEIsU0FBUyxHQUNSLHVGQUFzRmlCLE1BQU8sSUFBRztFQUN2RztBQUNGLENBQUMsQ0FBQzs7QUFFRjtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQSxTQUFTSSxlQUFlQSxDQUFDZixJQUFZLEVBQVc7RUFDOUM7RUFDQSxNQUFNZ0IsZUFBZSxHQUFHO0VBQ3RCO0VBQ0EseUVBQXlFLEVBQ3pFLGlFQUFpRTtFQUNqRTtFQUNBLGtDQUFrQztFQUNsQztFQUNBLDZEQUE2RDtFQUM3RDtFQUNBLGtDQUFrQyxDQUNuQzs7RUFFRDtFQUNBLE9BQU9BLGVBQWUsQ0FBQ0osSUFBSSxDQUFFSyxPQUFPLElBQUtBLE9BQU8sQ0FBQ0MsSUFBSSxDQUFDbEIsSUFBSSxDQUFDLENBQUM7QUFDOUQ7QUFFTyxTQUFTbUIsd0JBQXdCQSxDQUFDQyxNQUFlLEVBQUU7RUFDeERBLE1BQU0sQ0FBQ0MsSUFBSSxDQUNUO0lBQ0VyQixJQUFJLEVBQUcsR0FBRXNCLCtCQUFxQixXQUFVO0lBQ3hDWixRQUFRLEVBQUU7TUFDUmEsSUFBSSxFQUFFZixvQkFBTSxDQUFDZ0IsS0FBSyxDQUFDaEIsb0JBQU0sQ0FBQ2lCLEdBQUcsQ0FBQyxDQUFDLENBQUM7TUFDaENDLEtBQUssRUFBRWxCLG9CQUFNLENBQUNtQixNQUFNLENBQUM7UUFDbkJoQixNQUFNLEVBQUVKLGdCQUFnQjtRQUN4QlAsSUFBSSxFQUFFUSxvQkFBTSxDQUFDQyxNQUFNLENBQUMsQ0FBQztRQUNyQm1CLFlBQVksRUFBRXBCLG9CQUFNLENBQUNnQixLQUFLLENBQUNoQixvQkFBTSxDQUFDQyxNQUFNLENBQUMsQ0FBQztNQUM1QyxDQUFDO0lBQ0g7RUFDRixDQUFDLEVBQ0QsT0FBT29CLE9BQU8sRUFBRUMsT0FBTyxFQUFFQyxRQUFRLEtBQTZDO0lBQzVFLE1BQU07TUFBRXBCLE1BQU07TUFBRVg7SUFBSyxDQUFDLEdBQUc4QixPQUFPLENBQUNKLEtBQUs7SUFDdEMsTUFBTU0saUJBQWlCLEdBQUcsNERBQTRELENBQUNkLElBQUksQ0FDekZsQixJQUNGLENBQUM7SUFDRCxJQUFJaUMsUUFBUSxHQUFHLEVBQUU7SUFDakIsSUFBSUQsaUJBQWlCLEVBQUU7TUFBQSxJQUFBRSxXQUFBO01BQ3JCLE1BQU1DLFNBQVMsR0FBRyxFQUFBRCxXQUFBLEdBQUFsQyxJQUFJLENBQUNvQyxLQUFLLENBQUMsdUJBQXVCLENBQUMsY0FBQUYsV0FBQSx1QkFBbkNBLFdBQUEsQ0FBc0MsQ0FBQyxDQUFDLEtBQUksRUFBRTtNQUNoRUQsUUFBUSxHQUFJLEdBQUVILE9BQU8sQ0FBQ0osS0FBSyxDQUFDRSxZQUFZLElBQUksT0FBUSxJQUFHTyxTQUFVLEVBQUM7TUFDbEUsSUFBSWhELFlBQVksQ0FBQ2tELEdBQUcsQ0FBQ0osUUFBUSxDQUFDLEVBQUU7UUFDOUIsT0FBT0YsUUFBUSxDQUFDTyxFQUFFLENBQUM7VUFDakJmLElBQUksRUFBRXBDLFlBQVksQ0FBQ29ELEdBQUcsQ0FBQ04sUUFBUTtRQUNqQyxDQUFDLENBQUM7TUFDSjtJQUNGOztJQUVBO0lBQ0EsSUFBSSxDQUFDbEIsZUFBZSxDQUFDZixJQUFJLENBQUMsRUFBRTtNQUMxQixPQUFPK0IsUUFBUSxDQUFDUyxTQUFTLENBQUM7UUFDeEJqQixJQUFJLEVBQUcsd0JBQXVCdkIsSUFBSyw2Q0FBNEM7UUFDL0V5QyxPQUFPLEVBQUU7VUFDUCxjQUFjLEVBQUU7UUFDbEI7TUFDRixDQUFDLENBQUM7SUFDSjtJQUVBLElBQUk7TUFBQSxJQUFBQyxlQUFBO01BQ0YsTUFBTUMsU0FBUyxHQUFHLE1BQU0sSUFBQUMsbUNBQTRCLEVBQUM7UUFDbkRmLE9BQU87UUFDUEMsT0FBTztRQUNQRixZQUFZLEVBQUVFLE9BQU8sQ0FBQ0osS0FBSyxDQUFDRTtNQUM5QixDQUFDLENBQUM7TUFFRixNQUFNaUIsTUFBTSxHQUFHLE1BQU1GLFNBQVMsQ0FBQ2IsT0FBTyxDQUFDO1FBQ3JDOUIsSUFBSSxFQUFFRCxTQUFTLENBQUNDLElBQUksQ0FBQztRQUNyQlcsTUFBTTtRQUNOWSxJQUFJLEVBQUVPLE9BQU8sQ0FBQ1A7TUFDaEIsQ0FBQyxDQUFDO01BQ0YsSUFDRVMsaUJBQWlCLElBQ2pCLENBQUNhLE1BQU0sQ0FBQ0MsVUFBVSxJQUFJLEdBQUcsS0FBSyxHQUFHLElBQ2pDLENBQUNELE1BQU0sQ0FBQ0MsVUFBVSxJQUFJLEdBQUcsSUFBSSxHQUFHLEVBQ2hDO1FBQ0EzRCxZQUFZLENBQUM0RCxHQUFHLENBQUNkLFFBQVEsRUFBRVksTUFBTSxDQUFDdEIsSUFBSSxDQUFDO01BQ3pDO01BQ0EsTUFBTXlCLFdBQVcsSUFBQU4sZUFBQSxHQUFHRyxNQUFNLENBQUNKLE9BQU8sY0FBQUMsZUFBQSx1QkFBZEEsZUFBQSxDQUFpQixjQUFjLENBQUM7TUFFcEQsT0FBT1gsUUFBUSxDQUFDa0IsTUFBTSxDQUFDO1FBQ3JCMUIsSUFBSSxFQUFFc0IsTUFBTSxDQUFDdEIsSUFBSTtRQUNqQnVCLFVBQVUsRUFBRUQsTUFBTSxDQUFDQyxVQUFVLElBQUksR0FBRztRQUNwQ0wsT0FBTyxFQUFFO1VBQ1AsY0FBYyxFQUFFTztRQUNsQjtNQUNGLENBQUMsQ0FBQztJQUNKLENBQUMsQ0FBQyxPQUFPRSxLQUFLLEVBQUU7TUFDZCxPQUFPLElBQUFDLGtCQUFXLEVBQUNELEtBQUssRUFBRW5CLFFBQVEsQ0FBQztJQUNyQztFQUNGLENBQ0YsQ0FBQztBQUNIIn0=