"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerHypothesisRoute = registerHypothesisRoute;
var _configSchema = require("@osd/config-schema");
var _uuid = _interopRequireDefault(require("uuid"));
var _notebooks = require("../../../common/constants/notebooks");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerHypothesisRoute(router) {
  // Create Hypothesis
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/{noteId}/hypothesis`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        title: _configSchema.schema.string(),
        description: _configSchema.schema.string(),
        likelihood: _configSchema.schema.number({
          min: 1,
          max: 10
        }),
        supportingFindingParagraphIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const {
        noteId
      } = request.params;
      const {
        title,
        description,
        likelihood,
        supportingFindingParagraphIds
      } = request.body;

      // Get existing notebook
      const notebookObject = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId);
      const notebook = notebookObject.attributes.savedNotebook;

      // Create new hypothesis
      const newHypothesis = {
        id: 'hypothesis_' + (0, _uuid.default)(),
        title,
        description,
        likelihood,
        supportingFindingParagraphIds,
        dateCreated: new Date().toISOString(),
        dateModified: new Date().toISOString()
      };

      // Add hypothesis to notebook
      const updatedNotebook = {
        ...notebook,
        hypotheses: [...(notebook.hypotheses || []), newHypothesis],
        dateModified: new Date().toISOString()
      };
      await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId, {
        savedNotebook: updatedNotebook
      });
      return response.ok({
        body: newHypothesis
      });
    } catch (error) {
      const statusCode = error.statusCode || error.output.statusCode || error.output.payload.statusCode;
      return response.custom({
        statusCode: statusCode || 500,
        body: error.message
      });
    }
  });

  // Update Hypothesis
  router.put({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/{noteId}/hypothesis/{hypothesisId}`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        hypothesisId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        title: _configSchema.schema.maybe(_configSchema.schema.string()),
        description: _configSchema.schema.maybe(_configSchema.schema.string()),
        likelihood: _configSchema.schema.maybe(_configSchema.schema.number({
          min: 1,
          max: 10
        })),
        supportingFindingParagraphIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        irrelevantFindingParagraphIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        userSelectedFindingParagraphIds: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        status: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const {
        noteId,
        hypothesisId
      } = request.params;
      const updates = request.body;

      // Get existing notebook
      const notebookObject = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId);
      const notebook = notebookObject.attributes.savedNotebook;
      if (!notebook.hypotheses) {
        return response.notFound({
          body: 'Hypothesis not found'
        });
      }

      // Find and update hypothesis
      const hypothesisIndex = notebook.hypotheses.findIndex(h => h.id === hypothesisId);
      if (hypothesisIndex === -1) {
        return response.notFound({
          body: 'Hypothesis not found'
        });
      }
      const updatedHypothesis = {
        ...notebook.hypotheses[hypothesisIndex],
        ...updates,
        dateModified: new Date().toISOString()
      };
      notebook.hypotheses[hypothesisIndex] = updatedHypothesis;
      const updatedNotebook = {
        ...notebook,
        dateModified: new Date().toISOString()
      };
      await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId, {
        savedNotebook: updatedNotebook
      });
      return response.ok({
        body: updatedHypothesis
      });
    } catch (error) {
      const statusCode = error.statusCode || error.output.statusCode || error.output.payload.statusCode;
      return response.custom({
        statusCode: statusCode || 500,
        body: error.message
      });
    }
  });

  // Add findings to Hypothesis
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/{noteId}/hypothesis/{hypothesisId}/findings`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string(),
        hypothesisId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        paragraphIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const {
        noteId,
        hypothesisId
      } = request.params;
      const {
        paragraphIds
      } = request.body;

      // Get existing notebook
      const notebookObject = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId);
      const notebook = notebookObject.attributes.savedNotebook;
      if (!notebook.hypotheses) {
        return response.notFound({
          body: 'Hypothesis not found'
        });
      }

      // Find hypothesis
      const hypothesisIndex = notebook.hypotheses.findIndex(h => h.id === hypothesisId);
      if (hypothesisIndex === -1) {
        return response.notFound({
          body: 'Hypothesis not found'
        });
      }

      // Add new paragraph IDs to existing ones (avoid duplicates)
      const existingIds = new Set(notebook.hypotheses[hypothesisIndex].supportingFindingParagraphIds);
      paragraphIds.forEach(id => existingIds.add(id));
      const updatedHypothesis = {
        ...notebook.hypotheses[hypothesisIndex],
        supportingFindingParagraphIds: Array.from(existingIds),
        dateModified: new Date().toISOString()
      };
      notebook.hypotheses[hypothesisIndex] = updatedHypothesis;
      const updatedNotebook = {
        ...notebook,
        dateModified: new Date().toISOString()
      };
      await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId, {
        savedNotebook: updatedNotebook
      });
      return response.ok({
        body: updatedHypothesis
      });
    } catch (error) {
      const statusCode = error.statusCode || error.output.statusCode || error.output.payload.statusCode;
      return response.custom({
        statusCode: statusCode || 500,
        body: error.message
      });
    }
  });

  // Get all hypotheses for a notebook
  router.get({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/{noteId}/hypotheses`,
    validate: {
      params: _configSchema.schema.object({
        noteId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const {
        noteId
      } = request.params;

      // Get existing notebook
      const notebookObject = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId);
      const notebook = notebookObject.attributes.savedNotebook;
      return response.ok({
        body: notebook.hypotheses || []
      });
    } catch (error) {
      const statusCode = error.statusCode || error.output.statusCode || error.output.payload.statusCode;
      return response.custom({
        statusCode: statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/{notebookId}/deleteHypothesis/{hypothesisId}`,
    validate: {
      params: _configSchema.schema.object({
        notebookId: _configSchema.schema.string(),
        hypothesisId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      var _savedNotebook$hypoth;
      const notebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.notebookId);
      const savedNotebook = notebook.attributes.savedNotebook;
      const updatedHypotheses = ((_savedNotebook$hypoth = savedNotebook.hypotheses) === null || _savedNotebook$hypoth === void 0 ? void 0 : _savedNotebook$hypoth.filter(h => h.id !== request.params.hypothesisId)) || [];
      await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.notebookId, {
        savedNotebook: {
          ...savedNotebook,
          hypotheses: updatedHypotheses,
          dateModified: new Date().toISOString()
        }
      });
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.delete({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/savedNotebook/{notebookId}/deleteAllHypotheses`,
    validate: {
      params: _configSchema.schema.object({
        notebookId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.core.savedObjects.client;
    try {
      const notebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.notebookId);
      const savedNotebook = notebook.attributes.savedNotebook;
      await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, request.params.notebookId, {
        savedNotebook: {
          ...savedNotebook,
          hypotheses: [],
          dateModified: new Date().toISOString()
        }
      });
      return response.ok({
        body: {
          success: true
        }
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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