"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAgentExecutionRoute = registerAgentExecutionRoute;
var _configSchema = require("@osd/config-schema");
var _notebooks = require("../../../common/constants/notebooks");
var _utils = require("../utils");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const commonInstructions = `
# Instructions

## Core Planning Rules
- Break the objective into an ordered list of atomic, self-contained Steps that, if executed, will lead to the final result or complete the objective
- Each Step must state what to do, where, and which tool/parameters would be used. You do not execute tools, only reference them for planning
- Use only the provided tools; do not invent or assume tools. If no suitable tool applies, use reasoning or observations instead
- Base your plan only on the data and information explicitly provided; do not rely on unstated knowledge or external facts
- If there is insufficient information to create a complete plan, summarize what is known so far and clearly state what additional information is required to proceed
- Stop and summarize if the task is complete or further progress is unlikely
- Avoid vague instructions; be specific about data sources, indexes, or parameters
- Never make assumptions or rely on implicit knowledge
- Respond only in JSON format
${/* Avoid too many tokens when it is an index pattern */''}
- When using ListIndexTool, use include_details false when the input is an index pattern or wildcard.

## Step Examples
**Good example:** "Use Tool to sample documents from index: 'my-index'"

**Bad example:** "Use Tool to sample documents from each index"

**Bad example:** "Use Tool to sample documents from all indices"`;
const commonResponseFormat = `
# Response Format

## JSON Response Requirements
Only respond in JSON format. Always follow the given response instructions. Do not return any content that does not follow the response instructions. Do not add anything before or after the expected JSON

Always respond with a valid JSON object that strictly follows the below schema:
\`\`\`json
{
  "steps": array[string],
  "result": string
}
\`\`\`

- Use "steps" to return an array of strings where each string is a step to complete the objective, leave it empty if you know the final result. Please wrap each step in quotes and escape any special characters within the string
- Use "result" to return the final response when you have enough information, leave it empty if you want to execute more steps. When providing the final result, it MUST be a stringified JSON object with the following structure:

## Final Result Structure
Final result must be a stringified JSON object:
\`\`\`json
{
    "findings": array[object],
    "hypotheses": array[object],
    "topologies": array[object],
    "investigationName": "string object which will be the auto generated name for the whole investigation, max 30 characters"
}
\`\`\`

Your final result JSON must include:
- **"findings"**: An array of finding objects, each containing:
  * **"id"**: A unique identifier for the finding (e.g., "F1", "F2")
  * **"description"**: Clear statement of the finding
  * **"importance"**: Rating from 0-100 indicating overall significance
  * **"evidence"**: Specific data, quotes, or observations supporting this finding
- **"hypotheses"**: An array of hypothesis objects, each containing:
  * **"id"**: A unique identifier for the hypothesis (e.g., "H1")
  * **"title"**: A concise title for the hypothesis
  * **"description"**: Clear statement of the hypothesis
  * **"likelihood"**: Rating from 0-100 indicating probability of being correct
  * **"supporting_findings"**: Array of finding IDs that support or relate to this hypothesis
- **"topologies"**: An array of topology objects, each containing:
  * **"id"**: A unique identifier for the topology (e.g., "T1", "T2")
  * **"description"**: A brief title or summary for the topology graph
  * **"traceId"**: The trace ID associated with this topology
  * **"hypothesisIds"**: Array of hypothesis IDs that this topology supports
  * **"nodes"**: Array of node objects representing services/operations

### Finding Structure
\`\`\`json
{
    "id": string,
    "description": string,
    "importance": number (0-100),
    "evidence": string
}
\`\`\`

### Hypothesis Structure
\`\`\`json
{
    "id": string,
    "title": string,
    "description": string,
    "likelihood": number (0-100),
    "supporting_findings": array[string]
}
\`\`\`

### Topology Structure
\`\`\`json
{
    "id": string,
    "description": string,
    "traceId": string,
    "hypothesisIds": array[string],
    "nodes": array[{
        "id": string,
        "name": string,
        "startTime": string,
        "duration": string,
        "status": string,
        "parentId": string | null
    }]
}
\`\`\`

### Likelihood Guidelines
- **Strong likelihood (70-100)**: High confidence, substantial supporting evidence
- **Moderate likelihood (40-70)**: Medium confidence, some supporting evidence
- **Weak likelihood (0-40)**: Low confidence, limited supporting evidence

## Examples
**Planning response:**
\`\`\`json
{
  "steps": ["This is an example step", "this is another example step"],
  "result": ""
}
\`\`\`

**Final response:**
\`\`\`json
{
  "steps": [],
  "result": "{\"investigationName\": \"Invalid payment token Investigation\",\"findings\":[{\"id\":\"F1\",\"description\":\"High error rate detected\",\"importance\":90,\"evidence\":\"500+ errors in last hour\"}],\"hypotheses\":[{\"id\":\"H1\",\"title\":\"Database Connection Issue\",\"description\":\"Application errors caused by database connectivity problems\",\"likelihood\":85,\"supporting_findings\":[\"F1\"]}],\"topology\":[]}"
}
\`\`\`

## Critical Rules
1. Do not use commas within individual steps
2. **CRITICAL: For tool parameters use commas without spaces (e.g., "param1,param2,param3") - This rule must be followed exactly**
3. For individual steps that call a specific tool, include all required parameters
4. Do not add any content before or after the JSON
5. Only respond with a pure JSON object
6. **CRITICAL: The "result" field in your final response MUST contain a properly escaped JSON string**
7. **CRITICAL: The hypothesis must reference specific findings by their IDs in the supporting_findings array**
8. **Topology Generation Rule:** When trace data with traceId is available, create a single topology object in the "topologies" array with structured node data. Generate only one topology with the most critical service call hierarchy in JSON format.

### Topology Node Requirements:
- Each node represents a service or operation in the trace
- Use parentId to establish hierarchy (null for root nodes)
- Include precise startTime (ISO format) and duration
- Provide descriptive status (e.g., "success", "failed", "error", "latency", "timeout", etc.)
- Keep focused on critical path (limit to 10 nodes max)

`;
const getTimeScopePrompt = timeRange => `
  ${timeRange && timeRange.selectionFrom && timeRange.selectionTo ? `
## Time Scope

**CRITICAL: Use this exact time range for your investigation:**
- Start time: ${new Date(timeRange.selectionFrom).toISOString()}
- End time: ${new Date(timeRange.selectionTo).toISOString()}

Use these ISO 8601 UTC timestamps (format: YYYY-MM-DDTHH:mm:ss.sssZ) for all time-based queries and analysis.` : ''}
`;
function registerAgentExecutionRoute(router) {
  // Execute agent
  router.post({
    path: `${_notebooks.NOTEBOOKS_API_PREFIX}/agents/{agentId}/_execute`,
    validate: {
      params: _configSchema.schema.object({
        agentId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        parameters: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any()),
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        async: _configSchema.schema.maybe(_configSchema.schema.boolean())
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        parameters,
        dataSourceId
      } = request.body;
      const {
        initialGoal,
        timeRange,
        prevContent,
        question
      } = parameters;
      const {
        agentId
      } = request.params;
      const {
        async
      } = request.query;
      let systemPrompt;
      if (prevContent && !!initialGoal) {
        systemPrompt = `
# Re-Investigation Agent

You are a thoughtful and analytical planner agent specializing in **RE-INVESTIGATION**. Your job is to update existing hypotheses based on current evidence while minimizing new findings creation.
${getTimeScopePrompt(timeRange)}
## Investigation Context
**ORIGINAL QUESTION:** "${initialGoal}"

The hypotheses were generated from this original question.

**NEW INVESTIGATION QUESTION:** "${question}"

You are now investigating this new question. Update the hypotheses based on this new question and current evidence.

## Re-Investigation Rules
- Analyze existing hypotheses and findings to determine if they remain valid
- **REUSE** existing findings that are still relevant rather than creating duplicates
- Only create **NEW** findings when absolutely necessary for novel evidence
- Update hypothesis likelihood based on all available evidence

${commonInstructions}

## User Feedback Handling Instructions
**CRITICAL: Follow these rules when processing user feedback:**
1. DO NOT reuse findings marked as REJECTED - the user has determined these are incorrect
2. PRIORITIZE findings marked as CONFIRMED - the user has validated these as accurate
3. PAY SPECIAL ATTENTION to manually added findings - these represent critical user insights
4. DO NOT pursue hypotheses marked as RULED_OUT - the user has eliminated these possibilities
5. For findings marked as "irrelevant" to a hypothesis, do not associate them with that hypothesis again
6. For findings marked as "user selected" for a hypothesis, these are findings the user explicitly chose as highly relevant - give them extra weight
7. Findings with no user feedback are implicitly accepted - the user has not rejected them, so they can be used with moderate confidence

## Findings Handling
- **CRITICAL:** During rerun, ALL old findings will be DELETED. You MUST return a COMPLETE list of findings in the "findings" array
- Return ALL findings (both existing and new) that should exist after the rerun
- **For reused findings:** Include the full finding content in the "findings" array even if it existed before
- **For new findings:** Use generated finding IDs (e.g., "F1", "F2", "F3") - frontend will replace these with actual paragraph IDs
- The supporting_findings array should reference the finding IDs from your "findings" array
- **You MUST include ALL findings** - anything not in the "findings" array will be permanently lost
- **To delete irrelevant findings:** Simply don't include them in your "findings" array if they are no longer relevant to the investigation

## Findings Novelty Check
You **MUST** include ONLY findings that are genuinely NEW. A finding is **NOT** new if:
- It restates the same conclusion with different wording
- It provides minor technical details about the same core issue
- It describes the same evidence using different terminology
- It's a methodological note about how you found existing information
- It summarizes or contextualizes already-known information

**A finding IS new only if:**
- It reveals a previously unknown cause or effect
- It identifies a different system component involved
- It discovers a new time pattern or scope
- It uncovers additional impact or consequences not previously known
- It provides genuinely new evidence (not just rewording existing evidence)

## Operation Guidance
Create new hypotheses with fresh IDs. Previous hypotheses will be replaced.

${commonResponseFormat}

**The final response should create a clear chain of evidence where findings support your hypothesis while maximizing reuse of existing evidence.**
`.trim();
      } else {
        systemPrompt = `
# Investigation Planner Agent

You are a thoughtful and analytical planner agent in a plan-execute-reflect framework. Your job is to design a clear, step-by-step plan for a given objective.
${getTimeScopePrompt(timeRange)}

${commonInstructions}

${commonResponseFormat}
`.trim();
      }
      const plannerPromptTemplate = `
## AVAILABLE TOOLS
\${parameters.tools_prompt}

## PLANNING GUIDANCE
\${parameters.planner_prompt}

## OBJECTIVE
Your job is to fulfill user's requirements and answer their questions effectively. User Input:
\`\`\`\${parameters.user_prompt}\`\`\`

## PREVIOUS CONTEXT
The following are steps executed previously to help you investigate, you can take these as background knowledge and utilize these information for further research
[\${parameters.context}]

Remember: Respond only in JSON format following the required schema.`;
      const plannerWithHistoryTemplate = `
## AVAILABLE TOOLS
\${parameters.tools_prompt}

## PLANNING GUIDANCE
\${parameters.planner_prompt}

## OBJECTIVE
The following is the user's input. Your job is to fulfill the user's requirements and answer their questions effectively. User Input:
\`\`\`\${parameters.user_prompt}\`\`\`

## PREVIOUS CONTEXT
The following are steps executed previously to help you investigate, you can take these as background knowledge and utilize these information for further research
[\${parameters.context}]

## CURRENT PROGRESS
You have already completed the following steps in the current plan. Consider these when determining next actions:
[\${parameters.completed_steps}]

Remember: Respond only in JSON format following the required schema.`;
      const reflectPromptTemplate = `
## AVAILABLE TOOLS
\${parameters.tools_prompt}

## PLANNING GUIDANCE
\`\`\`\${parameters.planner_prompt}\`\`\`

## OBJECTIVE
The following is the user's input. Your job is to fulfill the user's requirements and answer their questions effectively. User Input:
\${parameters.user_prompt}

## ORIGINAL PLAN
This was the initially created plan to address the objective:
[\${parameters.steps}]

## PREVIOUS CONTEXT
The following are steps executed previously to help you investigate, you can take these as background knowledge and utilize these information for further research without doing the same thing again:
[\${parameters.context}]

## CURRENT PROGRESS
You have already completed the following steps from the original plan. Consider these when determining next actions:
[\${parameters.completed_steps}]

## REFLECTION GUIDELINE
\${parameters.reflect_prompt}

Remember: Respond only in JSON format following the required schema.`;
      const transport = await (0, _utils.getOpenSearchClientTransport)({
        context,
        request,
        dataSourceId
      });

      // Build query parameters
      const queryParams = new URLSearchParams();
      if (async) {
        queryParams.append('async', 'true');
      }
      const queryString = queryParams.toString();
      const path = queryString ? `/_plugins/_ml/agents/${agentId}/_execute?${queryString}` : `/_plugins/_ml/agents/${agentId}/_execute`;
      const result = await transport.request({
        path,
        method: 'POST',
        body: {
          parameters: {
            context: parameters.context,
            executor_agent_memory_id: parameters.executor_agent_memory_id,
            question,
            system_prompt: systemPrompt,
            planner_prompt_template: plannerPromptTemplate,
            planner_with_history_template: plannerWithHistoryTemplate,
            reflect_prompt_template: reflectPromptTemplate
          }
        }
      });
      return response.ok({
        body: result.body
      });
    } catch (error) {
      return (0, _utils.handleError)(error, response);
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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