"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addClickToMetric = addClickToMetric;
exports.addRequestToMetric = addRequestToMetric;
exports.getMetrics = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("./constants");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const time2CountWin = new Map();
function addClickToMetric(element, counter = 'count') {
  // remove outdated key-value pairs
  trim();
  const timeKey = getKey(Date.now());
  const rollingCounter = time2CountWin.get(timeKey) || _lodash.default.cloneDeep(_constants.DEFAULT_ROLLING_COUNTER);
  const key = `click.${element}.${counter}`;
  _lodash.default.set(rollingCounter, key, _lodash.default.get(rollingCounter, key, 0) + 1);
  if (counter === 'count') {
    const basicCounterKey = `click.${element}.total`;
    _lodash.default.set(_constants.GLOBAL_BASIC_COUNTER, basicCounterKey, _lodash.default.get(_constants.GLOBAL_BASIC_COUNTER, basicCounterKey, 0) + 1);
  }
  time2CountWin.set(timeKey, rollingCounter);
}
function addRequestToMetric(component, request, counterNameOrError) {
  const counter = typeof counterNameOrError === 'object' ? checkErrorType(counterNameOrError) : counterNameOrError;

  // remove outdated key-value pairs
  trim();
  const timeKey = getKey(Date.now());
  const rollingCounter = time2CountWin.get(timeKey) || _lodash.default.cloneDeep(_constants.DEFAULT_ROLLING_COUNTER);
  rollingCounter[component][request][counter]++;
  if (counter === 'count') {
    _constants.GLOBAL_BASIC_COUNTER[component][request].total++;
  }
  time2CountWin.set(timeKey, rollingCounter);
}
const getMetrics = () => {
  const preTimeKey = getPreKey(Date.now());
  const rollingCounters = time2CountWin.get(preTimeKey);
  return buildMetrics(rollingCounters);
};
exports.getMetrics = getMetrics;
const checkErrorType = error => {
  if (error.statusCode && Math.floor(error.statusCode / 100) === 4) {
    return 'user_error';
  } else {
    return 'system_error';
  }
};
const trim = () => {
  if (time2CountWin.size > _constants.CAPACITY) {
    const currentKey = getKey(Date.now() - _constants.WINDOW * _constants.MILLIS_MULTIPLIER);
    time2CountWin.forEach((_value, key, map) => {
      if (key < currentKey) {
        map.delete(key);
      }
    });
  }
};
const getKey = milliseconds => {
  return Math.floor(milliseconds / _constants.MILLIS_MULTIPLIER / _constants.INTERVAL);
};
const getPreKey = milliseconds => {
  return getKey(milliseconds) - 1;
};
const isComponent = arg => {
  return _constants.COMPONENTS.includes(arg);
};
const buildMetrics = rollingCounters => {
  if (!rollingCounters) {
    rollingCounters = _constants.DEFAULT_ROLLING_COUNTER;
  }
  const basicMetrics = _lodash.default.merge(rollingCounters, _constants.GLOBAL_BASIC_COUNTER);
  const overallActionMetrics = {
    request_total: 0,
    request_count: 0,
    success_count: 0,
    failed_request_count_system_error: 0,
    failed_request_count_user_error: 0
  };
  Object.keys(basicMetrics).forEach(key => {
    if (isComponent(key)) {
      for (const counter of Object.values(basicMetrics[key])) {
        overallActionMetrics.request_count += (counter === null || counter === void 0 ? void 0 : counter.count) || 0;
        overallActionMetrics.request_total += (counter === null || counter === void 0 ? void 0 : counter.total) || 0;
        overallActionMetrics.failed_request_count_system_error += (counter === null || counter === void 0 ? void 0 : counter.system_error) || 0;
        overallActionMetrics.failed_request_count_user_error += (counter === null || counter === void 0 ? void 0 : counter.user_error) || 0;
      }
    }
  });
  overallActionMetrics.success_count = overallActionMetrics.request_count - (overallActionMetrics.failed_request_count_system_error + overallActionMetrics.failed_request_count_user_error);
  return {
    ...basicMetrics,
    ...overallActionMetrics
  };
};
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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