"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSampleNotebooks = void 0;
var _uuid = require("uuid");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const getDemoNotebook = (dateString, visId) => {
  const uuids = [(0, _uuid.v4)()];
  const oneWeekAgo = new Date();
  oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
  return {
    name: 'OpenSearch Notebooks Quick Start Guide',
    path: 'OpenSearch Notebooks Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'kibana_1.0',
    context: {
      notebookType: 'Classic'
    },
    paragraphs: [{
      output: [{
        result: `An OpenSearch Dashboards notebook is an interface that lets you easily combine code snippets, live visualizations, and narrative text in a single notebook interface.

Notebooks let you interactively explore data by running different visualizations that you can share with team members to collaborate on a project.

A notebook is a document composed of two elements: code blocks (Markdown/SQL/PPL) and visualizations.

Common use cases include creating postmortem reports, designing runbooks, building live infrastructure reports, and writing documentation.

You can also generate [reports](https://opensearch.org/docs/dashboards/reporting/) directly from your notebooks.

For more information, refer to the [documentation](https://opensearch.org/docs/dashboards/notebooks/).`,
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
An OpenSearch Dashboards notebook is an interface that lets you easily combine code snippets, live visualizations, and narrative text in a single notebook interface.

Notebooks let you interactively explore data by running different visualizations that you can share with team members to collaborate on a project.

A notebook is a document composed of two elements: code blocks (Markdown/SQL/PPL) and visualizations.

Common use cases include creating postmortem reports, designing runbooks, building live infrastructure reports, and writing documentation.

You can also generate [reports](https://opensearch.org/docs/dashboards/reporting/) directly from your notebooks.

For more information, refer to the [documentation](https://opensearch.org/docs/dashboards/notebooks/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Notebooks combine code blocks and visualizations for describing data. Code blocks support markdown, SQL, and PPL languages. Specify the input language on the first line using %[language type] syntax. For example, type %md for markdown, %sql for SQL, and %ppl for PPL. A sample visualization is shown below:',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
Notebooks combine code blocks and visualizations for describing data. Code blocks support markdown, SQL, and PPL languages. Specify the input language on the first line using %\[language type\] syntax. For example, type %md for markdown, %sql for SQL, and %ppl for PPL. A sample visualization is shown below:`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visId}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[0]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[0]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'You can add a new paragraph from the **Paragraph actions** menu on the top right of the page, the context menu on the top right of each paragraph, or the **Add paragraph** button on the bottom of the page.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
You can add a new paragraph from the **Paragraph actions** menu on the top right of the page, the context menu on the top right of each paragraph, or the **Add paragraph** button on the bottom of the page.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'You can also reorder, duplicate, or delete paragraphs from these menus.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
You can also reorder, duplicate, or delete paragraphs from these menus.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "To execute a paragraph, choose **Run**. To make changes to the input block, choose the downward arrow that's next to the paragraph title.",
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
To execute a paragraph, choose **Run**. To make changes to the input block, choose the downward arrow that's next to the paragraph title.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `There are three view modes:
- **View both**: allows you to edit paragraphs and view outputs on the same page.
- **Input only**: only shows input of each paragraph, allowing easier editing.
- **Output only**: only shows output of each paragraph and hides panels.`,
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
There are three view modes:
- **View both**: allows you to edit paragraphs and view outputs on the same page.
- **Input only**: only shows input of each paragraph, allowing easier editing.
- **Output only**: only shows output of each paragraph and hides panels.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To go back, choose **Notebooks** on the top left of your screen.',
        outputType: 'MARKDOWN',
        execution_time: '0 ms'
      }],
      input: {
        inputText: `%md
To go back, choose **Notebooks** on the top left of your screen.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};
const getRootCauseNotebook = (dateString, visIds) => {
  const uuids = [(0, _uuid.v4)(), (0, _uuid.v4)()];
  const oneWeekAgo = new Date();
  oneWeekAgo.setDate(oneWeekAgo.getDate() - 7);
  return {
    name: '[Logs] Sample Root Cause Event Analysis',
    path: '[Logs] Sample Root Cause Event Analysis',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'kibana_1.0',
    context: {
      notebookType: 'Classic'
    },
    paragraphs: [{
      output: [{
        result: 'This notebook goes through a sample root cause event analysis using PPL and visualizations with the OpenSearch Dashboards sample web logs data.',
        outputType: 'MARKDOWN',
        execution_time: '0.016 ms'
      }],
      input: {
        inputText: `%md
This notebook goes through a sample root cause event analysis using PPL and visualizations with the OpenSearch Dashboards sample web logs data.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "First, let's bring up a few visualizations for a quick overview.",
        outputType: 'MARKDOWN',
        execution_time: '0.046 ms'
      }],
      input: {
        inputText: `%md
First, let's bring up a few visualizations for a quick overview.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visIds[0]}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[0]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[0]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: '',
        outputType: 'VISUALIZATION',
        execution_time: '0.017 ms'
      }],
      input: {
        inputText: `{"viewMode":"view","panels":{"1":{"gridData":{"x":0,"y":0,"w":50,"h":20,"i":"1"},"type":"visualization","explicitInput":{"id":"1","savedObjectId":"${visIds[1]}"}}},"isFullScreenMode":false,"filters":[],"useMargins":false,"id":"${uuids[1]}","timeRange":{"to":"${dateString}","from":"${oneWeekAgo}"},"title":"embed_viz_${uuids[1]}","query":{"query":"","language":"lucene"},"refreshConfig":{"pause":true,"value":15}}`,
        inputType: 'VISUALIZATION'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Let's take a look at the source data by the selected fields (search and fields).`,
        outputType: 'MARKDOWN',
        execution_time: '0.013 ms'
      }],
      input: {
        inputText: `%md
Let's take a look at the source data by the selected fields (search and fields).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, 
  response, bytes | head 20

`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, 
  response, bytes | head 20

`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Check for any error log with response code 404 or 503 (filter).',
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Check for any error log with response code 404 or 503 (filter).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404' | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404' | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "We see too many events. Let's quickly check which host has the issue (dedup).",
        outputType: 'MARKDOWN',
        execution_time: '0.014 ms'
      }],
      input: {
        inputText: `%md
We see too many events. Let's quickly check which host has the issue (dedup).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | dedup host | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | dedup host | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "We get too few events. Let's dedup in consecutive mode (dedup).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
We get too few events. Let's dedup in consecutive mode (dedup).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    dedup host consecutive=true | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    dedup host consecutive=true | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'How many IP addresses for each response (stats).',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
How many IP addresses for each response (stats).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count by response | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count by response | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "To dive deep, let's group by host and response, count, and sum (stats).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To dive deep, let's group by host and response, count, and sum (stats).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `We don't see a meaningful response. Let's change to resp_code (rename).`,
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
We don't see a meaningful response. Let's change to resp_code (rename).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response |
     rename response as resp_code | head 20

`,
        outputType: 'QUERY',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response |
     rename response as resp_code | head 20

`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "The data looks better now. Let's sort by `DESC count` and `ASC sum_bytes` (sort).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
The data looks better now. Let's sort by \`DESC count\` and \`ASC sum_bytes\` (sort).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "Let's check if we can perform aggregations after stats (eval).",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
Let's check if we can perform aggregations after stats (eval).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response | 
    rename response as resp_code | 
    sort - ip_count, + sum_bytes | 
    eval per_ip_bytes=sum_bytes/ip_count  | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' |
    stats count() as ip_count, sum(bytes) 
    as sum_bytes by host, response | 
    rename response as resp_code | 
    sort - ip_count, + sum_bytes | 
    eval per_ip_bytes=sum_bytes/ip_count  | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Wait, what's meant by an evaluation. Can we really perform an evaluation?`,
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
Wait, what's meant by an evaluation. Can we really perform an evaluation?`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | 
   eval per_ip_bytes=sum_bytes/ip_count, 
    double_per_ip_bytes = 2 * per_ip_bytes | head 20
`,
        outputType: 'QUERY',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%ppl
  source=opensearch_dashboards_sample_data_logs | fields host, clientip, response,
   bytes | where response='503' or response='404' | 
   stats count() as ip_count, sum(bytes) 
   as sum_bytes by host, response | 
   rename response as resp_code | 
   sort - ip_count, + sum_bytes | 
   eval per_ip_bytes=sum_bytes/ip_count, 
    double_per_ip_bytes = 2 * per_ip_bytes | head 20
`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};
const getSQLNotebook = dateString => {
  return {
    name: '[Flights] OpenSearch SQL Quick Start Guide',
    path: '[Flights] OpenSearch SQL Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'kibana_1.0',
    context: {
      notebookType: 'Classic'
    },
    paragraphs: [{
      output: [{
        result: `OpenSearch SQL lets you write queries in SQL rather than the [OpenSearch query domain-specific language (DSL)](https://opensearch.org/docs/opensearch/query-dsl/full-text/). If you’re already familiar with SQL and don’t want to learn the query DSL, this feature is a great option.

For more information, please refer to the [documentation](https://opensearch.org/docs/search-plugins/sql/index/).`,
        outputType: 'MARKDOWN',
        execution_time: '0.013 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL lets you write queries in SQL rather than the [OpenSearch query domain-specific language (DSL)](https://opensearch.org/docs/opensearch/query-dsl/full-text/). If you’re already familiar with SQL and don’t want to learn the query DSL, this feature is a great option.

For more information, please refer to the [documentation](https://opensearch.org/docs/search-plugins/sql/index/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'This notebook shows you how to use SQL with the [OpenSearch Dashboards sample flight data](/app/home#/tutorial_directory).',
        outputType: 'MARKDOWN',
        execution_time: '0.010 ms'
      }],
      input: {
        inputText: `%md
This notebook shows you how to use SQL with the [OpenSearch Dashboards sample flight data](/app/home#/tutorial_directory).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To use SQL, add a code paragraph, type %sql on the first line, and then add SQL queries on the next line.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
To use SQL, add a code paragraph, type %sql on the first line, and then add SQL queries on the next line.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'Select * from opensearch_dashboards_sample_data_flights limit 20;',
        outputType: 'QUERY',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%sql
Select * from opensearch_dashboards_sample_data_flights limit 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: "You can specify fields in the `SELECT` clause and use the `WHERE` clause to filter results. The following query finds flights heading to countries that start with 'A' that are more than 5000 miles away.",
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
You can specify fields in the \`SELECT\` clause and use the \`WHERE\` clause to filter results. The following query finds flights heading to countries that start with 'A' that are more than 5000 miles away.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT FlightNum,OriginCountry,OriginCityName,DestCountry,DestCityName,DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry LIKE 'A%' LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT FlightNum,OriginCountry,OriginCityName,DestCountry,DestCityName,DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry LIKE 'A%' LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL also supports subqueries:',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL also supports subqueries:`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT opensearch_dashboards_sample_data_flights.FlightNum,opensearch_dashboards_sample_data_flights.OriginCountry,opensearch_dashboards_sample_data_flights.OriginCityName,opensearch_dashboards_sample_data_flights.DestCountry,opensearch_dashboards_sample_data_flights.DestCityName,opensearch_dashboards_sample_data_flights.DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE FlightNum IN (SELECT FlightNum FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry = 'AU') LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%sql
SELECT opensearch_dashboards_sample_data_flights.FlightNum,opensearch_dashboards_sample_data_flights.OriginCountry,opensearch_dashboards_sample_data_flights.OriginCityName,opensearch_dashboards_sample_data_flights.DestCountry,opensearch_dashboards_sample_data_flights.DestCityName,opensearch_dashboards_sample_data_flights.DistanceMiles FROM opensearch_dashboards_sample_data_flights WHERE FlightNum IN (SELECT FlightNum FROM opensearch_dashboards_sample_data_flights WHERE DistanceMiles > 5000 AND DestCountry = 'AU') LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL supports inner joins, cross joins, and left outer joins. The following query joins the flights index with itself to find flights departed from countries that are both origins and destinations.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL supports inner joins, cross joins, and left outer joins. The following query joins the flights index with itself to find flights departed from countries that are both origins and destinations.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
SELECT a.FlightNum,a.OriginCountry,a.OriginCityName,a.DestCountry,a.DestCityName,a.DistanceMiles FROM opensearch_dashboards_sample_data_flights a JOIN opensearch_dashboards_sample_data_flights b on a.OriginCountry = b.DestCountry LIMIT 20`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT a.FlightNum,a.OriginCountry,a.OriginCityName,a.DestCountry,a.DestCityName,a.DistanceMiles FROM opensearch_dashboards_sample_data_flights a JOIN opensearch_dashboards_sample_data_flights b on a.OriginCountry = b.DestCountry LIMIT 20`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'For aggregations, use the `GROUP BY` clause. The following query finds the countries with more than 500 flights departed.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
For aggregations, use the \`GROUP BY\` clause. The following query finds the countries with more than 500 flights departed.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT OriginCountry,COUNT(1) FROM opensearch_dashboards_sample_data_flights GROUP BY OriginCountry HAVING COUNT(1) > 500 LIMIT 20;`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%sql
SELECT OriginCountry,COUNT(1) FROM opensearch_dashboards_sample_data_flights GROUP BY OriginCountry HAVING COUNT(1) > 500 LIMIT 20;`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'OpenSearch SQL supports expressions.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
OpenSearch SQL supports expressions.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `SELECT abs(-1.234), abs(-1 * abs(-5)), dayofmonth(DATE '2021-07-07');`,
        outputType: 'QUERY',
        execution_time: '0.005 ms'
      }],
      input: {
        inputText: `%sql
SELECT abs(-1.234), abs(-1 * abs(-5)), dayofmonth(DATE '2021-07-07');`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};
const getPPLNotebook = dateString => {
  return {
    name: '[Logs] OpenSearch Piped Processing Language (PPL) Quick Start Guide',
    path: '[Logs] OpenSearch Piped Processing Language (PPL) Quick Start Guide',
    dateCreated: dateString,
    dateModified: dateString,
    backend: 'kibana_1.0',
    context: {
      notebookType: 'Classic'
    },
    paragraphs: [{
      output: [{
        result: `Query DSL is powerful and fast, but it has a steep learning curve because it doesn’t have a human-readable interface. It's also difficult to create ad hoc queries and explore your data. SQL lets you extract and analyze data in a declarative manner.

OpenSearch makes its search and query engine robust by introducing Piped Processing Language (PPL).

PPL enables developers, DevOps engineers, support engineers, site reliability engineers (SREs), and IT managers to effectively discover and explore log data stored in OpenSearch.

With PPL, you can extract insights from your data with a sequence of commands delimited by pipes (|). PPL supports a comprehensive set of commands including search, where, fields, rename, dedup, sort, eval, head, top, and rare. PPL also supports functions, operators and expressions.

Even if you're new to OpenSearch and are only familiar with the pipe (|) syntax, you can still be productive from day one.`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Query DSL is powerful and fast, but it has a steep learning curve because it doesn’t have a human-readable interface. It's also difficult to create ad hoc queries and explore your data. SQL lets you extract and analyze data in a declarative manner.

OpenSearch makes its search and query engine robust by introducing Piped Processing Language (PPL).

PPL enables developers, DevOps engineers, support engineers, site reliability engineers (SREs), and IT managers to effectively discover and explore log data stored in OpenSearch.

With PPL, you can extract insights from your data with a sequence of commands delimited by pipes (|). PPL supports a comprehensive set of commands including search, where, fields, rename, dedup, sort, eval, head, top, and rare. PPL also supports functions, operators and expressions.

Even if you're new to OpenSearch and are only familiar with the pipe (|) syntax, you can still be productive from day one.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `Piped Processing Language (PPL) is a query language that lets you use pipe (|) syntax to explore, discover, and query data stored in OpenSearch.

For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
Piped Processing Language (PPL) is a query language that lets you use pipe (|) syntax to explore, discover, and query data stored in OpenSearch.

For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'In this notebook, we will demonstrate some simple PPL queries with the [OpenSearch Dashboards sample web logs](/app/home#/tutorial_directory). ',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
In this notebook, we will demonstrate some simple PPL queries with the [OpenSearch Dashboards sample web logs](/app/home#/tutorial_directory). `,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To use PPL, add a code paragraph, type `%ppl` on the first line, and add your PPL query on the next line. PPL uses `source` to specify indices and `head` to limit results.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To use PPL, add a code paragraph, type \`%ppl\` on the first line, and add your PPL query on the next line. PPL uses \`source\` to specify indices and \`head\` to limit results.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | head 20`,
        outputType: 'QUERY',
        execution_time: '0.026 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | head 20`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To specify fields to include and filter results, use the `field` and `where` commands. The next query returns hosts with failed responses.',
        outputType: 'MARKDOWN',
        execution_time: '0.007 ms'
      }],
      input: {
        inputText: `%md
To specify fields to include and filter results, use the \`field\` and \`where\` commands. The next query returns hosts with failed responses.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404'`,
        outputType: 'QUERY',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | fields host, clientip, response, bytes | where response='503' or response='404'`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To see most common hosts from the previous result, use the `top` command.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To see most common hosts from the previous result, use the \`top\` command.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | top host`,
        outputType: 'QUERY',
        execution_time: '0.008 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | top host`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: 'To perform aggregations on search results, use the `stats` command.',
        outputType: 'MARKDOWN',
        execution_time: '0.006 ms'
      }],
      input: {
        inputText: `%md
To perform aggregations on search results, use the \`stats\` command.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | stats count(1) by host`,
        outputType: 'QUERY',
        execution_time: '0.011 ms'
      }],
      input: {
        inputText: `%ppl
source=opensearch_dashboards_sample_data_logs | where response='503' or response='404' | stats count(1) by host`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }, {
      output: [{
        result: `For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).

To see how to perform sample root cause event analysis using PPL and visualizations, see the "\[Logs\] Sample Root Cause Event Analysis" notebook.`,
        outputType: 'MARKDOWN',
        execution_time: '0.009 ms'
      }],
      input: {
        inputText: `%md
For more information, refer to the [documentation](https://opensearch.org/docs/search-plugins/ppl/index/).

To see how to perform sample root cause event analysis using PPL and visualizations, see the "\[Logs\] Sample Root Cause Event Analysis" notebook.`,
        inputType: 'MARKDOWN'
      },
      dateCreated: dateString,
      dateModified: dateString,
      id: 'paragraph_' + (0, _uuid.v4)()
    }]
  };
};
const getSampleNotebooks = visIds => {
  const dateString = new Date().toISOString();
  return [{
    savedNotebook: getDemoNotebook(dateString, visIds[2]),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    savedNotebook: getSQLNotebook(dateString),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    savedNotebook: getPPLNotebook(dateString),
    dateModified: dateString,
    dateCreated: dateString
  }, {
    savedNotebook: getRootCauseNotebook(dateString, visIds),
    dateModified: dateString,
    dateCreated: dateString
  }];
};
exports.getSampleNotebooks = getSampleNotebooks;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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