"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.batchCreateParagraphs = batchCreateParagraphs;
exports.batchDeleteParagraphs = batchDeleteParagraphs;
exports.batchRunParagraphs = batchRunParagraphs;
exports.batchSaveParagraphs = batchSaveParagraphs;
exports.createParagraph = createParagraph;
exports.createParagraphs = createParagraphs;
exports.deleteParagraphs = deleteParagraphs;
exports.fetchNotebook = fetchNotebook;
exports.runParagraph = runParagraph;
exports.updateFetchParagraph = updateFetchParagraph;
exports.updateParagraphs = updateParagraphs;
exports.updateRunFetchParagraph = updateRunFetchParagraph;
var _performanceNow = _interopRequireDefault(require("performance-now"));
var _uuid = require("uuid");
var _observability_saved_object_attributes = require("../../../common/types/observability_saved_object_attributes");
var _query_helpers = require("../../common/helpers/notebooks/query_helpers");
var _paragraph = require("../../../common/utils/paragraph");
var _paragraph2 = require("../../common/helpers/notebooks/paragraph");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function createParagraph({
  input,
  dataSourceMDSId,
  aiGenerated
}) {
  const finalInput = {
    ...input
  };
  try {
    let paragraphType = finalInput.inputType;
    const {
      inputText,
      inputType
    } = finalInput;
    if (inputType === 'CODE') {
      if (inputText.substring(0, 3) === '%sql' || inputText.substring(0, 3) === '%ppl') {
        paragraphType = 'QUERY';
      } else {
        paragraphType = 'MARKDOWN';
      }
    }
    const outputObjects = [{
      outputType: paragraphType,
      result: '',
      execution_time: '0s'
    }];
    const newParagraph = {
      id: 'paragraph_' + (0, _uuid.v4)(),
      dateCreated: new Date().toISOString(),
      dateModified: new Date().toISOString(),
      input: finalInput,
      output: outputObjects,
      dataSourceMDSId,
      aiGenerated
    };
    return newParagraph;
  } catch (error) {
    throw new Error('Create Paragraph Error:' + error);
  }
}
async function fetchNotebook(noteId, opensearchNotebooksClient) {
  try {
    const notebook = await opensearchNotebooksClient.get(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteId);
    return notebook;
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
async function createParagraphs(params, opensearchNotebooksClient) {
  const notebookInfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const paragraphs = notebookInfo.attributes.savedNotebook.paragraphs;
  const newParagraph = createParagraph({
    input: params.input,
    dataSourceMDSId: params.dataSourceMDSId,
    aiGenerated: params.aiGenerated
  });
  paragraphs.splice(params.paragraphIndex, 0, newParagraph);
  const updateNotebook = {
    paragraphs,
    dateModified: new Date().toISOString()
  };
  await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
    savedNotebook: updateNotebook
  });
  await fetchNotebook(params.noteId, opensearchNotebooksClient);
  return newParagraph;
}
async function batchCreateParagraphs(params, opensearchNotebooksClient) {
  const notebookInfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const paragraphs = [...notebookInfo.attributes.savedNotebook.paragraphs];
  const newParagraphs = params.paragraphs.map(p => createParagraph(p));
  paragraphs.splice(params.startIndex, 0, ...newParagraphs);
  const updateNotebook = {
    paragraphs,
    dateModified: new Date().toISOString()
  };
  await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
    savedNotebook: updateNotebook
  });
  return {
    paragraphs: newParagraphs
  };
}
async function batchRunParagraphs(params, opensearchNotebooksClient) {
  const currentNotebook = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  let currentParagraphs = [...currentNotebook.attributes.savedNotebook.paragraphs];
  for (const paragraphData of params.paragraphs) {
    try {
      // Update paragraph input first
      currentParagraphs = updateParagraphs(currentParagraphs, paragraphData.id, paragraphData.input, paragraphData.dataSourceMDSId);
      // Then run the paragraph
      currentParagraphs = await runParagraph(currentParagraphs, paragraphData.id, currentNotebook);
    } catch (e) {
      console.error('Failed to run paragraph:', paragraphData.id, e);
    }
  }
  await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
    savedNotebook: {
      paragraphs: currentParagraphs,
      dateModified: new Date().toISOString()
    }
  });
  const paragraphIds = params.paragraphs.map(p => p.id);
  const ranParagraphs = currentParagraphs.filter(p => paragraphIds.includes(p.id));
  return {
    paragraphs: ranParagraphs
  };
}
async function batchSaveParagraphs(params, opensearchNotebooksClient) {
  const currentNotebook = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  let currentParagraphs = [...currentNotebook.attributes.savedNotebook.paragraphs];
  for (const paragraphData of params.paragraphs) {
    currentParagraphs = updateParagraphs(currentParagraphs, paragraphData.paragraphId, paragraphData.input, paragraphData.dataSourceMDSId, paragraphData.output);
  }
  await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
    savedNotebook: {
      paragraphs: currentParagraphs,
      dateModified: new Date().toISOString()
    }
  });
  const paragraphIds = params.paragraphs.map(p => p.paragraphId);
  const savedParagraphs = currentParagraphs.filter(p => paragraphIds.includes(p.id));
  return {
    paragraphs: savedParagraphs
  };
}
async function deleteParagraphs(params, opensearchNotebooksClient) {
  const noteBookInfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const updatedparagraphs = [];
  if (params.paragraphId !== undefined) {
    noteBookInfo.attributes.savedNotebook.paragraphs.map(paragraph => {
      if (paragraph.id !== params.paragraphId) {
        updatedparagraphs.push(paragraph);
      }
    });
  }
  noteBookInfo.attributes.savedNotebook.paragraphs = updatedparagraphs;
  try {
    noteBookInfo.attributes.savedNotebook.dateModified = new Date().toISOString();
    await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteBookInfo.attributes, {
      id: params.noteId,
      overwrite: true,
      version: noteBookInfo.version
    });
    return {
      paragraphs: updatedparagraphs
    };
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
async function batchDeleteParagraphs(params, opensearchNotebooksClient) {
  const noteBookInfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
  const updatedparagraphs = noteBookInfo.attributes.savedNotebook.paragraphs.filter(paragraph => !params.paragraphIds.includes(paragraph.id));
  noteBookInfo.attributes.savedNotebook.paragraphs = updatedparagraphs;
  noteBookInfo.attributes.savedNotebook.dateModified = new Date().toISOString();
  try {
    const result = await opensearchNotebooksClient.create(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, noteBookInfo.attributes, {
      id: params.noteId,
      overwrite: true,
      version: noteBookInfo.version
    });
    return {
      result
    };
  } catch (error) {
    throw new Error('delete Paragraphs Error:' + error);
  }
}
async function updateRunFetchParagraph(params, opensearchNotebooksClient) {
  try {
    const notebookInfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
    const updatedInputParagraphs = updateParagraphs(notebookInfo.attributes.savedNotebook.paragraphs, params.paragraphId, params.input, params.dataSourceMDSId);
    const updatedOutputParagraphs = await runParagraph(updatedInputParagraphs, params.paragraphId, notebookInfo);
    const updateNotebook = {
      paragraphs: updatedOutputParagraphs,
      dateModified: new Date().toISOString()
    };
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    let resultParagraph = {};
    let index = 0;
    for (index = 0; index < updatedOutputParagraphs.length; ++index) {
      if (params.paragraphId === updatedOutputParagraphs[index].id) {
        resultParagraph = updatedOutputParagraphs[index];
      }
    }
    return resultParagraph;
  } catch (error) {
    throw new Error('Update/Run Paragraph Error:' + error);
  }
}
async function runParagraph(paragraphs, paragraphId, notebookinfo) {
  try {
    const updatedParagraphs = [];
    let index = 0;
    for (index = 0; index < paragraphs.length; ++index) {
      const startTime = (0, _performanceNow.default)();

      // use string as default
      const updatedParagraph = {
        ...paragraphs[index]
      };
      const inputType = (0, _paragraph.getInputType)(updatedParagraph);
      if (paragraphs[index].id === paragraphId) {
        updatedParagraph.dateModified = new Date().toISOString();
        if ((0, _query_helpers.inputIsQuery)(paragraphs[index].input.inputText)) {
          updatedParagraph.output = [{
            outputType: 'QUERY',
            result: (0, _paragraph2.updateParagraphText)(paragraphs[index].input.inputText, notebookinfo),
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (paragraphs[index].input.inputText.substring(0, 3) === '%md') {
          updatedParagraph.output = [{
            outputType: 'MARKDOWN',
            result: (0, _paragraph2.updateParagraphText)(paragraphs[index].input.inputText, notebookinfo),
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (inputType === 'VISUALIZATION') {
          updatedParagraph.dateModified = new Date().toISOString();
          updatedParagraph.output = [{
            outputType: 'VISUALIZATION',
            result: '',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (inputType === 'OBSERVABILITY_VISUALIZATION') {
          updatedParagraph.dateModified = new Date().toISOString();
          updatedParagraph.output = [{
            outputType: 'OBSERVABILITY_VISUALIZATION',
            result: '',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if (inputType === 'LOG_PATTERN') {
          updatedParagraph.dateModified = new Date().toISOString();
          updatedParagraph.output = [{
            outputType: 'LOG_PATTERN',
            result: '',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        } else if ((0, _query_helpers.formatNotRecognized)(paragraphs[index].input.inputText)) {
          updatedParagraph.output = [{
            outputType: 'MARKDOWN',
            result: 'Please select an input type (%sql, %ppl, or %md)',
            execution_time: `${((0, _performanceNow.default)() - startTime).toFixed(3)} ms`
          }];
        }
      }
      updatedParagraphs.push(updatedParagraph);
    }
    return updatedParagraphs;
  } catch (error) {
    throw new Error('Running Paragraph Error:' + error);
  }
}
function updateParagraphs(paragraphs, paragraphId, input, dataSourceMDSId, paragraphOutput) {
  try {
    const updatedParagraphs = [];
    paragraphs.map(paragraph => {
      const updatedParagraph = {
        ...paragraph
      };
      if (paragraph.id === paragraphId) {
        updatedParagraph.dataSourceMDSId = dataSourceMDSId !== null && dataSourceMDSId !== void 0 ? dataSourceMDSId : paragraph.dataSourceMDSId;
        updatedParagraph.dateModified = new Date().toISOString();
        updatedParagraph.input = input;
        if (paragraphOutput) {
          updatedParagraph.output = paragraphOutput;
        }
      }
      updatedParagraphs.push(updatedParagraph);
    });
    return updatedParagraphs;
  } catch (error) {
    throw new Error('Update Paragraph Error:' + error);
  }
}
async function updateFetchParagraph(params, opensearchNotebooksClient) {
  try {
    const notebookinfo = await fetchNotebook(params.noteId, opensearchNotebooksClient);
    const updatedInputParagraphs = updateParagraphs(notebookinfo.attributes.savedNotebook.paragraphs, params.paragraphId, params.input, params.dataSourceMDSId, params.output);
    const updateNotebook = {
      paragraphs: updatedInputParagraphs,
      dateModified: new Date().toISOString()
    };
    await opensearchNotebooksClient.update(_observability_saved_object_attributes.NOTEBOOK_SAVED_OBJECT, params.noteId, {
      savedNotebook: updateNotebook
    });
    let resultParagraph = {};
    updatedInputParagraphs.map(paragraph => {
      if (params.paragraphId === paragraph.id) {
        resultParagraph = paragraph;
      }
    });
    return resultParagraph;
  } catch (error) {
    throw new Error('update Paragraph Error:' + error);
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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