"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _DataStreamService = require("./DataStreamService");
var _MDSEnabledClientService = require("./MDSEnabledClientService");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
class IndexService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "getIndices", async (context, request, response) => {
      try {
        // @ts-ignore
        const {
          from,
          size,
          sortField,
          sortDirection,
          terms,
          indices,
          dataStreams,
          showDataStreams,
          expandWildcards,
          exactSearch
        } = request.query;
        const params = {
          index: (0, _helpers.getSearchString)(terms, indices, dataStreams),
          format: "json"
        };
        if (sortField !== "managed" && sortField !== "data_stream") {
          params.s = `${sortField}:${sortDirection}`;
        }
        if (expandWildcards) {
          params.expand_wildcards = expandWildcards;
        }
        if (exactSearch) {
          params.index = exactSearch;
        }
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const [recoverys, tasks, indicesResponse, indexToDataStreamMapping] = await Promise.all([callWithRequest("cat.recovery", {
          format: "json",
          detailed: true
        }).catch(() => []), callWithRequest("cat.tasks", {
          format: "json",
          detailed: true,
          actions: "indices:data/write/reindex"
        }).catch(() => []), callWithRequest("cat.indices", params), (0, _DataStreamService.getIndexToDataStreamMapping)(callWithRequest)]);
        const formattedTasks = tasks.map(item => {
          const {
            description
          } = item;
          const regexp = /reindex from \[([^\]]+)\] to \[([^\]]+)\]/i;
          const matchResult = description.match(regexp);
          if (matchResult) {
            const [, fromIndex, toIndex] = matchResult;
            return {
              ...item,
              fromIndex,
              toIndex
            };
          } else {
            return {
              ...item,
              fromIndex: "",
              toIndex: ""
            };
          }
        });
        const onGoingRecovery = recoverys.filter(item => item.stage !== "done");

        // Augment the indices with their parent data stream name.
        indicesResponse.forEach(index => {
          index.data_stream = indexToDataStreamMapping[index.index] || null;
          let extraStatus = index.status;
          if (index.health === "green") {
            if (formattedTasks.find(item => item.toIndex === index.index)) {
              extraStatus = "reindex";
            }
          } else {
            if (onGoingRecovery.find(item => item.index === index.index)) {
              extraStatus = "recovery";
            }
          }
          if (extraStatus) {
            index.extraStatus = extraStatus;
          }
        });
        function customSort(array, key, sortDirection) {
          return array.sort((a, b) => {
            let flag;
            const aValue = a[key];
            const bValue = b[key];
            if (sortDirection === "asc") {
              flag = aValue < bValue;
            } else {
              flag = aValue > bValue;
            }
            return flag ? -1 : 1;
          });
        }
        if (sortField === "status") {
          // add new more status to status field so we need to sort
          customSort(indicesResponse, "extraStatus", sortDirection);
        }

        // Filtering out indices that belong to a data stream. This must be done before pagination.
        const filteredIndices = showDataStreams ? indicesResponse : indicesResponse.filter(index => index.data_stream === null);

        // _cat doesn't support pagination, do our own in server pagination to at least reduce network bandwidth
        const fromNumber = parseInt(from, 10);
        const sizeNumber = parseInt(size, 10);
        const paginatedIndices = filteredIndices.slice(fromNumber, fromNumber + sizeNumber);
        const indexNames = paginatedIndices.map(value => value.index);
        const managedStatus = await this._getManagedStatus(callWithRequest, indexNames);
        const allIndices = paginatedIndices.map(catIndex => ({
          ...catIndex,
          managed: managedStatus[catIndex.index] ? "Yes" : "No",
          managedPolicy: managedStatus[catIndex.index]
        }));

        // NOTE: Cannot use response.ok due to typescript type checking
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              indices: sortField === "managed" ? customSort(allIndices, "managed", sortDirection) : allIndices,
              totalIndices: filteredIndices.length
            }
          }
        });
      } catch (err) {
        // Throws an error if there is no index matching pattern
        if (err.statusCode === 404 && err.body.error.type === "index_not_found_exception") {
          return response.custom({
            statusCode: 200,
            body: {
              ok: true,
              response: {
                indices: [],
                totalIndices: 0
              }
            }
          });
        }
        console.error("Index Management - IndexService - getIndices:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "_getManagedStatus", async (callWithRequest, indexNames) => {
      try {
        const explainParamas = {
          index: indexNames.toString()
        };
        const explainResponse = await callWithRequest("ism.explain", explainParamas);
        const managed = {};
        for (const indexName in explainResponse) {
          if (indexName === "total_managed_indices") continue;
          const explain = explainResponse[indexName];
          managed[indexName] = explain["index.plugins.index_state_management.policy_id"] === null ? "" : explain["index.plugins.index_state_management.policy_id"];
        }
        return managed;
      } catch (err) {
        // otherwise it could be an unauthorized access error to config index or some other error
        // in which case we will return managed status N/A
        console.error("Index Management - IndexService - _getManagedStatus:", err);
        return indexNames.reduce((accu, value) => ({
          ...accu,
          [value]: "N/A"
        }), {});
      }
    });
    _defineProperty(this, "applyPolicy", async (context, request, response) => {
      try {
        const {
          indices,
          policyId
        } = request.body;
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const params = {
          index: indices.join(","),
          body: {
            policy_id: policyId
          }
        };
        const addResponse = await callWithRequest("ism.add", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              failures: addResponse.failures,
              updatedIndices: addResponse.updated_indices,
              failedIndices: addResponse.failed_indices.map(failedIndex => ({
                indexName: failedIndex.index_name,
                indexUuid: failedIndex.index_uuid,
                reason: failedIndex.reason
              }))
            }
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - applyPolicy:", err);
        // return { ok: false, error: err.message };
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
    _defineProperty(this, "editRolloverAlias", async (context, request, response) => {
      try {
        const {
          alias,
          index
        } = request.body;
        const callWithRequest = this.getClientBasedOnDataSource(context, request);
        const params = {
          index,
          body: {
            [_constants.Setting.RolloverAlias]: alias
          }
        };
        const rollOverResponse = await callWithRequest("indices.putSettings", params);
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: rollOverResponse
          }
        });
      } catch (err) {
        console.error("Index Management - IndexService - editRolloverAlias", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });
  }
}
exports.default = IndexService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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