"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OpenSearchRoutesService = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _configSchema = require("@osd/config-schema");
var _lodash = require("lodash");
var _semver = _interopRequireDefault(require("semver"));
var _common = require("../../common");
var _helpers = require("./helpers");
var _helpers2 = require("../utils/helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
/**
 * Server-side routes to process OpenSearch-related node API calls and execute the
 * corresponding API calls against the OpenSearch cluster.
 */
function registerOpenSearchRoutes(router, opensearchRoutesService) {
  router.get({
    path: `${_common.CAT_INDICES_NODE_API_PATH}/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/catIndices/{pattern}`,
    validate: {
      params: _configSchema.schema.object({
        pattern: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.catIndices);
  router.get({
    path: `${_common.GET_MAPPINGS_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/mappings/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getMappings);
  router.get({
    path: `${_common.GET_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIndex/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIndex);
  router.post({
    path: _common.SEARCH_INDEX_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.SEARCH_INDEX_NODE_API_PATH}/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/search/{index}/{search_pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        search_pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean(),
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, opensearchRoutesService.searchIndex);
  router.put({
    path: `${_common.INGEST_NODE_API_PATH}/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.put({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/ingest/{index}`,
    validate: {
      params: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.ingest);
  router.post({
    path: `${_common.BULK_NODE_API_PATH}/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk/{pipeline}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.BULK_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/bulk`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, opensearchRoutesService.bulk);
  router.post({
    path: _common.SIMULATE_PIPELINE_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.SIMULATE_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      body: _configSchema.schema.object({
        docs: _configSchema.schema.any()
      }),
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/simulatePipeline`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        pipeline: _configSchema.schema.any(),
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/simulatePipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string(),
        pipeline_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        docs: _configSchema.schema.any()
      }),
      query: _configSchema.schema.object({
        verbose: _configSchema.schema.boolean()
      })
    }
  }, opensearchRoutesService.simulatePipeline);
  router.get({
    path: `${_common.INGEST_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getIngestPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getIngestPipeline);
  router.get({
    path: `${_common.SEARCH_PIPELINE_NODE_API_PATH}/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getSearchPipeline/{pipeline_id}`,
    validate: {
      params: _configSchema.schema.object({
        pipeline_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchPipeline);
  router.get({
    path: _common.GET_SEARCH_TEMPLATES_NODE_API_PATH,
    validate: {}
  }, opensearchRoutesService.getSearchTemplates);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/opensearch/getSearchTemplates`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      })
    }
  }, opensearchRoutesService.getSearchTemplates);
}
class OpenSearchRoutesService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "catIndices", async (context, req, res) => {
      const {
        pattern
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('cat.indices', {
          index: pattern,
          format: 'json',
          h: 'health,index'
        });

        // re-formatting the index results to match Index
        const cleanedIndices = response.map(index => ({
          name: index.index,
          health: index.health
        }));
        return res.ok({
          body: cleanedIndices
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getMappings", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        var _response$index;
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.getMapping', {
          index
        });

        // Response will be a dict with key being the index name. Attempt to
        // pull out the mappings. If any errors found (missing index, etc.), an error
        // will be thrown.
        const mappings = (_response$index = response[index]) === null || _response$index === void 0 ? void 0 : _response$index.mappings;
        return res.ok({
          body: mappings
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIndex", async (context, req, res) => {
      const {
        index
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('indices.get', {
          index
        });
        // re-formatting the results to match IndexResponse
        const cleanedIndexDetails = Object.entries(response).map(([indexName, indexDetails]) => ({
          indexName,
          indexDetails
        }));
        return res.ok({
          body: cleanedIndexDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "searchIndex", async (context, req, res) => {
      const {
        index,
        search_pipeline
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      const {
        verbose = false,
        data_source_version = undefined
      } = req.query;
      const isPreV219 = data_source_version !== undefined ? _semver.default.lt(data_source_version, _common.MINIMUM_FULL_SUPPORTED_VERSION) : false;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        let response;
        // If verbose is false/undefined, or the version isn't eligible, omit the verbose param when searching.
        if (!verbose || isPreV219) {
          response = await callWithRequest('search', {
            index: index || '',
            body,
            search_pipeline
          });
        } else {
          response = await callWithRequest('search', {
            index,
            body,
            search_pipeline,
            verbose_pipeline: verbose
          });
        }
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "ingest", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        index
      } = req.params;
      const doc = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('index', {
          index,
          body: doc
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "bulk", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      const {
        pipeline
      } = req.params;
      const body = req.body;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('bulk', {
          body,
          pipeline
        });
        return res.ok({
          body: response
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "simulatePipeline", async (context, req, res) => {
      const {
        data_source_id = '',
        pipeline_id = ''
      } = req.params;
      const {
        pipeline,
        docs
      } = req.body;
      const {
        verbose = false
      } = req.query;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        let response = undefined;
        if (!(0, _lodash.isEmpty)(pipeline_id)) {
          response = await callWithRequest('ingest.simulate', {
            body: {
              docs
            },
            id: pipeline_id,
            verbose
          });
        } else {
          response = await callWithRequest('ingest.simulate', {
            body: {
              docs,
              pipeline
            },
            verbose
          });
        }
        return res.ok({
          body: {
            docs: response.docs
          }
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getIngestPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('ingest.getPipeline', {
          id: pipeline_id
        });
        // re-formatting the results to match IngestPipelineResponse
        const cleanedIngestPipelineDetails = Object.entries(response).map(([pipelineId, ingestPipelineDetails]) => ({
          pipelineId,
          ingestPipelineDetails
        }));
        return res.ok({
          body: cleanedIngestPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getSearchPipeline", async (context, req, res) => {
      const {
        pipeline_id
      } = req.params;
      const {
        data_source_id = ''
      } = req.params;
      try {
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('coreClient.getSearchPipeline', {
          pipeline_id: pipeline_id
        });

        // re-formatting the results to match SearchPipelineResponse
        const cleanedSearchPipelineDetails = Object.entries(response).map(([pipelineId, searchPipelineDetails]) => ({
          pipelineId,
          searchPipelineDetails
        }));
        return res.ok({
          body: cleanedSearchPipelineDetails
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getSearchTemplates", async (context, req, res) => {
      const {
        data_source_id = ''
      } = req.params;
      try {
        var _response$metadata;
        const callWithRequest = (0, _helpers2.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('cluster.state', {
          metric: 'metadata',
          filter_path: '**.stored_scripts'
        });
        const searchTemplates = (response === null || response === void 0 || (_response$metadata = response.metadata) === null || _response$metadata === void 0 ? void 0 : _response$metadata.stored_scripts) || {};
        return res.ok({
          body: searchTemplates
        });
      } catch (err) {
        return (0, _helpers.generateCustomError)(res, err);
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.OpenSearchRoutesService = OpenSearchRoutesService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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