"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.generateCustomError = generateCustomError;
exports.getAgentsFromResponses = getAgentsFromResponses;
exports.getConnectorsFromResponses = getConnectorsFromResponses;
exports.getModelsFromResponses = getModelsFromResponses;
exports.getResourcesCreatedFromResponse = getResourcesCreatedFromResponse;
exports.getWorkflowStateFromResponse = getWorkflowStateFromResponse;
exports.getWorkflowsFromResponses = getWorkflowsFromResponses;
exports.isIgnorableError = isIgnorableError;
exports.toAgentObj = toAgentObj;
exports.toWorkflowObj = toWorkflowObj;
var _common = require("../../common");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

// OSD does not provide an interface for this response, but this is following the suggested
// implementations. To prevent typescript complaining, leaving as loosely-typed 'any'
function generateCustomError(res, err) {
  var _err$body;
  if (isDatasourceError(err)) {
    return res.customError({
      statusCode: 404,
      body: {
        message: 'Data source not found'
      }
    });
  }
  return res.customError({
    statusCode: err.statusCode || 500,
    body: {
      message: err.message,
      attributes: {
        error: ((_err$body = err.body) === null || _err$body === void 0 ? void 0 : _err$body.error) || err.message
      }
    }
  });
}
function isDatasourceError(err) {
  if (err.message !== undefined && typeof err.message === 'string') {
    return err.message.includes(_common.INVALID_DATASOURCE_MSG);
  }
}

// Helper fn to filter out backend errors that we don't want to propagate on the frontend.
function isIgnorableError(error) {
  var _error$body, _error$body2, _error$body3;
  return ((_error$body = error.body) === null || _error$body === void 0 || (_error$body = _error$body.error) === null || _error$body === void 0 ? void 0 : _error$body.type) === _common.INDEX_NOT_FOUND_EXCEPTION || ((_error$body2 = error.body) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.error) === null || _error$body2 === void 0 || (_error$body2 = _error$body2.caused_by) === null || _error$body2 === void 0 ? void 0 : _error$body2.type) === _common.INDEX_NOT_FOUND_EXCEPTION || ((_error$body3 = error.body) === null || _error$body3 === void 0 ? void 0 : _error$body3.error) === _common.NO_MODIFICATIONS_FOUND_TEXT;
}

// Convert backend workflow into frontend workflow obj
function toWorkflowObj(hitSource, id) {
  return {
    id,
    name: hitSource.name,
    use_case: hitSource.use_case,
    description: hitSource.description || '',
    version: hitSource.version,
    workflows: hitSource.workflows,
    ui_metadata: hitSource.ui_metadata,
    lastUpdated: hitSource.last_updated_time,
    lastLaunched: hitSource.last_provisioned_time
  };
}

// TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
// https://github.com/opensearch-project/flow-framework/issues/171
// Current implementation combines 2 search responses to create a single set of workflows with
// static information + state information
function getWorkflowsFromResponses(workflowHits, workflowStateHits) {
  const workflowDict = {};
  workflowHits.forEach(workflowHit => {
    var _workflowStateHit$_so, _workflowStateHit$_so2, _workflowStateHit$_so3;
    const hitSource = workflowHit._source;
    workflowDict[workflowHit._id] = toWorkflowObj(hitSource, workflowHit._id);
    const workflowStateHit = workflowStateHits.find(workflowStateHit => workflowStateHit._id === workflowHit._id);
    const workflowState = getWorkflowStateFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so = workflowStateHit._source) === null || _workflowStateHit$_so === void 0 ? void 0 : _workflowStateHit$_so.state);
    const workflowError = workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so2 = workflowStateHit._source) === null || _workflowStateHit$_so2 === void 0 ? void 0 : _workflowStateHit$_so2.error;
    const workflowResourcesCreated = getResourcesCreatedFromResponse(workflowStateHit === null || workflowStateHit === void 0 || (_workflowStateHit$_so3 = workflowStateHit._source) === null || _workflowStateHit$_so3 === void 0 ? void 0 : _workflowStateHit$_so3.resources_created);
    workflowDict[workflowHit._id] = {
      ...workflowDict[workflowHit._id],
      // @ts-ignore
      state: workflowState,
      error: workflowError,
      resourcesCreated: workflowResourcesCreated
    };
  });
  return workflowDict;
}
function getModelsFromResponses(modelHits) {
  const modelDict = {};
  modelHits.forEach(modelHit => {
    // search model API returns hits for each deployed model chunk. ignore these hits
    if (modelHit._source.chunk_number === undefined) {
      var _modelHit$_source, _modelHit$_source2, _modelHit$_source3, _modelHit$_source4, _modelHit$_source5, _modelHit$_source6, _modelHit$_source7;
      const modelId = modelHit._id;

      // the persisted model interface (if available) is a mix of an obj and string.
      // We parse the string values for input/output to have a complete
      // end-to-end JSONSchema obj
      let indexedModelInterface = modelHit._source.interface;
      let modelInterface = undefined;
      if (indexedModelInterface !== undefined) {
        let parsedInput = undefined;
        let parsedOutput = undefined;
        try {
          parsedInput = JSON.parse(indexedModelInterface.input);
        } catch {}
        try {
          parsedOutput = JSON.parse(indexedModelInterface.output);
        } catch {}
        modelInterface = {
          input: parsedInput,
          output: parsedOutput
        };
      }

      // in case of schema changes from ML plugin, this may crash. That is ok, as the error
      // produced will help expose the root cause
      modelDict[modelId] = {
        id: modelId,
        name: (_modelHit$_source = modelHit._source) === null || _modelHit$_source === void 0 ? void 0 : _modelHit$_source.name,
        // @ts-ignore
        algorithm: _common.MODEL_ALGORITHM[(_modelHit$_source2 = modelHit._source) === null || _modelHit$_source2 === void 0 ? void 0 : _modelHit$_source2.algorithm],
        // @ts-ignore
        state: _common.MODEL_STATE[(_modelHit$_source3 = modelHit._source) === null || _modelHit$_source3 === void 0 ? void 0 : _modelHit$_source3.model_state],
        modelConfig: {
          modelType: (_modelHit$_source4 = modelHit._source) === null || _modelHit$_source4 === void 0 || (_modelHit$_source4 = _modelHit$_source4.model_config) === null || _modelHit$_source4 === void 0 ? void 0 : _modelHit$_source4.model_type,
          embeddingDimension: (_modelHit$_source5 = modelHit._source) === null || _modelHit$_source5 === void 0 || (_modelHit$_source5 = _modelHit$_source5.model_config) === null || _modelHit$_source5 === void 0 ? void 0 : _modelHit$_source5.embedding_dimension
        },
        interface: modelInterface,
        connectorId: (_modelHit$_source6 = modelHit._source) === null || _modelHit$_source6 === void 0 ? void 0 : _modelHit$_source6.connector_id,
        connector: (_modelHit$_source7 = modelHit._source) === null || _modelHit$_source7 === void 0 ? void 0 : _modelHit$_source7.connector
      };
    }
  });
  return modelDict;
}
function getConnectorsFromResponses(modelHits) {
  const connectorDict = {};
  modelHits.forEach(connectorHit => {
    var _connectorHit$_source, _connectorHit$_source2, _connectorHit$_source3, _connectorHit$_source4, _connectorHit$_source5, _connectorHit$_source6, _connectorHit$_source7;
    const connectorId = connectorHit._id;

    // in case of schema changes from ML plugin, this may crash. That is ok, as the error
    // produced will help expose the root cause
    connectorDict[connectorId] = {
      id: connectorId,
      name: (_connectorHit$_source = connectorHit._source) === null || _connectorHit$_source === void 0 ? void 0 : _connectorHit$_source.name,
      protocol: (_connectorHit$_source2 = connectorHit._source) === null || _connectorHit$_source2 === void 0 ? void 0 : _connectorHit$_source2.protocol,
      parameters: {
        model: (_connectorHit$_source3 = connectorHit._source) === null || _connectorHit$_source3 === void 0 || (_connectorHit$_source3 = _connectorHit$_source3.parameters) === null || _connectorHit$_source3 === void 0 ? void 0 : _connectorHit$_source3.model,
        dimensions: (_connectorHit$_source4 = connectorHit._source) === null || _connectorHit$_source4 === void 0 || (_connectorHit$_source4 = _connectorHit$_source4.parameters) === null || _connectorHit$_source4 === void 0 ? void 0 : _connectorHit$_source4.dimensions,
        service_name: (_connectorHit$_source5 = connectorHit._source) === null || _connectorHit$_source5 === void 0 || (_connectorHit$_source5 = _connectorHit$_source5.parameters) === null || _connectorHit$_source5 === void 0 ? void 0 : _connectorHit$_source5.service_name
      },
      actions: ((_connectorHit$_source6 = connectorHit._source) === null || _connectorHit$_source6 === void 0 ? void 0 : _connectorHit$_source6.actions) || [],
      client_config: ((_connectorHit$_source7 = connectorHit._source) === null || _connectorHit$_source7 === void 0 ? void 0 : _connectorHit$_source7.client_config) || {}
    };
  });
  return connectorDict;
}

// Convert the workflow state into a readable/presentable state on frontend
function getWorkflowStateFromResponse(state) {
  const finalState = state || _common.DEFAULT_NEW_WORKFLOW_STATE_TYPE;
  // @ts-ignore
  return _common.WORKFLOW_STATE[finalState];
}

// Convert the workflow resources into a readable/presentable state on frontend
function getResourcesCreatedFromResponse(resourcesCreated) {
  const finalResources = [];
  if (resourcesCreated) {
    resourcesCreated.forEach(backendResource => {
      finalResources.push({
        id: backendResource.resource_id,
        stepType: backendResource.workflow_step_name,
        type:
        // @ts-ignore
        _common.WORKFLOW_RESOURCE_TYPE[
        // the backend persists the types in lowercase. e.g., "pipeline_id"
        backendResource.resource_type.toUpperCase()]
      });
    });
  }
  return finalResources;
}

// Convert backend agent into a frontend agent obj
function toAgentObj(hitSource, id) {
  // MCP connectors are stringified when indexed. Convert back to an obj
  // to be consistently stored on client-side.
  let mcpConnectors;
  try {
    var _hitSource$parameters;
    mcpConnectors = JSON.parse(hitSource === null || hitSource === void 0 || (_hitSource$parameters = hitSource.parameters) === null || _hitSource$parameters === void 0 ? void 0 : _hitSource$parameters.mcp_connectors);
  } catch {}
  return {
    id,
    name: hitSource === null || hitSource === void 0 ? void 0 : hitSource.name,
    type: hitSource === null || hitSource === void 0 ? void 0 : hitSource.type,
    description: hitSource === null || hitSource === void 0 ? void 0 : hitSource.description,
    tools: hitSource === null || hitSource === void 0 ? void 0 : hitSource.tools,
    llm: hitSource === null || hitSource === void 0 ? void 0 : hitSource.llm,
    memory: hitSource === null || hitSource === void 0 ? void 0 : hitSource.memory,
    parameters: {
      ...(hitSource === null || hitSource === void 0 ? void 0 : hitSource.parameters),
      mcp_connectors: mcpConnectors
    }
  };
}

// Convert the agent hits into a readable/presentable state on frontend
function getAgentsFromResponses(agentHits) {
  const agents = {};
  for (const agentHit of agentHits) {
    const source = agentHit._source;
    const id = agentHit._id;
    // @ts-ignore
    agents[id] = toAgentObj(source, id);
  }
  return agents;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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