"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FlowFrameworkRoutesService = void 0;
exports.registerFlowFrameworkRoutes = registerFlowFrameworkRoutes;
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var _semver = _interopRequireDefault(require("semver"));
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helpers = require("../utils/helpers");
var _helpers2 = require("./helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
/**
 * Server-side routes to process flow-framework-related node API calls and execute the
 * corresponding API calls against the OpenSearch cluster.
 */
function registerFlowFrameworkRoutes(router, flowFrameworkRoutesService) {
  router.get({
    path: `${_common.GET_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflow);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflow);
  router.post({
    path: _common.SEARCH_WORKFLOWS_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.searchWorkflows);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/search`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.searchWorkflows);
  router.get({
    path: `${_common.GET_WORKFLOW_STATE_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflowState);
  router.get({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/state/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.getWorkflowState);
  router.post({
    path: _common.CREATE_WORKFLOW_NODE_API_PATH,
    validate: {
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.createWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/create`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string()
      }),
      body: _configSchema.schema.any()
    }
  }, flowFrameworkRoutesService.createWorkflow);
  router.put({
    path: `${_common.UPDATE_WORKFLOW_NODE_API_PATH}/{workflow_id}/{update_fields}/{reprovision}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        update_fields: _configSchema.schema.boolean(),
        reprovision: _configSchema.schema.boolean()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.updateWorkflow);
  router.put({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/update/{workflow_id}/{update_fields}/{reprovision}`,
    validate: {
      params: _configSchema.schema.object({
        data_source_id: _configSchema.schema.string(),
        workflow_id: _configSchema.schema.string(),
        update_fields: _configSchema.schema.boolean(),
        reprovision: _configSchema.schema.boolean()
      }),
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.updateWorkflow);
  router.post({
    path: `${_common.PROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.provisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/provision/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        data_source_version: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, flowFrameworkRoutesService.provisionWorkflow);
  router.post({
    path: `${_common.DEPROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/deprovision/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.DEPROVISION_WORKFLOW_NODE_API_PATH}/{workflow_id}/{resource_ids}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        resource_ids: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.post({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/deprovision/{workflow_id}/{resource_ids}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        resource_ids: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deprovisionWorkflow);
  router.delete({
    path: `${_common.DELETE_WORKFLOW_NODE_API_PATH}/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deleteWorkflow);
  router.delete({
    path: `${_common.BASE_NODE_API_PATH}/{data_source_id}/workflow/delete/{workflow_id}`,
    validate: {
      params: _configSchema.schema.object({
        workflow_id: _configSchema.schema.string(),
        data_source_id: _configSchema.schema.string()
      })
    }
  }, flowFrameworkRoutesService.deleteWorkflow);
  router.get({
    path: _common.GET_PRESET_WORKFLOWS_NODE_API_PATH,
    validate: {}
  }, flowFrameworkRoutesService.getPresetWorkflows);
}
class FlowFrameworkRoutesService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "client", void 0);
    // TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
    // https://github.com/opensearch-project/flow-framework/issues/171
    // Current implementation is making two calls and combining results via helper fn
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.getWorkflow', {
          workflow_id
        });
        const workflow = (0, _helpers2.toWorkflowObj)(response, workflow_id);
        const stateResponse = await callWithRequest('flowFramework.getWorkflowState', {
          workflow_id
        });
        const state = (0, _helpers2.getWorkflowStateFromResponse)(stateResponse.state);
        const resourcesCreated = (0, _helpers2.getResourcesCreatedFromResponse)(stateResponse.resources_created);
        const workflowWithState = {
          ...workflow,
          state,
          error: stateResponse.error,
          resourcesCreated
        };
        return res.ok({
          body: {
            workflow: workflowWithState
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    // TODO: can remove or simplify if we can fetch all data from a single API call. Tracking issue:
    // https://github.com/opensearch-project/flow-framework/issues/171
    // Current implementation is making two calls and combining results via helper fn
    _defineProperty(this, "searchWorkflows", async (context, req, res) => {
      const body = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const workflowsResponse = await callWithRequest('flowFramework.searchWorkflows', {
          body
        });
        const workflowHits = workflowsResponse.hits.hits;
        const workflowStatesResponse = await callWithRequest('flowFramework.searchWorkflowState', {
          body
        });
        const workflowStateHits = workflowStatesResponse.hits.hits;
        const workflowDict = (0, _helpers2.getWorkflowsFromResponses)(workflowHits, workflowStateHits);
        return res.ok({
          body: {
            workflows: workflowDict
          }
        });
      } catch (err) {
        if ((0, _helpers2.isIgnorableError)(err)) {
          return res.ok({
            body: {
              workflows: {}
            }
          });
        }
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getWorkflowState", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.getWorkflowState', {
          workflow_id
        });
        const state = (0, _helpers2.getWorkflowStateFromResponse)(response.state);
        const resourcesCreated = (0, _helpers2.getResourcesCreatedFromResponse)(response.resources_created);
        return res.ok({
          body: {
            workflowId: workflow_id,
            workflowState: state,
            resourcesCreated
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      const body = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.createWorkflow', {
          body
        });
        const workflowWithId = {
          ...body,
          id: response.workflow_id
        };
        return res.ok({
          body: {
            workflow: workflowWithId
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "updateWorkflow", async (context, req, res) => {
      const {
        workflow_id,
        update_fields,
        reprovision
      } = req.params;
      const workflowTemplate = req.body;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const {
          data_source_version
        } = req.query;
        const isPreV219 = data_source_version !== undefined ? _semver.default.lt(data_source_version, _common.MINIMUM_FULL_SUPPORTED_VERSION) : false;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        if (reprovision) {
          if (isPreV219) {
            await callWithRequest('flowFramework.updateAndReprovisionWorkflowAsync', {
              workflow_id,
              update_fields,
              body: workflowTemplate
            });
          } else {
            await callWithRequest('flowFramework.updateAndReprovisionWorkflow', {
              workflow_id,
              update_fields,
              body: workflowTemplate
            });
          }
        } else {
          await callWithRequest('flowFramework.updateWorkflow', {
            workflow_id,
            update_fields,
            body: workflowTemplate
          });
        }
        return res.ok({
          body: {
            workflowId: workflow_id,
            workflowTemplate
          }
        });
      } catch (err) {
        if ((0, _helpers2.isIgnorableError)(err)) {
          return res.ok({
            body: {
              workflowId: workflow_id,
              workflowTemplate
            }
          });
        }
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "provisionWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const {
          data_source_version
        } = req.query;
        const isPreV219 = data_source_version !== undefined ? _semver.default.lt(data_source_version, _common.MINIMUM_FULL_SUPPORTED_VERSION) : false;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        if (isPreV219) {
          await callWithRequest('flowFramework.provisionWorkflowAsync', {
            workflow_id
          });
        } else {
          await callWithRequest('flowFramework.provisionWorkflow', {
            workflow_id
          });
        }
        return res.ok();
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "deprovisionWorkflow", async (context, req, res) => {
      const {
        workflow_id,
        resource_ids
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        if (resource_ids !== undefined) {
          await callWithRequest('flowFramework.forceDeprovisionWorkflow', {
            workflow_id,
            resource_ids
          });
        } else {
          await callWithRequest('flowFramework.deprovisionWorkflow', {
            workflow_id
          });
        }
        return res.ok();
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      const {
        workflow_id
      } = req.params;
      try {
        const {
          data_source_id = ''
        } = req.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, req, data_source_id, this.client);
        const response = await callWithRequest('flowFramework.deleteWorkflow', {
          workflow_id
        });
        return res.ok({
          body: {
            id: response._id
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    _defineProperty(this, "getPresetWorkflows", async (context, req, res) => {
      try {
        // In the future we may get these from backend via some API. For now we can
        // persist a set of working presets on server-side.
        const jsonTemplateDir = _path.default.resolve(__dirname, '../resources/templates');
        const jsonTemplates = _fs.default.readdirSync(jsonTemplateDir).filter(file => _path.default.extname(file) === '.json');
        const workflowTemplates = [];
        jsonTemplates.forEach(jsonTemplate => {
          const templateData = _fs.default.readFileSync(_path.default.join(jsonTemplateDir, jsonTemplate));
          const workflowTemplate = JSON.parse(templateData.toString());
          workflowTemplates.push(workflowTemplate);
        });
        return res.ok({
          body: {
            workflowTemplates
          }
        });
      } catch (err) {
        return (0, _helpers2.generateCustomError)(res, err);
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.FlowFrameworkRoutesService = FlowFrameworkRoutesService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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