"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STATE = exports.WORKFLOW_RESOURCE_TYPE = exports.MODEL_STATE = exports.MODEL_ALGORITHM = void 0;
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
/**
 ********** WORKFLOW TYPES/INTERFACES **********
 */
/**
 ********** MODAL SUB-FORM TYPES/INTERFACES **********
 We persist sub-forms in the form modals s.t. data is only
 saved back to the parent form if the user explicitly saves.

 We define the structure of the forms here.
 */
// Ingest docs modal
// Search request modal
// Configure template modal
// Configure expression modal
// Configure multi-expression modal
/**
 ********** WORKSPACE TYPES/INTERFACES **********
 */
/**
 * An input field for a component. Specifies enough configuration for the
 * UI node to render it properly (help text, links, etc.)
 */
/**
 * The base interface the components will implement.
 */
/**
 * We need to include some extra instance-specific data to the ReactFlow component
 * to perform extra functionality, such as deleting the node from the ReactFlowInstance.
 */
/**
 ********** USE CASE TEMPLATE TYPES/INTERFACES **********
 */
// A stateless template of a workflow
// An instance of a workflow based on a workflow template
/**
 ********** ML PLUGIN TYPES/INTERFACES **********
 */
// Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/model/MLModelState.java
let MODEL_STATE = exports.MODEL_STATE = /*#__PURE__*/function (MODEL_STATE) {
  MODEL_STATE["REGISTERED"] = "Registered";
  MODEL_STATE["REGISTERING"] = "Registering";
  MODEL_STATE["DEPLOYING"] = "Deploying";
  MODEL_STATE["DEPLOYED"] = "Deployed";
  MODEL_STATE["PARTIALLY_DEPLOYED"] = "Partially deployed";
  MODEL_STATE["UNDEPLOYED"] = "Undeployed";
  MODEL_STATE["DEPLOY_FAILED"] = "Deploy failed";
  return MODEL_STATE;
}({}); // Based off of https://github.com/opensearch-project/ml-commons/blob/main/common/src/main/java/org/opensearch/ml/common/FunctionName.java
let MODEL_ALGORITHM = exports.MODEL_ALGORITHM = /*#__PURE__*/function (MODEL_ALGORITHM) {
  MODEL_ALGORITHM["LINEAR_REGRESSION"] = "Linear regression";
  MODEL_ALGORITHM["KMEANS"] = "K-means";
  MODEL_ALGORITHM["AD_LIBSVM"] = "AD LIBSVM";
  MODEL_ALGORITHM["SAMPLE_ALGO"] = "Sample algorithm";
  MODEL_ALGORITHM["LOCAL_SAMPLE_CALCULATOR"] = "Local sample calculator";
  MODEL_ALGORITHM["FIT_RCF"] = "Fit RCF";
  MODEL_ALGORITHM["BATCH_RCF"] = "Batch RCF";
  MODEL_ALGORITHM["ANOMALY_LOCALIZATION"] = "Anomaly localization";
  MODEL_ALGORITHM["RCF_SUMMARIZE"] = "RCF summarize";
  MODEL_ALGORITHM["LOGISTIC_REGRESSION"] = "Logistic regression";
  MODEL_ALGORITHM["TEXT_EMBEDDING"] = "Text embedding";
  MODEL_ALGORITHM["METRICS_CORRELATION"] = "Metrics correlation";
  MODEL_ALGORITHM["REMOTE"] = "Remote";
  MODEL_ALGORITHM["SPARSE_ENCODING"] = "Sparse encoding";
  MODEL_ALGORITHM["SPARSE_TOKENIZE"] = "Sparse tokenize";
  MODEL_ALGORITHM["TEXT_SIMILARITY"] = "Text similarity";
  MODEL_ALGORITHM["QUESTION_ANSWERING"] = "Question answering";
  MODEL_ALGORITHM["AGENT"] = "Agent";
  return MODEL_ALGORITHM;
}({}); // Based off of JSONSchema. For more info, see https://json-schema.org/understanding-json-schema/reference/type
// For rendering options, we extract the name (the key in the input/output obj) and combine into a single obj.
// Also persist an optional field to dynamically run / not run validation on the form later on
/**
 ********** MISC TYPES/INTERFACES ************
 */
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/model/State.java
let WORKFLOW_STATE = exports.WORKFLOW_STATE = /*#__PURE__*/function (WORKFLOW_STATE) {
  WORKFLOW_STATE["NOT_STARTED"] = "Not started";
  WORKFLOW_STATE["PROVISIONING"] = "Provisioning";
  WORKFLOW_STATE["FAILED"] = "Failed";
  WORKFLOW_STATE["COMPLETED"] = "Completed";
  return WORKFLOW_STATE;
}({});
// Based off of https://github.com/opensearch-project/flow-framework/blob/main/src/main/java/org/opensearch/flowframework/common/WorkflowResources.java
let WORKFLOW_RESOURCE_TYPE = exports.WORKFLOW_RESOURCE_TYPE = /*#__PURE__*/function (WORKFLOW_RESOURCE_TYPE) {
  WORKFLOW_RESOURCE_TYPE["PIPELINE_ID"] = "Ingest pipeline";
  WORKFLOW_RESOURCE_TYPE["INDEX_NAME"] = "Index";
  WORKFLOW_RESOURCE_TYPE["MODEL_ID"] = "Model";
  WORKFLOW_RESOURCE_TYPE["MODEL_GROUP_ID"] = "Model group";
  WORKFLOW_RESOURCE_TYPE["CONNECTOR_ID"] = "Connector";
  return WORKFLOW_RESOURCE_TYPE;
}({});
let WORKFLOW_STEP_TYPE = exports.WORKFLOW_STEP_TYPE = /*#__PURE__*/function (WORKFLOW_STEP_TYPE) {
  WORKFLOW_STEP_TYPE["CREATE_INGEST_PIPELINE_STEP_TYPE"] = "create_ingest_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_SEARCH_PIPELINE_STEP_TYPE"] = "create_search_pipeline";
  WORKFLOW_STEP_TYPE["CREATE_INDEX_STEP_TYPE"] = "create_index";
  return WORKFLOW_STEP_TYPE;
}({}); // We cannot disambiguate ingest vs. search pipelines based on workflow resource type. To work around
// that, we maintain this map from workflow step type -> formatted type
let WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = exports.WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP = /*#__PURE__*/function (WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP) {
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_ingest_pipeline"] = "Ingest pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_search_pipeline"] = "Search pipeline";
  WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP["create_index"] = "Index";
  return WORKFLOW_STEP_TO_RESOURCE_TYPE_MAP;
}({});
/**
 ********** OPENSEARCH TYPES/INTERFACES ************
 */
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#example-specify-a-pipeline-in-the-path
// verbose mode
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#query-parameters
// verbose mode
// from https://opensearch.org/docs/latest/ingest-pipelines/simulate-ingest/#query-parameters
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJNT0RFTF9TVEFURSIsImV4cG9ydHMiLCJNT0RFTF9BTEdPUklUSE0iLCJXT1JLRkxPV19TVEFURSIsIldPUktGTE9XX1JFU09VUkNFX1RZUEUiLCJXT1JLRkxPV19TVEVQX1RZUEUiLCJXT1JLRkxPV19TVEVQX1RPX1JFU09VUkNFX1RZUEVfTUFQIl0sInNvdXJjZXMiOlsiaW50ZXJmYWNlcy50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxyXG4gKiBDb3B5cmlnaHQgT3BlblNlYXJjaCBDb250cmlidXRvcnNcclxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcclxuICovXHJcblxyXG5pbXBvcnQgeyBOb2RlLCBFZGdlIH0gZnJvbSAncmVhY3RmbG93JztcclxuaW1wb3J0IHsgRm9ybWlrVmFsdWVzIH0gZnJvbSAnZm9ybWlrJztcclxuaW1wb3J0IHsgT2JqZWN0U2NoZW1hIH0gZnJvbSAneXVwJztcclxuaW1wb3J0IHtcclxuICBBR0VOVF9MTE1fSU5URVJGQUNFX1RZUEUsXHJcbiAgQUdFTlRfTUVNT1JZX1RZUEUsXHJcbiAgQUdFTlRfVFlQRSxcclxuICBDT01QT05FTlRfQ0xBU1MsXHJcbiAgQ09OTkVDVE9SX1BST1RPQ09MLFxyXG4gIFBST0NFU1NPUl9UWVBFLFxyXG4gIFRPT0xfVFlQRSxcclxuICBUUkFOU0ZPUk1fVFlQRSxcclxuICBXT1JLRkxPV19UWVBFLFxyXG59IGZyb20gJy4vY29uc3RhbnRzJztcclxuXHJcbmV4cG9ydCB0eXBlIEluZGV4ID0ge1xyXG4gIG5hbWU6IHN0cmluZztcclxuICBoZWFsdGg6ICdncmVlbicgfCAneWVsbG93JyB8ICdyZWQnO1xyXG59O1xyXG5cclxuLyoqXHJcbiAqKioqKioqKioqIFdPUktGTE9XIFRZUEVTL0lOVEVSRkFDRVMgKioqKioqKioqKlxyXG4gKi9cclxuZXhwb3J0IHR5cGUgTURTUXVlcnlQYXJhbXMgPSB7XHJcbiAgZGF0YVNvdXJjZUlkPzogc3RyaW5nO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQ29uZmlnRmllbGRUeXBlID1cclxuICB8ICdzdHJpbmcnXHJcbiAgfCAndGV4dEFyZWEnXHJcbiAgfCAnanNvbidcclxuICB8ICdqc29uQXJyYXknXHJcbiAgfCAnanNvblN0cmluZydcclxuICB8ICdqc29uTGluZXMnXHJcbiAgfCAnc2VsZWN0J1xyXG4gIHwgJ21vZGVsJ1xyXG4gIHwgJ21hcCdcclxuICB8ICdtYXBBcnJheSdcclxuICB8ICdib29sZWFuJ1xyXG4gIHwgJ251bWJlcidcclxuICB8ICdpbnB1dE1hcEFycmF5J1xyXG4gIHwgJ291dHB1dE1hcEFycmF5JztcclxuXHJcbmV4cG9ydCB0eXBlIENvbmZpZ0ZpZWxkVmFsdWUgPSBzdHJpbmcgfCB7fTtcclxuXHJcbmV4cG9ydCBpbnRlcmZhY2UgSUNvbmZpZ0ZpZWxkIHtcclxuICB0eXBlOiBDb25maWdGaWVsZFR5cGU7XHJcbiAgaWQ6IHN0cmluZztcclxuICB2YWx1ZT86IENvbmZpZ0ZpZWxkVmFsdWU7XHJcbiAgc2VsZWN0T3B0aW9ucz86IENvbmZpZ0ZpZWxkVmFsdWVbXTtcclxufVxyXG5cclxuZXhwb3J0IGludGVyZmFjZSBJQ29uZmlnIHtcclxuICBpZDogc3RyaW5nO1xyXG4gIG5hbWU6IHN0cmluZztcclxuICBmaWVsZHM6IElDb25maWdGaWVsZFtdO1xyXG4gIG9wdGlvbmFsRmllbGRzPzogSUNvbmZpZ0ZpZWxkW107XHJcbn1cclxuXHJcbmV4cG9ydCBpbnRlcmZhY2UgSVByb2Nlc3NvckNvbmZpZyBleHRlbmRzIElDb25maWcge1xyXG4gIHR5cGU6IFBST0NFU1NPUl9UWVBFO1xyXG59XHJcblxyXG5leHBvcnQgdHlwZSBQcm9jZXNzb3JzQ29uZmlnID0ge1xyXG4gIHByb2Nlc3NvcnM6IElQcm9jZXNzb3JDb25maWdbXTtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIEluZGV4Q29uZmlnID0ge1xyXG4gIG5hbWU6IElDb25maWdGaWVsZDtcclxuICBtYXBwaW5nczogSUNvbmZpZ0ZpZWxkO1xyXG4gIHNldHRpbmdzOiBJQ29uZmlnRmllbGQ7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hJbmRleENvbmZpZyA9IHtcclxuICBuYW1lOiBJQ29uZmlnRmllbGQ7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBJbmdlc3RDb25maWcgPSB7XHJcbiAgZW5hYmxlZDogSUNvbmZpZ0ZpZWxkO1xyXG4gIHBpcGVsaW5lTmFtZTogSUNvbmZpZ0ZpZWxkO1xyXG4gIGVucmljaDogUHJvY2Vzc29yc0NvbmZpZztcclxuICBpbmRleDogSW5kZXhDb25maWc7XHJcbiAgYWdlbnRJZD86IElDb25maWdGaWVsZDtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFNlYXJjaENvbmZpZyA9IHtcclxuICByZXF1ZXN0OiBJQ29uZmlnRmllbGQ7XHJcbiAgaW5kZXg6IFNlYXJjaEluZGV4Q29uZmlnO1xyXG4gIHBpcGVsaW5lTmFtZTogSUNvbmZpZ0ZpZWxkO1xyXG4gIGVucmljaFJlcXVlc3Q6IFByb2Nlc3NvcnNDb25maWc7XHJcbiAgZW5yaWNoUmVzcG9uc2U6IFByb2Nlc3NvcnNDb25maWc7XHJcbiAgcmVxdWVzdEFnZW50SWQ/OiBJQ29uZmlnRmllbGQ7XHJcbiAgcmVzcG9uc2VBZ2VudElkPzogSUNvbmZpZ0ZpZWxkO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgV29ya2Zsb3dDb25maWcgPSB7XHJcbiAgaW5nZXN0OiBJbmdlc3RDb25maWc7XHJcbiAgc2VhcmNoOiBTZWFyY2hDb25maWc7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBNYXBFbnRyeSA9IHtcclxuICBrZXk6IHN0cmluZztcclxuICB2YWx1ZTogc3RyaW5nO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgTWFwRm9ybVZhbHVlID0gTWFwRW50cnlbXTtcclxuXHJcbmV4cG9ydCB0eXBlIE1hcEFycmF5Rm9ybVZhbHVlID0gTWFwRm9ybVZhbHVlW107XHJcblxyXG5leHBvcnQgdHlwZSBFeHByZXNzaW9uVmFyID0ge1xyXG4gIG5hbWU6IHN0cmluZztcclxuICB0cmFuc2Zvcm06IHN0cmluZztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFRyYW5zZm9ybSA9IHtcclxuICB0cmFuc2Zvcm1UeXBlOiBUUkFOU0ZPUk1fVFlQRTtcclxuICB2YWx1ZT86IHN0cmluZztcclxuICBvcHRpb25hbD86IGJvb2xlYW47XHJcbiAgLy8gVGVtcGxhdGVzIG1heSBwZXJzaXN0IHRoZWlyIG93biBzZXQgb2YgbmVzdGVkIHRyYW5zZm9ybXNcclxuICAvLyB0byBiZSBkeW5hbWljYWxseSBpbmplY3RlZCBpbnRvIHRoZSB0ZW1wbGF0ZVxyXG4gIG5lc3RlZFZhcnM/OiBFeHByZXNzaW9uVmFyW107XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBJbnB1dE1hcEVudHJ5ID0ge1xyXG4gIGtleTogc3RyaW5nO1xyXG4gIHZhbHVlOiBUcmFuc2Zvcm07XHJcbn07XHJcbmV4cG9ydCB0eXBlIE91dHB1dE1hcEVudHJ5ID0gSW5wdXRNYXBFbnRyeTtcclxuXHJcbmV4cG9ydCB0eXBlIElucHV0TWFwRm9ybVZhbHVlID0gSW5wdXRNYXBFbnRyeVtdO1xyXG5leHBvcnQgdHlwZSBPdXRwdXRNYXBGb3JtVmFsdWUgPSBPdXRwdXRNYXBFbnRyeVtdO1xyXG5cclxuZXhwb3J0IHR5cGUgTWFwQ2FjaGUgPSB7XHJcbiAgW2lkeDogbnVtYmVyXTogVHJhbnNmb3JtW107XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBJbnB1dE1hcEFycmF5Rm9ybVZhbHVlID0gSW5wdXRNYXBGb3JtVmFsdWVbXTtcclxuZXhwb3J0IHR5cGUgT3V0cHV0TWFwQXJyYXlGb3JtVmFsdWUgPSBPdXRwdXRNYXBGb3JtVmFsdWVbXTtcclxuXHJcbmV4cG9ydCB0eXBlIFdvcmtmbG93Rm9ybVZhbHVlcyA9IHtcclxuICBpbmdlc3Q6IEZvcm1pa1ZhbHVlcztcclxuICBzZWFyY2g6IEZvcm1pa1ZhbHVlcztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFdvcmtmbG93U2NoZW1hT2JqID0ge1xyXG4gIFtrZXk6IHN0cmluZ106IE9iamVjdFNjaGVtYTxhbnksIGFueSwgYW55PjtcclxufTtcclxuZXhwb3J0IHR5cGUgV29ya2Zsb3dTY2hlbWEgPSBPYmplY3RTY2hlbWE8V29ya2Zsb3dTY2hlbWFPYmo+O1xyXG5cclxuLyoqXHJcbiAqKioqKioqKioqIE1PREFMIFNVQi1GT1JNIFRZUEVTL0lOVEVSRkFDRVMgKioqKioqKioqKlxyXG4gV2UgcGVyc2lzdCBzdWItZm9ybXMgaW4gdGhlIGZvcm0gbW9kYWxzIHMudC4gZGF0YSBpcyBvbmx5XHJcbiBzYXZlZCBiYWNrIHRvIHRoZSBwYXJlbnQgZm9ybSBpZiB0aGUgdXNlciBleHBsaWNpdGx5IHNhdmVzLlxyXG5cclxuIFdlIGRlZmluZSB0aGUgc3RydWN0dXJlIG9mIHRoZSBmb3JtcyBoZXJlLlxyXG4gKi9cclxuXHJcbi8vIEluZ2VzdCBkb2NzIG1vZGFsXHJcbmV4cG9ydCB0eXBlIEluZ2VzdERvY3NGb3JtVmFsdWVzID0ge1xyXG4gIGRvY3M6IEZvcm1pa1ZhbHVlcztcclxufTtcclxuXHJcbi8vIFNlYXJjaCByZXF1ZXN0IG1vZGFsXHJcbmV4cG9ydCB0eXBlIFJlcXVlc3RGb3JtVmFsdWVzID0ge1xyXG4gIHJlcXVlc3Q6IENvbmZpZ0ZpZWxkVmFsdWU7XHJcbn07XHJcblxyXG4vLyBDb25maWd1cmUgdGVtcGxhdGUgbW9kYWxcclxuZXhwb3J0IHR5cGUgVGVtcGxhdGVGb3JtVmFsdWVzID0gT21pdDxUcmFuc2Zvcm0sICd0cmFuc2Zvcm1UeXBlJz47XHJcblxyXG4vLyBDb25maWd1cmUgZXhwcmVzc2lvbiBtb2RhbFxyXG5leHBvcnQgdHlwZSBFeHByZXNzaW9uRm9ybVZhbHVlcyA9IHtcclxuICBleHByZXNzaW9uOiBzdHJpbmc7XHJcbn07XHJcblxyXG4vLyBDb25maWd1cmUgbXVsdGktZXhwcmVzc2lvbiBtb2RhbFxyXG5leHBvcnQgdHlwZSBNdWx0aUV4cHJlc3Npb25Gb3JtVmFsdWVzID0ge1xyXG4gIGV4cHJlc3Npb25zOiBFeHByZXNzaW9uVmFyW107XHJcbn07XHJcblxyXG4vKipcclxuICoqKioqKioqKiogV09SS1NQQUNFIFRZUEVTL0lOVEVSRkFDRVMgKioqKioqKioqKlxyXG4gKi9cclxuXHJcbmV4cG9ydCB0eXBlIEZpZWxkVHlwZSA9ICdzdHJpbmcnIHwgJ2pzb24nIHwgJ3NlbGVjdCcgfCAnbW9kZWwnO1xyXG5leHBvcnQgdHlwZSBGaWVsZFZhbHVlID0gc3RyaW5nIHwge307XHJcbmV4cG9ydCB0eXBlIENvbXBvbmVudEZvcm1WYWx1ZXMgPSBGb3JtaWtWYWx1ZXM7XHJcbmV4cG9ydCB0eXBlIFdvcmtzcGFjZUZvcm1WYWx1ZXMgPSB7XHJcbiAgW2NvbXBvbmVudElkOiBzdHJpbmddOiBDb21wb25lbnRGb3JtVmFsdWVzO1xyXG59O1xyXG5leHBvcnQgdHlwZSBXb3Jrc3BhY2VTY2hlbWFPYmogPSB7XHJcbiAgW2NvbXBvbmVudElkOiBzdHJpbmddOiBPYmplY3RTY2hlbWE8YW55LCBhbnksIGFueT47XHJcbn07XHJcbmV4cG9ydCB0eXBlIFdvcmtzcGFjZVNjaGVtYSA9IE9iamVjdFNjaGVtYTxXb3Jrc3BhY2VTY2hlbWFPYmo+O1xyXG5cclxuZXhwb3J0IGludGVyZmFjZSBJQ29tcG9uZW50SW5wdXQge1xyXG4gIGlkOiBzdHJpbmc7XHJcbiAgbGFiZWw6IHN0cmluZztcclxuICBhY2NlcHRNdWx0aXBsZTogYm9vbGVhbjtcclxufVxyXG5cclxuZXhwb3J0IGludGVyZmFjZSBJQ29tcG9uZW50T3V0cHV0IHtcclxuICBpZDogc3RyaW5nO1xyXG4gIGxhYmVsOiBzdHJpbmc7XHJcbn1cclxuXHJcbi8qKlxyXG4gKiBBbiBpbnB1dCBmaWVsZCBmb3IgYSBjb21wb25lbnQuIFNwZWNpZmllcyBlbm91Z2ggY29uZmlndXJhdGlvbiBmb3IgdGhlXHJcbiAqIFVJIG5vZGUgdG8gcmVuZGVyIGl0IHByb3Blcmx5IChoZWxwIHRleHQsIGxpbmtzLCBldGMuKVxyXG4gKi9cclxuZXhwb3J0IGludGVyZmFjZSBJQ29tcG9uZW50RmllbGQge1xyXG4gIGxhYmVsOiBzdHJpbmc7XHJcbiAgdHlwZTogRmllbGRUeXBlO1xyXG4gIGlkOiBzdHJpbmc7XHJcbiAgdmFsdWU/OiBGaWVsZFZhbHVlO1xyXG4gIHBsYWNlaG9sZGVyPzogc3RyaW5nO1xyXG4gIGhlbHBUZXh0Pzogc3RyaW5nO1xyXG4gIGhlbHBMaW5rPzogc3RyaW5nO1xyXG59XHJcblxyXG4vKipcclxuICogVGhlIGJhc2UgaW50ZXJmYWNlIHRoZSBjb21wb25lbnRzIHdpbGwgaW1wbGVtZW50LlxyXG4gKi9cclxuZXhwb3J0IGludGVyZmFjZSBJQ29tcG9uZW50IHtcclxuICB0eXBlOiBDT01QT05FTlRfQ0xBU1M7XHJcbiAgbGFiZWw6IHN0cmluZztcclxuICBpY29uVHlwZT86IHN0cmluZztcclxuICBkZXNjcmlwdGlvbj86IHN0cmluZztcclxuICBpbnB1dHM/OiBJQ29tcG9uZW50SW5wdXRbXTtcclxuICBvdXRwdXRzPzogSUNvbXBvbmVudE91dHB1dFtdO1xyXG59XHJcblxyXG4vKipcclxuICogV2UgbmVlZCB0byBpbmNsdWRlIHNvbWUgZXh0cmEgaW5zdGFuY2Utc3BlY2lmaWMgZGF0YSB0byB0aGUgUmVhY3RGbG93IGNvbXBvbmVudFxyXG4gKiB0byBwZXJmb3JtIGV4dHJhIGZ1bmN0aW9uYWxpdHksIHN1Y2ggYXMgZGVsZXRpbmcgdGhlIG5vZGUgZnJvbSB0aGUgUmVhY3RGbG93SW5zdGFuY2UuXHJcbiAqL1xyXG5leHBvcnQgaW50ZXJmYWNlIElDb21wb25lbnREYXRhIGV4dGVuZHMgSUNvbXBvbmVudCB7XHJcbiAgaWQ6IHN0cmluZztcclxuICBzZWxlY3RlZD86IGJvb2xlYW47XHJcbn1cclxuXHJcbmV4cG9ydCB0eXBlIFJlYWN0Rmxvd0NvbXBvbmVudCA9IE5vZGU8SUNvbXBvbmVudERhdGE+O1xyXG5leHBvcnQgdHlwZSBSZWFjdEZsb3dFZGdlID0gRWRnZTx7fT4gJiB7XHJcbiAga2V5OiBzdHJpbmc7XHJcbiAgc291cmNlQ2xhc3NlczogQ09NUE9ORU5UX0NMQVNTW107XHJcbiAgdGFyZ2V0Q2xhc3NlczogQ09NUE9ORU5UX0NMQVNTW107XHJcbn07XHJcblxyXG50eXBlIFJlYWN0Rmxvd1ZpZXdwb3J0ID0ge1xyXG4gIHg6IG51bWJlcjtcclxuICB5OiBudW1iZXI7XHJcbiAgem9vbTogbnVtYmVyO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgVUlTdGF0ZSA9IHtcclxuICBzY2hlbWFfdmVyc2lvbjogbnVtYmVyO1xyXG4gIGNvbmZpZzogV29ya2Zsb3dDb25maWc7XHJcbiAgdHlwZTogV09SS0ZMT1dfVFlQRTtcclxuICAvLyBXaWxsIGJlIHVzZWQgaW4gZnV0dXJlIHdoZW4gY2hhbmdpbmcgZnJvbSBmb3JtLWJhc2VkIHRvIGZsb3ctYmFzZWQgY29uZmlncyB2aWEgZHJhZy1hbmQtZHJvcFxyXG4gIHdvcmtzcGFjZV9mbG93PzogV29ya3NwYWNlRmxvd1N0YXRlO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgV29ya3NwYWNlRmxvd1N0YXRlID0ge1xyXG4gIG5vZGVzOiBSZWFjdEZsb3dDb21wb25lbnRbXTtcclxuICBlZGdlczogUmVhY3RGbG93RWRnZVtdO1xyXG4gIHZpZXdwb3J0PzogUmVhY3RGbG93Vmlld3BvcnQ7XHJcbn07XHJcblxyXG4vKipcclxuICoqKioqKioqKiogVVNFIENBU0UgVEVNUExBVEUgVFlQRVMvSU5URVJGQUNFUyAqKioqKioqKioqXHJcbiAqL1xyXG5cclxuZXhwb3J0IHR5cGUgSW5nZXN0UHJvY2Vzc29yID0ge307XHJcbmV4cG9ydCB0eXBlIFNlYXJjaFByb2Nlc3NvciA9IHt9O1xyXG5leHBvcnQgdHlwZSBTZWFyY2hSZXF1ZXN0UHJvY2Vzc29yID0gU2VhcmNoUHJvY2Vzc29yICYge307XHJcbmV4cG9ydCB0eXBlIFNlYXJjaFJlc3BvbnNlUHJvY2Vzc29yID0gU2VhcmNoUHJvY2Vzc29yICYge307XHJcbmV4cG9ydCB0eXBlIFNlYXJjaFBoYXNlUmVzdWx0c1Byb2Nlc3NvciA9IFNlYXJjaFByb2Nlc3NvciAmIHt9O1xyXG5cclxuZXhwb3J0IHR5cGUgSW5nZXN0UGlwZWxpbmVDb25maWcgPSB7XHJcbiAgZGVzY3JpcHRpb24/OiBzdHJpbmc7XHJcbiAgcHJvY2Vzc29yczogSW5nZXN0UHJvY2Vzc29yW107XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hQaXBlbGluZUNvbmZpZyA9IHtcclxuICBkZXNjcmlwdGlvbj86IHN0cmluZztcclxuICByZXF1ZXN0X3Byb2Nlc3NvcnM/OiBTZWFyY2hSZXF1ZXN0UHJvY2Vzc29yW107XHJcbiAgcmVzcG9uc2VfcHJvY2Vzc29ycz86IFNlYXJjaFJlc3BvbnNlUHJvY2Vzc29yW107XHJcbiAgcGhhc2VfcmVzdWx0c19wcm9jZXNzb3JzPzogU2VhcmNoUGhhc2VSZXN1bHRzUHJvY2Vzc29yW107XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hUZW1wbGF0ZUNvbmZpZyA9IHtcclxuICBsYW5nOiBzdHJpbmc7XHJcbiAgc291cmNlOiBhbnk7XHJcbiAgb3B0aW9uczogYW55O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgTUxJbmZlcmVuY2VQcm9jZXNzb3IgPSBJbmdlc3RQcm9jZXNzb3IgJiB7XHJcbiAgbWxfaW5mZXJlbmNlOiB7XHJcbiAgICBtb2RlbF9pZDogc3RyaW5nO1xyXG4gICAgaW5wdXRfbWFwPzoge31bXTtcclxuICAgIG91dHB1dF9tYXA/OiB7fVtdO1xyXG4gICAgW2tleTogc3RyaW5nXTogYW55O1xyXG4gIH07XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBOb3JtYWxpemF0aW9uUHJvY2Vzc29yID0gU2VhcmNoUHJvY2Vzc29yICYge1xyXG4gIG5vcm1hbGl6YXRpb246IHtcclxuICAgIHRlY2huaXF1ZTogc3RyaW5nO1xyXG4gIH07XHJcbiAgY29tYmluYXRpb246IHtcclxuICAgIHRlY2huaXF1ZTogc3RyaW5nO1xyXG4gICAgcGFyYW1ldGVyczoge1xyXG4gICAgICB3ZWlnaHRzOiBudW1iZXJbXTtcclxuICAgIH07XHJcbiAgfTtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIEluZGV4Q29uZmlndXJhdGlvbiA9IHtcclxuICBzZXR0aW5nczogeyBba2V5OiBzdHJpbmddOiBhbnkgfTtcclxuICBtYXBwaW5nczogSW5kZXhNYXBwaW5ncztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIEluZGV4TWFwcGluZ3MgPSB7XHJcbiAgcHJvcGVydGllczogeyBba2V5OiBzdHJpbmddOiBhbnkgfTtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFRlbXBsYXRlTm9kZSA9IHtcclxuICBpZDogc3RyaW5nO1xyXG4gIHR5cGU6IHN0cmluZztcclxuICBwcmV2aW91c19ub2RlX2lucHV0cz86IHt9O1xyXG4gIHVzZXJfaW5wdXRzPzoge307XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBDcmVhdGVJbmdlc3RQaXBlbGluZU5vZGUgPSBUZW1wbGF0ZU5vZGUgJiB7XHJcbiAgdXNlcl9pbnB1dHM6IHtcclxuICAgIHBpcGVsaW5lX2lkOiBzdHJpbmc7XHJcbiAgICBtb2RlbF9pZD86IHN0cmluZztcclxuICAgIGlucHV0X2ZpZWxkPzogc3RyaW5nO1xyXG4gICAgb3V0cHV0X2ZpZWxkPzogc3RyaW5nO1xyXG4gICAgY29uZmlndXJhdGlvbnM6IHN0cmluZztcclxuICB9O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQ3JlYXRlU2VhcmNoUGlwZWxpbmVOb2RlID0gVGVtcGxhdGVOb2RlICYge1xyXG4gIHVzZXJfaW5wdXRzOiB7XHJcbiAgICBwaXBlbGluZV9pZDogc3RyaW5nO1xyXG4gICAgY29uZmlndXJhdGlvbnM6IHN0cmluZztcclxuICB9O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQ3JlYXRlSW5kZXhOb2RlID0gVGVtcGxhdGVOb2RlICYge1xyXG4gIHByZXZpb3VzX25vZGVfaW5wdXRzPzoge1xyXG4gICAgW2luZ2VzdF9waXBlbGluZV9zdGVwX2lkOiBzdHJpbmddOiBzdHJpbmc7XHJcbiAgfTtcclxuICB1c2VyX2lucHV0czoge1xyXG4gICAgaW5kZXhfbmFtZTogc3RyaW5nO1xyXG4gICAgY29uZmlndXJhdGlvbnM6IHN0cmluZztcclxuICB9O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgVGVtcGxhdGVFZGdlID0ge1xyXG4gIHNvdXJjZTogc3RyaW5nO1xyXG4gIGRlc3Q6IHN0cmluZztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFRlbXBsYXRlRmxvdyA9IHtcclxuICBub2RlczogVGVtcGxhdGVOb2RlW107XHJcbiAgZWRnZXM/OiBUZW1wbGF0ZUVkZ2VbXTtcclxuICB1c2VyX3BhcmFtcz86IHt9O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgVGVtcGxhdGVGbG93cyA9IHtcclxuICBwcm92aXNpb246IFRlbXBsYXRlRmxvdztcclxufTtcclxuXHJcbi8vIEEgc3RhdGVsZXNzIHRlbXBsYXRlIG9mIGEgd29ya2Zsb3dcclxuZXhwb3J0IHR5cGUgV29ya2Zsb3dUZW1wbGF0ZSA9IHtcclxuICAvLyBOYW1lIGlzIHRoZSBvbmx5IHJlcXVpcmVkIGZpZWxkOiBzZWUgaHR0cHM6Ly9vcGVuc2VhcmNoLm9yZy9kb2NzL2xhdGVzdC9hdXRvbWF0aW5nLWNvbmZpZ3VyYXRpb25zL2FwaS9jcmVhdGUtd29ya2Zsb3cvI3JlcXVlc3QtZmllbGRzXHJcbiAgbmFtZTogc3RyaW5nO1xyXG4gIGRlc2NyaXB0aW9uPzogc3RyaW5nO1xyXG4gIC8vIFRPRE86IGZpbmFsaXplIG9uIHZlcnNpb24gdHlwZSB3aGVuIHRoYXQgaXMgaW1wbGVtZW50ZWRcclxuICAvLyBodHRwczovL2dpdGh1Yi5jb20vb3BlbnNlYXJjaC1wcm9qZWN0L2Zsb3ctZnJhbWV3b3JrL2lzc3Vlcy81MjZcclxuICB2ZXJzaW9uPzogYW55O1xyXG4gIHdvcmtmbG93cz86IFRlbXBsYXRlRmxvd3M7XHJcbiAgdXNlX2Nhc2U/OiBzdHJpbmc7XHJcbiAgLy8gVUkgc3RhdGUgYW5kIGFueSBSZWFjdEZsb3cgc3RhdGUgbWF5IG5vdCBleGlzdCBpZiBhIHdvcmtmbG93IGlzIGNyZWF0ZWQgdmlhIEFQSS9iYWNrZW5kLW9ubHkuXHJcbiAgdWlfbWV0YWRhdGE/OiBVSVN0YXRlO1xyXG59O1xyXG5cclxuLy8gQW4gaW5zdGFuY2Ugb2YgYSB3b3JrZmxvdyBiYXNlZCBvbiBhIHdvcmtmbG93IHRlbXBsYXRlXHJcbmV4cG9ydCB0eXBlIFdvcmtmbG93ID0gV29ya2Zsb3dUZW1wbGF0ZSAmIHtcclxuICAvLyB3b24ndCBleGlzdCB1bnRpbCBjcmVhdGVkIGluIGJhY2tlbmRcclxuICBpZD86IHN0cmluZztcclxuICAvLyB3b24ndCBleGlzdCB1bnRpbCBjcmVhdGVkIGluIGJhY2tlbmRcclxuICBsYXN0VXBkYXRlZD86IG51bWJlcjtcclxuICAvLyB3b24ndCBleGlzdCB1bnRpbCBsYXVuY2hlZC9wcm92aXNpb25lZCBpbiBiYWNrZW5kXHJcbiAgbGFzdExhdW5jaGVkPzogbnVtYmVyO1xyXG4gIC8vIHdvbid0IGV4aXN0IHVudGlsIGxhdW5jaGVkL3Byb3Zpc2lvbmVkIGluIGJhY2tlbmRcclxuICBzdGF0ZT86IFdPUktGTE9XX1NUQVRFO1xyXG4gIC8vIHdvbid0IGV4aXN0IHVudGlsIGxhdW5jaGVkL3Byb3Zpc2lvbmVkIGluIGJhY2tlbmRcclxuICBlcnJvcj86IHN0cmluZztcclxuICAvLyB3b24ndCBleGlzdCB1bnRpbCBsYXVuY2hlZC9wcm92aXNpb25lZCBpbiBiYWNrZW5kXHJcbiAgcmVzb3VyY2VzQ3JlYXRlZD86IFdvcmtmbG93UmVzb3VyY2VbXTtcclxufTtcclxuXHJcbi8qKlxyXG4gKioqKioqKioqKiBNTCBQTFVHSU4gVFlQRVMvSU5URVJGQUNFUyAqKioqKioqKioqXHJcbiAqL1xyXG5cclxuLy8gQmFzZWQgb2ZmIG9mIGh0dHBzOi8vZ2l0aHViLmNvbS9vcGVuc2VhcmNoLXByb2plY3QvbWwtY29tbW9ucy9ibG9iL21haW4vY29tbW9uL3NyYy9tYWluL2phdmEvb3JnL29wZW5zZWFyY2gvbWwvY29tbW9uL21vZGVsL01MTW9kZWxTdGF0ZS5qYXZhXHJcbmV4cG9ydCBlbnVtIE1PREVMX1NUQVRFIHtcclxuICBSRUdJU1RFUkVEID0gJ1JlZ2lzdGVyZWQnLFxyXG4gIFJFR0lTVEVSSU5HID0gJ1JlZ2lzdGVyaW5nJyxcclxuICBERVBMT1lJTkcgPSAnRGVwbG95aW5nJyxcclxuICBERVBMT1lFRCA9ICdEZXBsb3llZCcsXHJcbiAgUEFSVElBTExZX0RFUExPWUVEID0gJ1BhcnRpYWxseSBkZXBsb3llZCcsXHJcbiAgVU5ERVBMT1lFRCA9ICdVbmRlcGxveWVkJyxcclxuICBERVBMT1lfRkFJTEVEID0gJ0RlcGxveSBmYWlsZWQnLFxyXG59XHJcblxyXG4vLyBCYXNlZCBvZmYgb2YgaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9tbC1jb21tb25zL2Jsb2IvbWFpbi9jb21tb24vc3JjL21haW4vamF2YS9vcmcvb3BlbnNlYXJjaC9tbC9jb21tb24vRnVuY3Rpb25OYW1lLmphdmFcclxuZXhwb3J0IGVudW0gTU9ERUxfQUxHT1JJVEhNIHtcclxuICBMSU5FQVJfUkVHUkVTU0lPTiA9ICdMaW5lYXIgcmVncmVzc2lvbicsXHJcbiAgS01FQU5TID0gJ0stbWVhbnMnLFxyXG4gIEFEX0xJQlNWTSA9ICdBRCBMSUJTVk0nLFxyXG4gIFNBTVBMRV9BTEdPID0gJ1NhbXBsZSBhbGdvcml0aG0nLFxyXG4gIExPQ0FMX1NBTVBMRV9DQUxDVUxBVE9SID0gJ0xvY2FsIHNhbXBsZSBjYWxjdWxhdG9yJyxcclxuICBGSVRfUkNGID0gJ0ZpdCBSQ0YnLFxyXG4gIEJBVENIX1JDRiA9ICdCYXRjaCBSQ0YnLFxyXG4gIEFOT01BTFlfTE9DQUxJWkFUSU9OID0gJ0Fub21hbHkgbG9jYWxpemF0aW9uJyxcclxuICBSQ0ZfU1VNTUFSSVpFID0gJ1JDRiBzdW1tYXJpemUnLFxyXG4gIExPR0lTVElDX1JFR1JFU1NJT04gPSAnTG9naXN0aWMgcmVncmVzc2lvbicsXHJcbiAgVEVYVF9FTUJFRERJTkcgPSAnVGV4dCBlbWJlZGRpbmcnLFxyXG4gIE1FVFJJQ1NfQ09SUkVMQVRJT04gPSAnTWV0cmljcyBjb3JyZWxhdGlvbicsXHJcbiAgUkVNT1RFID0gJ1JlbW90ZScsXHJcbiAgU1BBUlNFX0VOQ09ESU5HID0gJ1NwYXJzZSBlbmNvZGluZycsXHJcbiAgU1BBUlNFX1RPS0VOSVpFID0gJ1NwYXJzZSB0b2tlbml6ZScsXHJcbiAgVEVYVF9TSU1JTEFSSVRZID0gJ1RleHQgc2ltaWxhcml0eScsXHJcbiAgUVVFU1RJT05fQU5TV0VSSU5HID0gJ1F1ZXN0aW9uIGFuc3dlcmluZycsXHJcbiAgQUdFTlQgPSAnQWdlbnQnLFxyXG59XHJcblxyXG5leHBvcnQgdHlwZSBNb2RlbENvbmZpZyA9IHtcclxuICBtb2RlbFR5cGU/OiBzdHJpbmc7XHJcbiAgZW1iZWRkaW5nRGltZW5zaW9uPzogbnVtYmVyO1xyXG59O1xyXG5cclxuLy8gQmFzZWQgb2ZmIG9mIEpTT05TY2hlbWEuIEZvciBtb3JlIGluZm8sIHNlZSBodHRwczovL2pzb24tc2NoZW1hLm9yZy91bmRlcnN0YW5kaW5nLWpzb24tc2NoZW1hL3JlZmVyZW5jZS90eXBlXHJcbmV4cG9ydCB0eXBlIE1vZGVsSW5wdXQgPSB7XHJcbiAgdHlwZTogc3RyaW5nO1xyXG4gIGRlc2NyaXB0aW9uPzogc3RyaW5nO1xyXG4gIGl0ZW1zPzogTW9kZWxJbnB1dDtcclxuICBwcm9wZXJ0aWVzPzogTW9kZWxJbnB1dE1hcDtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIE1vZGVsT3V0cHV0ID0gTW9kZWxJbnB1dDtcclxuXHJcbmV4cG9ydCB0eXBlIE1vZGVsSW5wdXRNYXAgPSB7IFtrZXk6IHN0cmluZ106IE1vZGVsSW5wdXQgfTtcclxuZXhwb3J0IHR5cGUgTW9kZWxPdXRwdXRNYXAgPSB7IFtrZXk6IHN0cmluZ106IE1vZGVsT3V0cHV0IH07XHJcblxyXG4vLyBGb3IgcmVuZGVyaW5nIG9wdGlvbnMsIHdlIGV4dHJhY3QgdGhlIG5hbWUgKHRoZSBrZXkgaW4gdGhlIGlucHV0L291dHB1dCBvYmopIGFuZCBjb21iaW5lIGludG8gYSBzaW5nbGUgb2JqLlxyXG4vLyBBbHNvIHBlcnNpc3QgYW4gb3B0aW9uYWwgZmllbGQgdG8gZHluYW1pY2FsbHkgcnVuIC8gbm90IHJ1biB2YWxpZGF0aW9uIG9uIHRoZSBmb3JtIGxhdGVyIG9uXHJcbmV4cG9ydCB0eXBlIE1vZGVsSW5wdXRGb3JtRmllbGQgPSBNb2RlbElucHV0ICYge1xyXG4gIGxhYmVsOiBzdHJpbmc7XHJcbiAgb3B0aW9uYWw/OiBib29sZWFuO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgTW9kZWxPdXRwdXRGb3JtRmllbGQgPSBNb2RlbElucHV0Rm9ybUZpZWxkO1xyXG5cclxuZXhwb3J0IHR5cGUgTW9kZWxJbnRlcmZhY2UgPSB7XHJcbiAgaW5wdXQ/OiBNb2RlbElucHV0O1xyXG4gIG91dHB1dD86IE1vZGVsT3V0cHV0O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQ29ubmVjdG9yUGFyYW1ldGVycyA9IHtcclxuICBtb2RlbD86IHN0cmluZztcclxuICBkaW1lbnNpb25zPzogbnVtYmVyO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgTW9kZWwgPSB7XHJcbiAgaWQ6IHN0cmluZztcclxuICBuYW1lOiBzdHJpbmc7XHJcbiAgYWxnb3JpdGhtOiBNT0RFTF9BTEdPUklUSE07XHJcbiAgc3RhdGU6IE1PREVMX1NUQVRFO1xyXG4gIG1vZGVsQ29uZmlnPzogTW9kZWxDb25maWc7XHJcbiAgaW50ZXJmYWNlPzogTW9kZWxJbnRlcmZhY2U7XHJcbiAgY29ubmVjdG9ySWQ/OiBzdHJpbmc7XHJcbiAgY29ubmVjdG9yPzogUGFydGlhbDxDb25uZWN0b3I+O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQ29ubmVjdG9yID0ge1xyXG4gIGlkOiBzdHJpbmc7XHJcbiAgbmFtZTogc3RyaW5nO1xyXG4gIHByb3RvY29sPzogQ09OTkVDVE9SX1BST1RPQ09MO1xyXG4gIHBhcmFtZXRlcnM/OiBDb25uZWN0b3JQYXJhbWV0ZXJzO1xyXG4gIGFjdGlvbnM6IHt9W107XHJcbiAgY2xpZW50X2NvbmZpZz86IHt9O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgTW9kZWxEaWN0ID0ge1xyXG4gIFttb2RlbElkOiBzdHJpbmddOiBNb2RlbDtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIENvbm5lY3RvckRpY3QgPSB7XHJcbiAgW2Nvbm5lY3RvcklkOiBzdHJpbmddOiBDb25uZWN0b3I7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBNQ1BDb25uZWN0b3IgPSB7XHJcbiAgbWNwX2Nvbm5lY3Rvcl9pZDogc3RyaW5nO1xyXG4gIHRvb2xfZmlsdGVyczogc3RyaW5nW107XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBBZ2VudENvbmZpZ1BhcmFtZXRlcnMgPSB7XHJcbiAgX2xsbV9pbnRlcmZhY2U/OiBBR0VOVF9MTE1fSU5URVJGQUNFX1RZUEU7XHJcbiAgbWNwX2Nvbm5lY3RvcnM/OiBNQ1BDb25uZWN0b3JbXTtcclxuICBba2V5OiBzdHJpbmddOiBhbnk7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBUb29sID0ge1xyXG4gIHR5cGU6IFRPT0xfVFlQRTtcclxuICBkZXNjcmlwdGlvbj86IHN0cmluZztcclxuICBwYXJhbWV0ZXJzPzogQWdlbnRDb25maWdQYXJhbWV0ZXJzO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQWdlbnRMTE0gPSB7XHJcbiAgbW9kZWxfaWQ6IHN0cmluZztcclxuICBwYXJhbWV0ZXJzPzogQWdlbnRDb25maWdQYXJhbWV0ZXJzO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQWdlbnRNZW1vcnkgPSB7XHJcbiAgdHlwZTogQUdFTlRfTUVNT1JZX1RZUEU7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBBZ2VudCA9IHtcclxuICBpZDogc3RyaW5nO1xyXG4gIG5hbWU6IHN0cmluZztcclxuICB0eXBlOiBBR0VOVF9UWVBFO1xyXG4gIGRlc2NyaXB0aW9uPzogc3RyaW5nO1xyXG4gIHRvb2xzOiBUb29sW107XHJcbiAgbGxtPzogQWdlbnRMTE07XHJcbiAgbWVtb3J5PzogQWdlbnRNZW1vcnk7XHJcbiAgcGFyYW1ldGVycz86IEFnZW50Q29uZmlnUGFyYW1ldGVycztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIEFnZW50RGljdCA9IHtcclxuICBbYWdlbnRJZDogc3RyaW5nXTogQWdlbnQ7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBNb2RlbEZvcm1WYWx1ZSA9IHtcclxuICBpZDogc3RyaW5nO1xyXG4gIGFsZ29yaXRobT86IE1PREVMX0FMR09SSVRITTtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIE1vZGVsSXRlbSA9IE1vZGVsRm9ybVZhbHVlICYge1xyXG4gIG5hbWU6IHN0cmluZztcclxuICBpbnRlcmZhY2U/OiB7fTtcclxufTtcclxuXHJcbi8qKlxyXG4gKioqKioqKioqKiBNSVNDIFRZUEVTL0lOVEVSRkFDRVMgKioqKioqKioqKioqXHJcbiAqL1xyXG5cclxuLy8gQmFzZWQgb2ZmIG9mIGh0dHBzOi8vZ2l0aHViLmNvbS9vcGVuc2VhcmNoLXByb2plY3QvZmxvdy1mcmFtZXdvcmsvYmxvYi9tYWluL3NyYy9tYWluL2phdmEvb3JnL29wZW5zZWFyY2gvZmxvd2ZyYW1ld29yay9tb2RlbC9TdGF0ZS5qYXZhXHJcbmV4cG9ydCBlbnVtIFdPUktGTE9XX1NUQVRFIHtcclxuICBOT1RfU1RBUlRFRCA9ICdOb3Qgc3RhcnRlZCcsXHJcbiAgUFJPVklTSU9OSU5HID0gJ1Byb3Zpc2lvbmluZycsXHJcbiAgRkFJTEVEID0gJ0ZhaWxlZCcsXHJcbiAgQ09NUExFVEVEID0gJ0NvbXBsZXRlZCcsXHJcbn1cclxuXHJcbmV4cG9ydCB0eXBlIFdvcmtmbG93UmVzb3VyY2UgPSB7XHJcbiAgaWQ6IHN0cmluZztcclxuICBzdGVwVHlwZTogV09SS0ZMT1dfU1RFUF9UWVBFO1xyXG4gIHR5cGU6IFdPUktGTE9XX1JFU09VUkNFX1RZUEU7XHJcbn07XHJcblxyXG4vLyBCYXNlZCBvZmYgb2YgaHR0cHM6Ly9naXRodWIuY29tL29wZW5zZWFyY2gtcHJvamVjdC9mbG93LWZyYW1ld29yay9ibG9iL21haW4vc3JjL21haW4vamF2YS9vcmcvb3BlbnNlYXJjaC9mbG93ZnJhbWV3b3JrL2NvbW1vbi9Xb3JrZmxvd1Jlc291cmNlcy5qYXZhXHJcbmV4cG9ydCBlbnVtIFdPUktGTE9XX1JFU09VUkNFX1RZUEUge1xyXG4gIFBJUEVMSU5FX0lEID0gJ0luZ2VzdCBwaXBlbGluZScsXHJcbiAgSU5ERVhfTkFNRSA9ICdJbmRleCcsXHJcbiAgTU9ERUxfSUQgPSAnTW9kZWwnLFxyXG4gIE1PREVMX0dST1VQX0lEID0gJ01vZGVsIGdyb3VwJyxcclxuICBDT05ORUNUT1JfSUQgPSAnQ29ubmVjdG9yJyxcclxufVxyXG5cclxuZXhwb3J0IGVudW0gV09SS0ZMT1dfU1RFUF9UWVBFIHtcclxuICBDUkVBVEVfSU5HRVNUX1BJUEVMSU5FX1NURVBfVFlQRSA9ICdjcmVhdGVfaW5nZXN0X3BpcGVsaW5lJyxcclxuICBDUkVBVEVfU0VBUkNIX1BJUEVMSU5FX1NURVBfVFlQRSA9ICdjcmVhdGVfc2VhcmNoX3BpcGVsaW5lJyxcclxuICBDUkVBVEVfSU5ERVhfU1RFUF9UWVBFID0gJ2NyZWF0ZV9pbmRleCcsXHJcbn1cclxuXHJcbi8vIFdlIGNhbm5vdCBkaXNhbWJpZ3VhdGUgaW5nZXN0IHZzLiBzZWFyY2ggcGlwZWxpbmVzIGJhc2VkIG9uIHdvcmtmbG93IHJlc291cmNlIHR5cGUuIFRvIHdvcmsgYXJvdW5kXHJcbi8vIHRoYXQsIHdlIG1haW50YWluIHRoaXMgbWFwIGZyb20gd29ya2Zsb3cgc3RlcCB0eXBlIC0+IGZvcm1hdHRlZCB0eXBlXHJcbmV4cG9ydCBlbnVtIFdPUktGTE9XX1NURVBfVE9fUkVTT1VSQ0VfVFlQRV9NQVAge1xyXG4gICdjcmVhdGVfaW5nZXN0X3BpcGVsaW5lJyA9ICdJbmdlc3QgcGlwZWxpbmUnLFxyXG4gICdjcmVhdGVfc2VhcmNoX3BpcGVsaW5lJyA9ICdTZWFyY2ggcGlwZWxpbmUnLFxyXG4gICdjcmVhdGVfaW5kZXgnID0gJ0luZGV4JyxcclxufVxyXG5cclxuZXhwb3J0IHR5cGUgV29ya2Zsb3dEaWN0ID0ge1xyXG4gIFt3b3JrZmxvd0lkOiBzdHJpbmddOiBXb3JrZmxvdztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFF1ZXJ5UHJlc2V0ID0ge1xyXG4gIG5hbWU6IHN0cmluZztcclxuICBxdWVyeTogc3RyaW5nO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgUHJvbXB0UHJlc2V0ID0ge1xyXG4gIG5hbWU6IHN0cmluZztcclxuICBwcm9tcHQ6IHN0cmluZztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFF1aWNrQ29uZmlndXJlRmllbGRzID0ge1xyXG4gIGVtYmVkZGluZ01vZGVsSWQ/OiBzdHJpbmc7XHJcbiAgbGxtSWQ/OiBzdHJpbmc7XHJcbiAgdmVjdG9yRmllbGQ/OiBzdHJpbmc7XHJcbiAgdGV4dEZpZWxkPzogc3RyaW5nO1xyXG4gIGltYWdlRmllbGQ/OiBzdHJpbmc7XHJcbiAgbGFiZWxGaWVsZD86IHN0cmluZztcclxuICBwcm9tcHRGaWVsZD86IHN0cmluZztcclxuICBlbWJlZGRpbmdMZW5ndGg/OiBudW1iZXI7XHJcbiAgbGxtUmVzcG9uc2VGaWVsZD86IHN0cmluZztcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIFF1ZXJ5UGFyYW1UeXBlID0gJ1RleHQnIHwgJ0JpbmFyeSc7XHJcblxyXG5leHBvcnQgdHlwZSBRdWVyeVBhcmFtID0ge1xyXG4gIG5hbWU6IHN0cmluZztcclxuICB0eXBlOiBRdWVyeVBhcmFtVHlwZTtcclxuICB2YWx1ZTogc3RyaW5nO1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgQ2FjaGVkRm9ybWlrU3RhdGUgPSB7XHJcbiAgZXJyb3JzPzoge307XHJcbiAgdG91Y2hlZD86IHt9O1xyXG59O1xyXG5cclxuZXhwb3J0IHR5cGUgSW5nZXN0UGlwZWxpbmVFcnJvcnMgPSB7XHJcbiAgW2lkeDogbnVtYmVyXTogeyBwcm9jZXNzb3JUeXBlOiBzdHJpbmc7IGVycm9yTXNnOiBzdHJpbmcgfTtcclxufTtcclxuXHJcbi8qKlxyXG4gKioqKioqKioqKiBPUEVOU0VBUkNIIFRZUEVTL0lOVEVSRkFDRVMgKioqKioqKioqKioqXHJcbiAqL1xyXG5cclxuLy8gZnJvbSBodHRwczovL29wZW5zZWFyY2gub3JnL2RvY3MvbGF0ZXN0L2luZ2VzdC1waXBlbGluZXMvc2ltdWxhdGUtaW5nZXN0LyNleGFtcGxlLXNwZWNpZnktYS1waXBlbGluZS1pbi10aGUtcGF0aFxyXG5leHBvcnQgdHlwZSBTaW11bGF0ZUluZ2VzdFBpcGVsaW5lRG9jID0ge1xyXG4gIF9pbmRleDogc3RyaW5nO1xyXG4gIF9pZDogc3RyaW5nO1xyXG4gIF9zb3VyY2U6IGFueTtcclxufTtcclxuXHJcbi8vIGZyb20gaHR0cHM6Ly9vcGVuc2VhcmNoLm9yZy9kb2NzL2xhdGVzdC9pbmdlc3QtcGlwZWxpbmVzL3NpbXVsYXRlLWluZ2VzdC8jZXhhbXBsZS1zcGVjaWZ5LWEtcGlwZWxpbmUtaW4tdGhlLXBhdGhcclxuZXhwb3J0IHR5cGUgU2ltdWxhdGVJbmdlc3RQaXBlbGluZURvY1Jlc3BvbnNlID0ge1xyXG4gIGRvYzogU2ltdWxhdGVJbmdlc3RQaXBlbGluZURvYyAmIHtcclxuICAgIF9pbmdlc3Q6IHtcclxuICAgICAgdGltZXN0YW1wOiBzdHJpbmc7XHJcbiAgICB9O1xyXG4gIH07XHJcbiAgZXJyb3I/OiB7XHJcbiAgICByZWFzb246IHN0cmluZztcclxuICB9O1xyXG59O1xyXG5cclxuLy8gZnJvbSBodHRwczovL29wZW5zZWFyY2gub3JnL2RvY3MvbGF0ZXN0L2luZ2VzdC1waXBlbGluZXMvc2ltdWxhdGUtaW5nZXN0LyNleGFtcGxlLXNwZWNpZnktYS1waXBlbGluZS1pbi10aGUtcGF0aFxyXG5leHBvcnQgdHlwZSBTaW11bGF0ZUluZ2VzdFBpcGVsaW5lUmVzcG9uc2UgPSB7XHJcbiAgZG9jczogU2ltdWxhdGVJbmdlc3RQaXBlbGluZURvY1Jlc3BvbnNlW107XHJcbn07XHJcblxyXG4vLyB2ZXJib3NlIG1vZGVcclxuLy8gZnJvbSBodHRwczovL29wZW5zZWFyY2gub3JnL2RvY3MvbGF0ZXN0L2luZ2VzdC1waXBlbGluZXMvc2ltdWxhdGUtaW5nZXN0LyNxdWVyeS1wYXJhbWV0ZXJzXHJcbmV4cG9ydCB0eXBlIFNpbXVsYXRlSW5nZXN0UGlwZWxpbmVEb2NSZXNwb25zZVZlcmJvc2UgPSBTaW11bGF0ZUluZ2VzdFBpcGVsaW5lRG9jUmVzcG9uc2UgJiB7XHJcbiAgcHJvY2Vzc29yX3R5cGU6IHN0cmluZztcclxuICBzdGF0dXM6ICdzdWNjZXNzJyB8ICdlcnJvcic7XHJcbiAgZGVzY3JpcHRpb24/OiBzdHJpbmc7XHJcbn07XHJcblxyXG4vLyB2ZXJib3NlIG1vZGVcclxuLy8gZnJvbSBodHRwczovL29wZW5zZWFyY2gub3JnL2RvY3MvbGF0ZXN0L2luZ2VzdC1waXBlbGluZXMvc2ltdWxhdGUtaW5nZXN0LyNxdWVyeS1wYXJhbWV0ZXJzXHJcbmV4cG9ydCB0eXBlIFNpbXVsYXRlSW5nZXN0UGlwZWxpbmVSZXNwb25zZVZlcmJvc2UgPSB7XHJcbiAgZG9jczogW1xyXG4gICAge1xyXG4gICAgICBwcm9jZXNzb3JfcmVzdWx0czogU2ltdWxhdGVJbmdlc3RQaXBlbGluZURvY1Jlc3BvbnNlVmVyYm9zZVtdO1xyXG4gICAgfVxyXG4gIF07XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hIaXQgPSBTaW11bGF0ZUluZ2VzdFBpcGVsaW5lRG9jO1xyXG5cclxuZXhwb3J0IHR5cGUgU2VhcmNoUmVzcG9uc2UgPSB7XHJcbiAgdG9vazogbnVtYmVyO1xyXG4gIHRpbWVkX291dDogYm9vbGVhbjtcclxuICBfc2hhcmRzOiB7XHJcbiAgICB0b3RhbDogbnVtYmVyO1xyXG4gICAgc3VjY2Vzc2Z1bDogbnVtYmVyO1xyXG4gICAgc2tpcHBlZDogbnVtYmVyO1xyXG4gICAgZmFpbGVkOiBudW1iZXI7XHJcbiAgfTtcclxuICBoaXRzOiB7XHJcbiAgICB0b3RhbDoge1xyXG4gICAgICB2YWx1ZTogbnVtYmVyO1xyXG4gICAgICByZWxhdGlvbjogc3RyaW5nO1xyXG4gICAgfTtcclxuICAgIG1heF9zY29yZTogbnVtYmVyO1xyXG4gICAgaGl0czogU2VhcmNoSGl0W107XHJcbiAgfTtcclxuICBhZ2dyZWdhdGlvbnM/OiB7fTtcclxuICBleHQ/OiBhbnk7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hQcm9jZXNzb3JJbnB1dERhdGEgPSB7XHJcbiAgX2luZGV4OiBzdHJpbmc7XHJcbiAgX2lkOiBzdHJpbmc7XHJcbiAgX3Njb3JlOiBudW1iZXI7XHJcbiAgX3NvdXJjZToge307XHJcbn07XHJcbmV4cG9ydCB0eXBlIFNlYXJjaFByb2Nlc3Nvck91dHB1dERhdGEgPSBTZWFyY2hQcm9jZXNzb3JJbnB1dERhdGE7XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hQcm9jZXNzb3JSZXN1bHQgPSB7XHJcbiAgcHJvY2Vzc29yX25hbWU6IHN0cmluZztcclxuICBkdXJhdGlvbl9taWxsaXM6IG51bWJlcjtcclxuICBzdGF0dXM6ICdzdWNjZXNzJyB8ICdmYWlsJztcclxuICBlcnJvcj86IHN0cmluZztcclxuICBpbnB1dF9kYXRhOiBTZWFyY2hQcm9jZXNzb3JJbnB1dERhdGFbXSB8IG51bGw7XHJcbiAgb3V0cHV0X2RhdGE6IFNlYXJjaFByb2Nlc3Nvck91dHB1dERhdGFbXSB8IG51bGw7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hSZXNwb25zZVZlcmJvc2UgPSBTZWFyY2hSZXNwb25zZSAmIHtcclxuICBwcm9jZXNzb3JfcmVzdWx0czogU2VhcmNoUHJvY2Vzc29yUmVzdWx0W107XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hQaXBlbGluZUVycm9ycyA9IEluZ2VzdFBpcGVsaW5lRXJyb3JzO1xyXG5cclxuZXhwb3J0IHR5cGUgSW5kZXhSZXNwb25zZSA9IHtcclxuICBpbmRleE5hbWU6IHN0cmluZztcclxuICBpbmRleERldGFpbHM6IEluZGV4Q29uZmlndXJhdGlvbjtcclxufTtcclxuXHJcbmV4cG9ydCB0eXBlIEluZ2VzdFBpcGVsaW5lUmVzcG9uc2UgPSB7XHJcbiAgcGlwZWxpbmVJZDogc3RyaW5nO1xyXG4gIGluZ2VzdFBpcGVsaW5lRGV0YWlsczogSW5nZXN0UGlwZWxpbmVDb25maWc7XHJcbn07XHJcblxyXG5leHBvcnQgdHlwZSBTZWFyY2hQaXBlbGluZVJlc3BvbnNlID0ge1xyXG4gIHBpcGVsaW5lSWQ6IHN0cmluZztcclxuICBzZWFyY2hQaXBlbGluZURldGFpbHM6IFNlYXJjaFBpcGVsaW5lQ29uZmlnO1xyXG59O1xyXG4iXSwibWFwcGluZ3MiOiI7Ozs7OztBQUFBO0FBQ0E7QUFDQTtBQUNBO0FBc0JBO0FBQ0E7QUFDQTtBQStIQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUVBO0FBS0E7QUFLQTtBQUdBO0FBS0E7QUFLQTtBQUNBO0FBQ0E7QUF3QkE7QUFDQTtBQUNBO0FBQ0E7QUFXQTtBQUNBO0FBQ0E7QUFVQTtBQUNBO0FBQ0E7QUFDQTtBQWlDQTtBQUNBO0FBQ0E7QUF5R0E7QUFjQTtBQWdCQTtBQUNBO0FBQ0E7QUFFQTtBQUFBLElBQ1lBLFdBQVcsR0FBQUMsT0FBQSxDQUFBRCxXQUFBLDBCQUFYQSxXQUFXO0VBQVhBLFdBQVc7RUFBWEEsV0FBVztFQUFYQSxXQUFXO0VBQVhBLFdBQVc7RUFBWEEsV0FBVztFQUFYQSxXQUFXO0VBQVhBLFdBQVc7RUFBQSxPQUFYQSxXQUFXO0FBQUEsT0FVdkI7QUFBQSxJQUNZRSxlQUFlLEdBQUFELE9BQUEsQ0FBQUMsZUFBQSwwQkFBZkEsZUFBZTtFQUFmQSxlQUFlO0VBQWZBLGVBQWU7RUFBZkEsZUFBZTtFQUFmQSxlQUFlO0VBQWZBLGVBQWU7RUFBZkEsZUFBZTtFQUFmQSxlQUFlO0VBQWZBLGVBQWU7RUFBZkEsZUFBZTtFQUFmQSxlQUFlO0VBQWZBLGVBQWU7RUFBZkEsZUFBZTtFQUFmQSxlQUFlO0VBQWZBLGVBQWU7RUFBZkEsZUFBZTtFQUFmQSxlQUFlO0VBQWZBLGVBQWU7RUFBZkEsZUFBZTtFQUFBLE9BQWZBLGVBQWU7QUFBQSxPQTBCM0I7QUFhQTtBQUNBO0FBaUdBO0FBQ0E7QUFDQTtBQUVBO0FBQUEsSUFDWUMsY0FBYyxHQUFBRixPQUFBLENBQUFFLGNBQUEsMEJBQWRBLGNBQWM7RUFBZEEsY0FBYztFQUFkQSxjQUFjO0VBQWRBLGNBQWM7RUFBZEEsY0FBYztFQUFBLE9BQWRBLGNBQWM7QUFBQTtBQWExQjtBQUFBLElBQ1lDLHNCQUFzQixHQUFBSCxPQUFBLENBQUFHLHNCQUFBLDBCQUF0QkEsc0JBQXNCO0VBQXRCQSxzQkFBc0I7RUFBdEJBLHNCQUFzQjtFQUF0QkEsc0JBQXNCO0VBQXRCQSxzQkFBc0I7RUFBdEJBLHNCQUFzQjtFQUFBLE9BQXRCQSxzQkFBc0I7QUFBQTtBQUFBLElBUXRCQyxrQkFBa0IsR0FBQUosT0FBQSxDQUFBSSxrQkFBQSwwQkFBbEJBLGtCQUFrQjtFQUFsQkEsa0JBQWtCO0VBQWxCQSxrQkFBa0I7RUFBbEJBLGtCQUFrQjtFQUFBLE9BQWxCQSxrQkFBa0I7QUFBQSxPQU05QjtBQUNBO0FBQUEsSUFDWUMsa0NBQWtDLEdBQUFMLE9BQUEsQ0FBQUssa0NBQUEsMEJBQWxDQSxrQ0FBa0M7RUFBbENBLGtDQUFrQztFQUFsQ0Esa0NBQWtDO0VBQWxDQSxrQ0FBa0M7RUFBQSxPQUFsQ0Esa0NBQWtDO0FBQUE7QUFpRDlDO0FBQ0E7QUFDQTtBQUVBO0FBT0E7QUFZQTtBQUtBO0FBQ0E7QUFPQTtBQUNBIn0=