"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentFrameworkStorageService = void 0;
var _message_parser_runner = require("../../utils/message_parser_runner");
var _constants = require("../../utils/constants");
var _format = require("../../utils/format");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

class AgentFrameworkStorageService {
  constructor(clientTransport, messageParsers = []) {
    this.clientTransport = clientTransport;
    this.messageParsers = messageParsers;
  }
  async getConversation(conversationId) {
    const [interactionsResp, conversation] = await Promise.all([this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}/messages?max_results=1000`
    }), this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}`
    })]);
    const finalInteractions = interactionsResp.body.messages.map(item => (0, _format.formatInteractionFromBackend)(item));
    return {
      title: conversation.body.name,
      createdTimeMs: +new Date(conversation.body.create_time),
      updatedTimeMs: +new Date(conversation.body.updated_time),
      messages: await this.getMessagesFromInteractions(finalInteractions),
      interactions: finalInteractions
    };
  }
  async getConversations(query) {
    let sortField = '';
    if (query.sortField === 'updatedTimeMs') {
      sortField = 'updated_time';
    } else if (query.sortField === 'createTimeMs') {
      sortField = 'create_time';
    }
    let searchFields = [];
    if (query.search && query.searchFields) {
      if (typeof query.searchFields === 'string') {
        searchFields = [...searchFields, query.searchFields.replace('title', 'name')];
      } else {
        searchFields = query.searchFields.map(item => item.replace('title', 'name'));
      }
    }
    const requestParams = {
      from: (query.page - 1) * query.perPage,
      size: query.perPage,
      ...(searchFields.length > 0 && {
        query: {
          multi_match: {
            query: query.search,
            fields: searchFields
          }
        }
      }),
      ...(searchFields.length === 0 && {
        query: {
          match_all: {}
        }
      }),
      ...(sortField && query.sortOrder && {
        sort: [{
          [sortField]: query.sortOrder
        }]
      })
    };
    const conversations = await this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/_search`,
      body: requestParams
    });
    return {
      objects: conversations.body.hits.hits.filter(hit => hit._source !== null && hit._source !== undefined).map(item => ({
        id: item._id,
        title: item._source.name,
        version: 1,
        createdTimeMs: Date.parse(item._source.create_time),
        updatedTimeMs: Date.parse(item._source.updated_time),
        messages: []
      })),
      total: typeof conversations.body.hits.total === 'number' ? conversations.body.hits.total : conversations.body.hits.total.value
    };
  }
  async saveMessages(title, conversationId, messages) {
    throw new Error('Method is not needed');
  }
  async deleteConversation(conversationId) {
    const response = await this.clientTransport.request({
      method: 'DELETE',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}`
    });
    if (response.statusCode === 200) {
      return {
        success: true
      };
    } else {
      return {
        success: false,
        statusCode: response.statusCode,
        message: JSON.stringify(response.body)
      };
    }
  }
  async updateConversation(conversationId, title) {
    const response = await this.clientTransport.request({
      method: 'PUT',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/${encodeURIComponent(conversationId)}`,
      body: {
        name: title
      }
    });
    if (response.statusCode === 200) {
      return {
        success: true
      };
    } else {
      return {
        success: false,
        statusCode: response.statusCode,
        message: JSON.stringify(response.body)
      };
    }
  }
  async getTraces(interactionId) {
    const response = await this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/message/${encodeURIComponent(interactionId)}/traces`
    });
    return response.body.traces.map(item => ({
      interactionId: item.message_id,
      input: item.input,
      output: item.response,
      createTime: item.create_time,
      origin: item.origin,
      traceNumber: item.trace_number
    }));
  }
  async updateInteraction(interactionId, additionalInfo) {
    const response = await this.clientTransport.request({
      method: 'PUT',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/message/${encodeURIComponent(interactionId)}`,
      body: {
        additional_info: additionalInfo
      }
    });
    if (response.statusCode === 200) {
      return {
        success: true
      };
    } else {
      return {
        success: false,
        statusCode: response.statusCode,
        message: JSON.stringify(response.body)
      };
    }
  }
  async getMessagesFromInteractions(interactions) {
    const messageParserRunner = new _message_parser_runner.MessageParserRunner(this.messageParsers);
    const finalInteractions = [...interactions];
    let finalMessages = [];
    for (const interaction of finalInteractions) {
      finalMessages = [...finalMessages, ...(await messageParserRunner.run(interaction, {
        interactions: [...(finalInteractions || [])]
      }))];
    }
    return finalMessages;
  }
  async getInteraction(conversationId, interactionId) {
    if (!conversationId) {
      throw new Error('conversationId is required');
    }
    if (!interactionId) {
      throw new Error('interactionId is required');
    }
    const interactionsResp = await this.clientTransport.request({
      method: 'GET',
      path: `${_constants.ML_COMMONS_BASE_API}/memory/message/${encodeURIComponent(interactionId)}`
    });
    return (0, _format.formatInteractionFromBackend)(interactionsResp.body);
  }
}
exports.AgentFrameworkStorageService = AgentFrameworkStorageService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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