"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerText2VizRoutes = registerText2VizRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _error_handler = require("./error_handler");
var _style_single_metric = require("../utils/style_single_metric");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const inputSchema = _configSchema.schema.string({
  maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT,
  validate(value) {
    if (!value || value.trim().length === 0) {
      return "can't be empty or blank.";
    }
  }
});
function registerText2VizRoutes(router, assistantService) {
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2VEGA,
    validate: {
      body: _configSchema.schema.object({
        input_question: inputSchema,
        input_instruction: _configSchema.schema.maybe(_configSchema.schema.string({
          maxLength: _llm.TEXT2VEGA_INPUT_SIZE_LIMIT
        })),
        ppl: _configSchema.schema.string(),
        dataSchema: _configSchema.schema.string(),
        sampleData: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const agentConfigName = req.body.input_instruction ? _llm.TEXT2VEGA_WITH_INSTRUCTIONS_AGENT_CONFIG_ID : _llm.TEXT2VEGA_RULE_BASED_AGENT_CONFIG_ID;
      const response = await assistantClient.executeAgentByConfigName(agentConfigName, {
        input_question: req.body.input_question,
        input_instruction: req.body.input_instruction,
        ppl: req.body.ppl,
        dataSchema: req.body.dataSchema,
        sampleData: req.body.sampleData
      });
      let textContent = response.body.inference_results[0].output[0].result;
      // Check if the visualization is single value:
      // it should have exactly 1 metric and no dimensions.
      let ifSingleMetric = (0, _style_single_metric.checkSingleMetric)(textContent);

      // extra content between tag <vega-lite></vega-lite>
      const startTag = '<vega-lite>';
      const endTag = '</vega-lite>';
      const startIndex = textContent.indexOf(startTag);
      const endIndex = textContent.indexOf(endTag);
      if (startIndex !== -1 && endIndex !== -1 && startIndex < endIndex) {
        // Extract the content between the tags
        textContent = textContent.substring(startIndex + startTag.length, endIndex).trim();
      }
      // extract json object
      const jsonMatch = textContent.match(/\{.*\}/s);
      if (jsonMatch) {
        let result = JSON.parse(jsonMatch[0]);
        // sometimes llm returns {response: <schema>} instead of <schema>
        if (result.response) {
          result = JSON.parse(result.response);
        }
        // Sometimes the response contains width and height which is not needed, here delete the these fields
        delete result.width;
        delete result.height;
        if (ifSingleMetric) {
          result = (0, _style_single_metric.addTitleTextLayer)(result);
        }
        // make sure $schema field always been added, sometimes, LLM 'forgot' to add this field
        result.$schema = 'https://vega.github.io/schema/vega-lite/v5.json';
        return res.ok({
          body: result
        });
      }
      return res.badRequest();
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
  router.post({
    path: _llm.TEXT2VIZ_API.TEXT2PPL,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.string(),
        question: inputSchema
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const response = await assistantClient.executeAgentByConfigName(_llm.TEXT2PPL_AGENT_CONFIG_ID, {
        question: req.body.question,
        index: req.body.index
      });
      const result = JSON.parse(response.body.inference_results[0].output[0].result);
      return res.ok({
        body: result
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfY29uZmlnU2NoZW1hIiwicmVxdWlyZSIsIl9sbG0iLCJfZXJyb3JfaGFuZGxlciIsIl9zdHlsZV9zaW5nbGVfbWV0cmljIiwiaW5wdXRTY2hlbWEiLCJzY2hlbWEiLCJzdHJpbmciLCJtYXhMZW5ndGgiLCJURVhUMlZFR0FfSU5QVVRfU0laRV9MSU1JVCIsInZhbGlkYXRlIiwidmFsdWUiLCJ0cmltIiwibGVuZ3RoIiwicmVnaXN0ZXJUZXh0MlZpelJvdXRlcyIsInJvdXRlciIsImFzc2lzdGFudFNlcnZpY2UiLCJwb3N0IiwicGF0aCIsIlRFWFQyVklaX0FQSSIsIlRFWFQyVkVHQSIsImJvZHkiLCJvYmplY3QiLCJpbnB1dF9xdWVzdGlvbiIsImlucHV0X2luc3RydWN0aW9uIiwibWF5YmUiLCJwcGwiLCJkYXRhU2NoZW1hIiwic2FtcGxlRGF0YSIsInF1ZXJ5IiwiZGF0YVNvdXJjZUlkIiwiaGFuZGxlTGVnYWN5RXJyb3JzIiwiY29udGV4dCIsInJlcSIsInJlcyIsImFzc2lzdGFudENsaWVudCIsImdldFNjb3BlZENsaWVudCIsImFnZW50Q29uZmlnTmFtZSIsIlRFWFQyVkVHQV9XSVRIX0lOU1RSVUNUSU9OU19BR0VOVF9DT05GSUdfSUQiLCJURVhUMlZFR0FfUlVMRV9CQVNFRF9BR0VOVF9DT05GSUdfSUQiLCJyZXNwb25zZSIsImV4ZWN1dGVBZ2VudEJ5Q29uZmlnTmFtZSIsInRleHRDb250ZW50IiwiaW5mZXJlbmNlX3Jlc3VsdHMiLCJvdXRwdXQiLCJyZXN1bHQiLCJpZlNpbmdsZU1ldHJpYyIsImNoZWNrU2luZ2xlTWV0cmljIiwic3RhcnRUYWciLCJlbmRUYWciLCJzdGFydEluZGV4IiwiaW5kZXhPZiIsImVuZEluZGV4Iiwic3Vic3RyaW5nIiwianNvbk1hdGNoIiwibWF0Y2giLCJKU09OIiwicGFyc2UiLCJ3aWR0aCIsImhlaWdodCIsImFkZFRpdGxlVGV4dExheWVyIiwiJHNjaGVtYSIsIm9rIiwiYmFkUmVxdWVzdCIsImUiLCJoYW5kbGVFcnJvciIsImFzc2lzdGFudF9wbHVnaW4iLCJsb2dnZXIiLCJURVhUMlBQTCIsImluZGV4IiwicXVlc3Rpb24iLCJURVhUMlBQTF9BR0VOVF9DT05GSUdfSUQiXSwic291cmNlcyI6WyJ0ZXh0MnZpel9yb3V0ZXMudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLypcclxuICogQ29weXJpZ2h0IE9wZW5TZWFyY2ggQ29udHJpYnV0b3JzXHJcbiAqIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBBcGFjaGUtMi4wXHJcbiAqL1xyXG5cclxuaW1wb3J0IHsgc2NoZW1hIH0gZnJvbSAnQG9zZC9jb25maWctc2NoZW1hJztcclxuaW1wb3J0IHsgSVJvdXRlciB9IGZyb20gJy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XHJcbmltcG9ydCB7XHJcbiAgVEVYVDJQUExfQUdFTlRfQ09ORklHX0lELFxyXG4gIFRFWFQyVkVHQV9SVUxFX0JBU0VEX0FHRU5UX0NPTkZJR19JRCxcclxuICBURVhUMlZFR0FfSU5QVVRfU0laRV9MSU1JVCxcclxuICBURVhUMlZFR0FfV0lUSF9JTlNUUlVDVElPTlNfQUdFTlRfQ09ORklHX0lELFxyXG4gIFRFWFQyVklaX0FQSSxcclxufSBmcm9tICcuLi8uLi9jb21tb24vY29uc3RhbnRzL2xsbSc7XHJcbmltcG9ydCB7IEFzc2lzdGFudFNlcnZpY2VTZXR1cCB9IGZyb20gJy4uL3NlcnZpY2VzL2Fzc2lzdGFudF9zZXJ2aWNlJztcclxuaW1wb3J0IHsgaGFuZGxlRXJyb3IgfSBmcm9tICcuL2Vycm9yX2hhbmRsZXInO1xyXG5pbXBvcnQgeyBjaGVja1NpbmdsZU1ldHJpYywgYWRkVGl0bGVUZXh0TGF5ZXIgfSBmcm9tICcuLi91dGlscy9zdHlsZV9zaW5nbGVfbWV0cmljJztcclxuXHJcbmNvbnN0IGlucHV0U2NoZW1hID0gc2NoZW1hLnN0cmluZyh7XHJcbiAgbWF4TGVuZ3RoOiBURVhUMlZFR0FfSU5QVVRfU0laRV9MSU1JVCxcclxuICB2YWxpZGF0ZSh2YWx1ZSkge1xyXG4gICAgaWYgKCF2YWx1ZSB8fCB2YWx1ZS50cmltKCkubGVuZ3RoID09PSAwKSB7XHJcbiAgICAgIHJldHVybiBcImNhbid0IGJlIGVtcHR5IG9yIGJsYW5rLlwiO1xyXG4gICAgfVxyXG4gIH0sXHJcbn0pO1xyXG5cclxuZXhwb3J0IGZ1bmN0aW9uIHJlZ2lzdGVyVGV4dDJWaXpSb3V0ZXMocm91dGVyOiBJUm91dGVyLCBhc3Npc3RhbnRTZXJ2aWNlOiBBc3Npc3RhbnRTZXJ2aWNlU2V0dXApIHtcclxuICByb3V0ZXIucG9zdChcclxuICAgIHtcclxuICAgICAgcGF0aDogVEVYVDJWSVpfQVBJLlRFWFQyVkVHQSxcclxuICAgICAgdmFsaWRhdGU6IHtcclxuICAgICAgICBib2R5OiBzY2hlbWEub2JqZWN0KHtcclxuICAgICAgICAgIGlucHV0X3F1ZXN0aW9uOiBpbnB1dFNjaGVtYSxcclxuICAgICAgICAgIGlucHV0X2luc3RydWN0aW9uOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZyh7IG1heExlbmd0aDogVEVYVDJWRUdBX0lOUFVUX1NJWkVfTElNSVQgfSkpLFxyXG4gICAgICAgICAgcHBsOiBzY2hlbWEuc3RyaW5nKCksXHJcbiAgICAgICAgICBkYXRhU2NoZW1hOiBzY2hlbWEuc3RyaW5nKCksXHJcbiAgICAgICAgICBzYW1wbGVEYXRhOiBzY2hlbWEuc3RyaW5nKCksXHJcbiAgICAgICAgfSksXHJcbiAgICAgICAgcXVlcnk6IHNjaGVtYS5vYmplY3Qoe1xyXG4gICAgICAgICAgZGF0YVNvdXJjZUlkOiBzY2hlbWEubWF5YmUoc2NoZW1hLnN0cmluZygpKSxcclxuICAgICAgICB9KSxcclxuICAgICAgfSxcclxuICAgIH0sXHJcbiAgICByb3V0ZXIuaGFuZGxlTGVnYWN5RXJyb3JzKGFzeW5jIChjb250ZXh0LCByZXEsIHJlcykgPT4ge1xyXG4gICAgICBjb25zdCBhc3Npc3RhbnRDbGllbnQgPSBhc3Npc3RhbnRTZXJ2aWNlLmdldFNjb3BlZENsaWVudChyZXEsIGNvbnRleHQpO1xyXG4gICAgICB0cnkge1xyXG4gICAgICAgIGNvbnN0IGFnZW50Q29uZmlnTmFtZSA9IHJlcS5ib2R5LmlucHV0X2luc3RydWN0aW9uXHJcbiAgICAgICAgICA/IFRFWFQyVkVHQV9XSVRIX0lOU1RSVUNUSU9OU19BR0VOVF9DT05GSUdfSURcclxuICAgICAgICAgIDogVEVYVDJWRUdBX1JVTEVfQkFTRURfQUdFTlRfQ09ORklHX0lEO1xyXG4gICAgICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgYXNzaXN0YW50Q2xpZW50LmV4ZWN1dGVBZ2VudEJ5Q29uZmlnTmFtZShhZ2VudENvbmZpZ05hbWUsIHtcclxuICAgICAgICAgIGlucHV0X3F1ZXN0aW9uOiByZXEuYm9keS5pbnB1dF9xdWVzdGlvbixcclxuICAgICAgICAgIGlucHV0X2luc3RydWN0aW9uOiByZXEuYm9keS5pbnB1dF9pbnN0cnVjdGlvbixcclxuICAgICAgICAgIHBwbDogcmVxLmJvZHkucHBsLFxyXG4gICAgICAgICAgZGF0YVNjaGVtYTogcmVxLmJvZHkuZGF0YVNjaGVtYSxcclxuICAgICAgICAgIHNhbXBsZURhdGE6IHJlcS5ib2R5LnNhbXBsZURhdGEsXHJcbiAgICAgICAgfSk7XHJcblxyXG4gICAgICAgIGxldCB0ZXh0Q29udGVudCA9IHJlc3BvbnNlLmJvZHkuaW5mZXJlbmNlX3Jlc3VsdHNbMF0ub3V0cHV0WzBdLnJlc3VsdDtcclxuICAgICAgICAvLyBDaGVjayBpZiB0aGUgdmlzdWFsaXphdGlvbiBpcyBzaW5nbGUgdmFsdWU6XHJcbiAgICAgICAgLy8gaXQgc2hvdWxkIGhhdmUgZXhhY3RseSAxIG1ldHJpYyBhbmQgbm8gZGltZW5zaW9ucy5cclxuICAgICAgICBsZXQgaWZTaW5nbGVNZXRyaWMgPSBjaGVja1NpbmdsZU1ldHJpYyh0ZXh0Q29udGVudCk7XHJcblxyXG4gICAgICAgIC8vIGV4dHJhIGNvbnRlbnQgYmV0d2VlbiB0YWcgPHZlZ2EtbGl0ZT48L3ZlZ2EtbGl0ZT5cclxuICAgICAgICBjb25zdCBzdGFydFRhZyA9ICc8dmVnYS1saXRlPic7XHJcbiAgICAgICAgY29uc3QgZW5kVGFnID0gJzwvdmVnYS1saXRlPic7XHJcblxyXG4gICAgICAgIGNvbnN0IHN0YXJ0SW5kZXggPSB0ZXh0Q29udGVudC5pbmRleE9mKHN0YXJ0VGFnKTtcclxuICAgICAgICBjb25zdCBlbmRJbmRleCA9IHRleHRDb250ZW50LmluZGV4T2YoZW5kVGFnKTtcclxuXHJcbiAgICAgICAgaWYgKHN0YXJ0SW5kZXggIT09IC0xICYmIGVuZEluZGV4ICE9PSAtMSAmJiBzdGFydEluZGV4IDwgZW5kSW5kZXgpIHtcclxuICAgICAgICAgIC8vIEV4dHJhY3QgdGhlIGNvbnRlbnQgYmV0d2VlbiB0aGUgdGFnc1xyXG4gICAgICAgICAgdGV4dENvbnRlbnQgPSB0ZXh0Q29udGVudC5zdWJzdHJpbmcoc3RhcnRJbmRleCArIHN0YXJ0VGFnLmxlbmd0aCwgZW5kSW5kZXgpLnRyaW0oKTtcclxuICAgICAgICB9XHJcbiAgICAgICAgLy8gZXh0cmFjdCBqc29uIG9iamVjdFxyXG4gICAgICAgIGNvbnN0IGpzb25NYXRjaCA9IHRleHRDb250ZW50Lm1hdGNoKC9cXHsuKlxcfS9zKTtcclxuICAgICAgICBpZiAoanNvbk1hdGNoKSB7XHJcbiAgICAgICAgICBsZXQgcmVzdWx0ID0gSlNPTi5wYXJzZShqc29uTWF0Y2hbMF0pO1xyXG4gICAgICAgICAgLy8gc29tZXRpbWVzIGxsbSByZXR1cm5zIHtyZXNwb25zZTogPHNjaGVtYT59IGluc3RlYWQgb2YgPHNjaGVtYT5cclxuICAgICAgICAgIGlmIChyZXN1bHQucmVzcG9uc2UpIHtcclxuICAgICAgICAgICAgcmVzdWx0ID0gSlNPTi5wYXJzZShyZXN1bHQucmVzcG9uc2UpO1xyXG4gICAgICAgICAgfVxyXG4gICAgICAgICAgLy8gU29tZXRpbWVzIHRoZSByZXNwb25zZSBjb250YWlucyB3aWR0aCBhbmQgaGVpZ2h0IHdoaWNoIGlzIG5vdCBuZWVkZWQsIGhlcmUgZGVsZXRlIHRoZSB0aGVzZSBmaWVsZHNcclxuICAgICAgICAgIGRlbGV0ZSByZXN1bHQud2lkdGg7XHJcbiAgICAgICAgICBkZWxldGUgcmVzdWx0LmhlaWdodDtcclxuXHJcbiAgICAgICAgICBpZiAoaWZTaW5nbGVNZXRyaWMpIHtcclxuICAgICAgICAgICAgcmVzdWx0ID0gYWRkVGl0bGVUZXh0TGF5ZXIocmVzdWx0KTtcclxuICAgICAgICAgIH1cclxuICAgICAgICAgIC8vIG1ha2Ugc3VyZSAkc2NoZW1hIGZpZWxkIGFsd2F5cyBiZWVuIGFkZGVkLCBzb21ldGltZXMsIExMTSAnZm9yZ290JyB0byBhZGQgdGhpcyBmaWVsZFxyXG4gICAgICAgICAgcmVzdWx0LiRzY2hlbWEgPSAnaHR0cHM6Ly92ZWdhLmdpdGh1Yi5pby9zY2hlbWEvdmVnYS1saXRlL3Y1Lmpzb24nO1xyXG4gICAgICAgICAgcmV0dXJuIHJlcy5vayh7IGJvZHk6IHJlc3VsdCB9KTtcclxuICAgICAgICB9XHJcbiAgICAgICAgcmV0dXJuIHJlcy5iYWRSZXF1ZXN0KCk7XHJcbiAgICAgIH0gY2F0Y2ggKGUpIHtcclxuICAgICAgICByZXR1cm4gaGFuZGxlRXJyb3IoZSwgcmVzLCBjb250ZXh0LmFzc2lzdGFudF9wbHVnaW4ubG9nZ2VyKTtcclxuICAgICAgfVxyXG4gICAgfSlcclxuICApO1xyXG5cclxuICByb3V0ZXIucG9zdChcclxuICAgIHtcclxuICAgICAgcGF0aDogVEVYVDJWSVpfQVBJLlRFWFQyUFBMLFxyXG4gICAgICB2YWxpZGF0ZToge1xyXG4gICAgICAgIGJvZHk6IHNjaGVtYS5vYmplY3Qoe1xyXG4gICAgICAgICAgaW5kZXg6IHNjaGVtYS5zdHJpbmcoKSxcclxuICAgICAgICAgIHF1ZXN0aW9uOiBpbnB1dFNjaGVtYSxcclxuICAgICAgICB9KSxcclxuICAgICAgICBxdWVyeTogc2NoZW1hLm9iamVjdCh7XHJcbiAgICAgICAgICBkYXRhU291cmNlSWQ6IHNjaGVtYS5tYXliZShzY2hlbWEuc3RyaW5nKCkpLFxyXG4gICAgICAgIH0pLFxyXG4gICAgICB9LFxyXG4gICAgfSxcclxuICAgIHJvdXRlci5oYW5kbGVMZWdhY3lFcnJvcnMoYXN5bmMgKGNvbnRleHQsIHJlcSwgcmVzKSA9PiB7XHJcbiAgICAgIGNvbnN0IGFzc2lzdGFudENsaWVudCA9IGFzc2lzdGFudFNlcnZpY2UuZ2V0U2NvcGVkQ2xpZW50KHJlcSwgY29udGV4dCk7XHJcbiAgICAgIHRyeSB7XHJcbiAgICAgICAgY29uc3QgcmVzcG9uc2UgPSBhd2FpdCBhc3Npc3RhbnRDbGllbnQuZXhlY3V0ZUFnZW50QnlDb25maWdOYW1lKFRFWFQyUFBMX0FHRU5UX0NPTkZJR19JRCwge1xyXG4gICAgICAgICAgcXVlc3Rpb246IHJlcS5ib2R5LnF1ZXN0aW9uLFxyXG4gICAgICAgICAgaW5kZXg6IHJlcS5ib2R5LmluZGV4LFxyXG4gICAgICAgIH0pO1xyXG5cclxuICAgICAgICBjb25zdCByZXN1bHQgPSBKU09OLnBhcnNlKHJlc3BvbnNlLmJvZHkuaW5mZXJlbmNlX3Jlc3VsdHNbMF0ub3V0cHV0WzBdLnJlc3VsdCk7XHJcbiAgICAgICAgcmV0dXJuIHJlcy5vayh7IGJvZHk6IHJlc3VsdCB9KTtcclxuICAgICAgfSBjYXRjaCAoZSkge1xyXG4gICAgICAgIHJldHVybiBoYW5kbGVFcnJvcihlLCByZXMsIGNvbnRleHQuYXNzaXN0YW50X3BsdWdpbi5sb2dnZXIpO1xyXG4gICAgICB9XHJcbiAgICB9KVxyXG4gICk7XHJcbn1cclxuIl0sIm1hcHBpbmdzIjoiOzs7Ozs7QUFLQSxJQUFBQSxhQUFBLEdBQUFDLE9BQUE7QUFFQSxJQUFBQyxJQUFBLEdBQUFELE9BQUE7QUFRQSxJQUFBRSxjQUFBLEdBQUFGLE9BQUE7QUFDQSxJQUFBRyxvQkFBQSxHQUFBSCxPQUFBO0FBaEJBO0FBQ0E7QUFDQTtBQUNBOztBQWVBLE1BQU1JLFdBQVcsR0FBR0Msb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDO0VBQ2hDQyxTQUFTLEVBQUVDLCtCQUEwQjtFQUNyQ0MsUUFBUUEsQ0FBQ0MsS0FBSyxFQUFFO0lBQ2QsSUFBSSxDQUFDQSxLQUFLLElBQUlBLEtBQUssQ0FBQ0MsSUFBSSxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxLQUFLLENBQUMsRUFBRTtNQUN2QyxPQUFPLDBCQUEwQjtJQUNuQztFQUNGO0FBQ0YsQ0FBQyxDQUFDO0FBRUssU0FBU0Msc0JBQXNCQSxDQUFDQyxNQUFlLEVBQUVDLGdCQUF1QyxFQUFFO0VBQy9GRCxNQUFNLENBQUNFLElBQUksQ0FDVDtJQUNFQyxJQUFJLEVBQUVDLGlCQUFZLENBQUNDLFNBQVM7SUFDNUJWLFFBQVEsRUFBRTtNQUNSVyxJQUFJLEVBQUVmLG9CQUFNLENBQUNnQixNQUFNLENBQUM7UUFDbEJDLGNBQWMsRUFBRWxCLFdBQVc7UUFDM0JtQixpQkFBaUIsRUFBRWxCLG9CQUFNLENBQUNtQixLQUFLLENBQUNuQixvQkFBTSxDQUFDQyxNQUFNLENBQUM7VUFBRUMsU0FBUyxFQUFFQztRQUEyQixDQUFDLENBQUMsQ0FBQztRQUN6RmlCLEdBQUcsRUFBRXBCLG9CQUFNLENBQUNDLE1BQU0sQ0FBQyxDQUFDO1FBQ3BCb0IsVUFBVSxFQUFFckIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDLENBQUM7UUFDM0JxQixVQUFVLEVBQUV0QixvQkFBTSxDQUFDQyxNQUFNLENBQUM7TUFDNUIsQ0FBQyxDQUFDO01BQ0ZzQixLQUFLLEVBQUV2QixvQkFBTSxDQUFDZ0IsTUFBTSxDQUFDO1FBQ25CUSxZQUFZLEVBQUV4QixvQkFBTSxDQUFDbUIsS0FBSyxDQUFDbkIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDLENBQUM7TUFDNUMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNEUSxNQUFNLENBQUNnQixrQkFBa0IsQ0FBQyxPQUFPQyxPQUFPLEVBQUVDLEdBQUcsRUFBRUMsR0FBRyxLQUFLO0lBQ3JELE1BQU1DLGVBQWUsR0FBR25CLGdCQUFnQixDQUFDb0IsZUFBZSxDQUFDSCxHQUFHLEVBQUVELE9BQU8sQ0FBQztJQUN0RSxJQUFJO01BQ0YsTUFBTUssZUFBZSxHQUFHSixHQUFHLENBQUNaLElBQUksQ0FBQ0csaUJBQWlCLEdBQzlDYyxnREFBMkMsR0FDM0NDLHlDQUFvQztNQUN4QyxNQUFNQyxRQUFRLEdBQUcsTUFBTUwsZUFBZSxDQUFDTSx3QkFBd0IsQ0FBQ0osZUFBZSxFQUFFO1FBQy9FZCxjQUFjLEVBQUVVLEdBQUcsQ0FBQ1osSUFBSSxDQUFDRSxjQUFjO1FBQ3ZDQyxpQkFBaUIsRUFBRVMsR0FBRyxDQUFDWixJQUFJLENBQUNHLGlCQUFpQjtRQUM3Q0UsR0FBRyxFQUFFTyxHQUFHLENBQUNaLElBQUksQ0FBQ0ssR0FBRztRQUNqQkMsVUFBVSxFQUFFTSxHQUFHLENBQUNaLElBQUksQ0FBQ00sVUFBVTtRQUMvQkMsVUFBVSxFQUFFSyxHQUFHLENBQUNaLElBQUksQ0FBQ087TUFDdkIsQ0FBQyxDQUFDO01BRUYsSUFBSWMsV0FBVyxHQUFHRixRQUFRLENBQUNuQixJQUFJLENBQUNzQixpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDQyxNQUFNO01BQ3JFO01BQ0E7TUFDQSxJQUFJQyxjQUFjLEdBQUcsSUFBQUMsc0NBQWlCLEVBQUNMLFdBQVcsQ0FBQzs7TUFFbkQ7TUFDQSxNQUFNTSxRQUFRLEdBQUcsYUFBYTtNQUM5QixNQUFNQyxNQUFNLEdBQUcsY0FBYztNQUU3QixNQUFNQyxVQUFVLEdBQUdSLFdBQVcsQ0FBQ1MsT0FBTyxDQUFDSCxRQUFRLENBQUM7TUFDaEQsTUFBTUksUUFBUSxHQUFHVixXQUFXLENBQUNTLE9BQU8sQ0FBQ0YsTUFBTSxDQUFDO01BRTVDLElBQUlDLFVBQVUsS0FBSyxDQUFDLENBQUMsSUFBSUUsUUFBUSxLQUFLLENBQUMsQ0FBQyxJQUFJRixVQUFVLEdBQUdFLFFBQVEsRUFBRTtRQUNqRTtRQUNBVixXQUFXLEdBQUdBLFdBQVcsQ0FBQ1csU0FBUyxDQUFDSCxVQUFVLEdBQUdGLFFBQVEsQ0FBQ25DLE1BQU0sRUFBRXVDLFFBQVEsQ0FBQyxDQUFDeEMsSUFBSSxDQUFDLENBQUM7TUFDcEY7TUFDQTtNQUNBLE1BQU0wQyxTQUFTLEdBQUdaLFdBQVcsQ0FBQ2EsS0FBSyxDQUFDLFNBQVMsQ0FBQztNQUM5QyxJQUFJRCxTQUFTLEVBQUU7UUFDYixJQUFJVCxNQUFNLEdBQUdXLElBQUksQ0FBQ0MsS0FBSyxDQUFDSCxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDckM7UUFDQSxJQUFJVCxNQUFNLENBQUNMLFFBQVEsRUFBRTtVQUNuQkssTUFBTSxHQUFHVyxJQUFJLENBQUNDLEtBQUssQ0FBQ1osTUFBTSxDQUFDTCxRQUFRLENBQUM7UUFDdEM7UUFDQTtRQUNBLE9BQU9LLE1BQU0sQ0FBQ2EsS0FBSztRQUNuQixPQUFPYixNQUFNLENBQUNjLE1BQU07UUFFcEIsSUFBSWIsY0FBYyxFQUFFO1VBQ2xCRCxNQUFNLEdBQUcsSUFBQWUsc0NBQWlCLEVBQUNmLE1BQU0sQ0FBQztRQUNwQztRQUNBO1FBQ0FBLE1BQU0sQ0FBQ2dCLE9BQU8sR0FBRyxpREFBaUQ7UUFDbEUsT0FBTzNCLEdBQUcsQ0FBQzRCLEVBQUUsQ0FBQztVQUFFekMsSUFBSSxFQUFFd0I7UUFBTyxDQUFDLENBQUM7TUFDakM7TUFDQSxPQUFPWCxHQUFHLENBQUM2QixVQUFVLENBQUMsQ0FBQztJQUN6QixDQUFDLENBQUMsT0FBT0MsQ0FBQyxFQUFFO01BQ1YsT0FBTyxJQUFBQywwQkFBVyxFQUFDRCxDQUFDLEVBQUU5QixHQUFHLEVBQUVGLE9BQU8sQ0FBQ2tDLGdCQUFnQixDQUFDQyxNQUFNLENBQUM7SUFDN0Q7RUFDRixDQUFDLENBQ0gsQ0FBQztFQUVEcEQsTUFBTSxDQUFDRSxJQUFJLENBQ1Q7SUFDRUMsSUFBSSxFQUFFQyxpQkFBWSxDQUFDaUQsUUFBUTtJQUMzQjFELFFBQVEsRUFBRTtNQUNSVyxJQUFJLEVBQUVmLG9CQUFNLENBQUNnQixNQUFNLENBQUM7UUFDbEIrQyxLQUFLLEVBQUUvRCxvQkFBTSxDQUFDQyxNQUFNLENBQUMsQ0FBQztRQUN0QitELFFBQVEsRUFBRWpFO01BQ1osQ0FBQyxDQUFDO01BQ0Z3QixLQUFLLEVBQUV2QixvQkFBTSxDQUFDZ0IsTUFBTSxDQUFDO1FBQ25CUSxZQUFZLEVBQUV4QixvQkFBTSxDQUFDbUIsS0FBSyxDQUFDbkIsb0JBQU0sQ0FBQ0MsTUFBTSxDQUFDLENBQUM7TUFDNUMsQ0FBQztJQUNIO0VBQ0YsQ0FBQyxFQUNEUSxNQUFNLENBQUNnQixrQkFBa0IsQ0FBQyxPQUFPQyxPQUFPLEVBQUVDLEdBQUcsRUFBRUMsR0FBRyxLQUFLO0lBQ3JELE1BQU1DLGVBQWUsR0FBR25CLGdCQUFnQixDQUFDb0IsZUFBZSxDQUFDSCxHQUFHLEVBQUVELE9BQU8sQ0FBQztJQUN0RSxJQUFJO01BQ0YsTUFBTVEsUUFBUSxHQUFHLE1BQU1MLGVBQWUsQ0FBQ00sd0JBQXdCLENBQUM4Qiw2QkFBd0IsRUFBRTtRQUN4RkQsUUFBUSxFQUFFckMsR0FBRyxDQUFDWixJQUFJLENBQUNpRCxRQUFRO1FBQzNCRCxLQUFLLEVBQUVwQyxHQUFHLENBQUNaLElBQUksQ0FBQ2dEO01BQ2xCLENBQUMsQ0FBQztNQUVGLE1BQU14QixNQUFNLEdBQUdXLElBQUksQ0FBQ0MsS0FBSyxDQUFDakIsUUFBUSxDQUFDbkIsSUFBSSxDQUFDc0IsaUJBQWlCLENBQUMsQ0FBQyxDQUFDLENBQUNDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQ0MsTUFBTSxDQUFDO01BQzlFLE9BQU9YLEdBQUcsQ0FBQzRCLEVBQUUsQ0FBQztRQUFFekMsSUFBSSxFQUFFd0I7TUFBTyxDQUFDLENBQUM7SUFDakMsQ0FBQyxDQUFDLE9BQU9tQixDQUFDLEVBQUU7TUFDVixPQUFPLElBQUFDLDBCQUFXLEVBQUNELENBQUMsRUFBRTlCLEdBQUcsRUFBRUYsT0FBTyxDQUFDa0MsZ0JBQWdCLENBQUNDLE1BQU0sQ0FBQztJQUM3RDtFQUNGLENBQUMsQ0FDSCxDQUFDO0FBQ0gifQ==