"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.postProcessing = postProcessing;
exports.registerData2SummaryRoutes = registerData2SummaryRoutes;
exports.registerSummaryAssistantRoutes = registerSummaryAssistantRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
var _get_agent = require("./get_agent");
var _index_type_detect = require("./index_type_detect");
var _error_handler = require("./error_handler");
var _errors = require("./errors");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const SUMMARY_AGENT_CONFIG_ID = 'os_summary';
const LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID = 'os_summary_with_log_pattern';
const DATA2SUMMARY_AGENT_CONFIG_ID = 'os_data2summary';
const LOG_PATTERN_DATA2SUMMARY_AGENT_CONFIG_ID = 'os_data2summary_with_log_pattern';
function postProcessing(output) {
  const pattern = /<summarization>(.*?)<\/summarization>.*?<final insights>(.*?)<\/final insights>/s;
  const match = output.match(pattern);
  if (match) {
    const [, summarization, finalInsights] = match;
    const processedOutput = `${summarization.trim()}\n${finalInsights.trim()}`;
    return processedOutput;
  }
  return output;
}
function registerSummaryAssistantRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.SUMMARIZE,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.maybe(_configSchema.schema.string()),
        question: _configSchema.schema.string(),
        context: _configSchema.schema.maybe(_configSchema.schema.string()),
        index: _configSchema.schema.maybe(_configSchema.schema.string()),
        dsl: _configSchema.schema.maybe(_configSchema.schema.string()),
        topNLogPatternData: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    try {
      var _response$body$infere;
      const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
        context,
        dataSourceId: req.query.dataSourceId
      });
      const assistantClient = assistantService.getScopedClient(req, context);
      let isLogIndex = false;
      if (req.body.index) {
        isLogIndex = await (0, _index_type_detect.detectIndexType)(client, assistantClient, req.body.index, req.query.dataSourceId);
      }
      const agentConfigId = req.body.index && req.body.dsl && isLogIndex ? LOG_PATTERN_SUMMARY_AGENT_CONFIG_ID : SUMMARY_AGENT_CONFIG_ID;
      const response = await assistantClient.executeAgentByConfigName(agentConfigId, {
        context: req.body.context,
        question: req.body.question,
        index: req.body.index,
        input: req.body.dsl,
        topNLogPatternData: req.body.topNLogPatternData
      });
      let insightAgentIdExists = false;
      if (req.body.insightType) {
        insightAgentIdExists = !!(await detectInsightAgentId(req.body.insightType, req.body.summaryType, client));
      }
      const summary = (_response$body$infere = response.body.inference_results[0]) === null || _response$body$infere === void 0 || (_response$body$infere = _response$body$infere.output[0]) === null || _response$body$infere === void 0 ? void 0 : _response$body$infere.result;
      if (!summary) {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
      return res.ok({
        body: {
          summary,
          insightAgentIdExists
        }
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.INSIGHT,
    validate: {
      body: _configSchema.schema.object({
        summaryType: _configSchema.schema.string(),
        insightType: _configSchema.schema.string(),
        summary: _configSchema.schema.string(),
        context: _configSchema.schema.string(),
        question: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    try {
      var _response$body$infere2;
      const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
        context,
        dataSourceId: req.query.dataSourceId
      });
      const assistantClient = assistantService.getScopedClient(req, context);
      const insightAgentId = await detectInsightAgentId(req.body.insightType, req.body.summaryType, client);
      if (!insightAgentId) {
        return res.notFound({
          body: 'Agent not found'
        });
      }
      const response = await assistantClient.executeAgent(insightAgentId, {
        context: req.body.context,
        summary: req.body.summary,
        question: req.body.question
      });
      const insight = (_response$body$infere2 = response.body.inference_results[0]) === null || _response$body$infere2 === void 0 || (_response$body$infere2 = _response$body$infere2.output[0]) === null || _response$body$infere2 === void 0 ? void 0 : _response$body$infere2.result;
      if (!insight) {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
      return res.ok({
        body: insight
      });
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
async function detectInsightAgentId(insightType, summaryType, client) {
  // for user_insight, we can
  // only get it by searching on name since it is not stored in agent config.
  try {
    if (insightType === 'user_insight' && summaryType === 'alerts') {
      return await (0, _get_agent.searchAgent)({
        name: 'KB_For_Alert_Insight'
      }, client);
    }
  } catch (e) {
    // It only detects if the agent exists, we don't want to throw the error when not found the agent
    // So we return `undefined` to indicate the insight agent id not found
    if (e instanceof _errors.AgentNotFoundError) {
      return undefined;
    }
    throw e;
  }
}
function registerData2SummaryRoutes(router, assistantService) {
  router.post({
    path: _llm.SUMMARY_ASSISTANT_API.DATA2SUMMARY,
    validate: {
      body: _configSchema.schema.object({
        sample_data: _configSchema.schema.string(),
        sample_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        total_count: _configSchema.schema.maybe(_configSchema.schema.number()),
        question: _configSchema.schema.maybe(_configSchema.schema.string()),
        ppl: _configSchema.schema.maybe(_configSchema.schema.string()),
        index: _configSchema.schema.maybe(_configSchema.schema.string())
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, router.handleLegacyErrors(async (context, req, res) => {
    const assistantClient = assistantService.getScopedClient(req, context);
    try {
      const client = await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
        context,
        dataSourceId: req.query.dataSourceId
      });
      let agentConfigId = DATA2SUMMARY_AGENT_CONFIG_ID;
      if (req.body.index) {
        const isLogIndex = await (0, _index_type_detect.detectIndexType)(client, assistantClient, req.body.index, req.query.dataSourceId);
        if (isLogIndex) {
          const isAgentExist = await detectAgentIdExist(assistantClient, LOG_PATTERN_DATA2SUMMARY_AGENT_CONFIG_ID);
          agentConfigId = isAgentExist ? LOG_PATTERN_DATA2SUMMARY_AGENT_CONFIG_ID : agentConfigId;
        }
      }
      const response = await assistantClient.executeAgentByConfigName(agentConfigId, {
        sample_data: req.body.sample_data,
        total_count: req.body.total_count,
        sample_count: req.body.sample_count,
        ppl: req.body.ppl,
        question: req.body.question
      });
      let result = response.body.inference_results[0].output[0].result;
      result = postProcessing(result);
      if (result) {
        return res.ok({
          body: result
        });
      } else {
        return res.customError({
          body: 'Execute agent failed with empty response!',
          statusCode: 500
        });
      }
    } catch (e) {
      return (0, _error_handler.handleError)(e, res, context.assistant_plugin.logger);
    }
  }));
}
async function detectAgentIdExist(assistantClient, configName) {
  try {
    const result = await assistantClient.getAgentIdByConfigName(configName);
    return Boolean(result);
  } catch (e) {
    return false;
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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