"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerChatRoutes = registerChatRoutes;
var _configSchema = require("@osd/config-schema");
var _llm = require("../../common/constants/llm");
var _olly_chat_service = require("../services/chat/olly_chat_service");
var _agent_framework_storage_service = require("../services/storage/agent_framework_storage_service");
var _get_opensearch_client_transport = require("../utils/get_opensearch_client_transport");
var _error_handler = require("./error_handler");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const llmRequestRoute = {
  path: _llm.ASSISTANT_API.SEND_MESSAGE,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.maybe(_configSchema.schema.string()),
      messages: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
      input: _configSchema.schema.object({
        type: _configSchema.schema.literal('input'),
        context: _configSchema.schema.object({
          appId: _configSchema.schema.maybe(_configSchema.schema.string()),
          content: _configSchema.schema.maybe(_configSchema.schema.string()),
          datasourceId: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        content: _configSchema.schema.string(),
        contentType: _configSchema.schema.literal('text'),
        promptPrefix: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string()),
      nextToken: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const abortAgentExecutionRoute = {
  path: `${_llm.ASSISTANT_API.ABORT_AGENT_EXECUTION}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const regenerateRoute = {
  path: `${_llm.ASSISTANT_API.REGENERATE}`,
  validate: {
    body: _configSchema.schema.object({
      conversationId: _configSchema.schema.string(),
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getConversationsRoute = {
  path: _llm.ASSISTANT_API.CONVERSATIONS,
  validate: {
    query: _configSchema.schema.object({
      perPage: _configSchema.schema.number({
        min: 0,
        defaultValue: 20
      }),
      page: _configSchema.schema.number({
        min: 0,
        defaultValue: 1
      }),
      sortOrder: _configSchema.schema.maybe(_configSchema.schema.string()),
      sortField: _configSchema.schema.maybe(_configSchema.schema.string()),
      fields: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
      search: _configSchema.schema.maybe(_configSchema.schema.string()),
      searchFields: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.arrayOf(_configSchema.schema.string())])),
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const deleteConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const updateConversationRoute = {
  path: `${_llm.ASSISTANT_API.CONVERSATION}/{conversationId}`,
  validate: {
    params: _configSchema.schema.object({
      conversationId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      title: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const getTracesRoute = {
  path: `${_llm.ASSISTANT_API.TRACE}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const feedbackRoute = {
  path: `${_llm.ASSISTANT_API.FEEDBACK}/{interactionId}`,
  validate: {
    params: _configSchema.schema.object({
      interactionId: _configSchema.schema.string()
    }),
    body: _configSchema.schema.object({
      satisfaction: _configSchema.schema.boolean()
    }),
    query: _configSchema.schema.object({
      dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
    })
  }
};
const accountRoute = {
  path: `${_llm.ASSISTANT_API.ACCOUNT}`,
  validate: {}
};
function registerChatRoutes(router, routeOptions) {
  const createStorageService = async (context, dataSourceId) => new _agent_framework_storage_service.AgentFrameworkStorageService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }), routeOptions.messageParsers);
  const createChatService = async (context, dataSourceId) => new _olly_chat_service.OllyChatService(await (0, _get_opensearch_client_transport.getOpenSearchClientTransport)({
    context,
    dataSourceId
  }));
  router.post(llmRequestRoute, async (context, request, response) => {
    var _outputs, _outputs2;
    const {
      messages = [],
      input,
      conversationId: conversationIdInRequestBody
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.requestLLM({
        messages,
        input,
        conversationId: conversationIdInRequestBody
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
    if (outputs.stream) {
      const result = response.ok({
        headers: {
          // Browsers often need to buffer the entire response before decompressing, which defeats the purpose of streaming.
          // need to set 'Content-Encoding' as 'identity' here to prevent browser buffering the response.
          'Content-Encoding': 'identity',
          Connection: 'keep-alive',
          'Content-Type': 'text/event-stream'
        },
        body: outputs.stream
      });
      return result;
    }

    /**
     * Retrieve latest interactions from memory
     */
    const conversationId = ((_outputs = outputs) === null || _outputs === void 0 ? void 0 : _outputs.conversationId) || conversationIdInRequestBody;
    const interactionId = ((_outputs2 = outputs) === null || _outputs2 === void 0 ? void 0 : _outputs2.interactionId) || '';
    try {
      if (!conversationId) {
        throw new Error('Not a valid conversation');
      }
      const resultPayload = {
        messages: [],
        interactions: [],
        conversationId
      };
      if (!conversationIdInRequestBody) {
        /**
         * If no conversationId is provided in request payload,
         * it means it is a brand new conversation,
         * need to fetch all the details including title.
         */
        const conversation = await storageService.getConversation(conversationId);
        resultPayload.interactions = conversation.interactions;
        resultPayload.messages = conversation.messages;
        resultPayload.title = conversation.title;
      } else {
        /**
         * Only response with the latest interaction.
         * It may have some issues in Concurrent case like a user may use two tabs to chat with Chatbot in one conversation.
         * But for now we will ignore this case, can be optimized by always fetching conversation if we need to take this case into consideration.
         */
        const interaction = await storageService.getInteraction(conversationId, interactionId);
        resultPayload.interactions = [interaction].filter(item => item);
        resultPayload.messages = resultPayload.interactions.length ? await storageService.getMessagesFromInteractions(resultPayload.interactions) : [];
      }
      resultPayload.messages.filter(message => message.type === 'input').forEach(msg => {
        // hide additional conetxt to how was it generated
        const index = msg.content.indexOf('answer question:');
        const len = 'answer question:'.length;
        if (index !== -1) {
          msg.content = msg.content.substring(index + len);
        }
      });
      return response.ok({
        body: resultPayload
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getConversationsRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getConversations(request.query);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      var _error$meta;
      // Return empty result for 404 errors
      if ((error === null || error === void 0 || (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.statusCode) === 404) {
        return response.ok({
          body: {
            objects: [],
            total: 0
          }
        });
      }
      context.assistant_plugin.logger.error(error);
      return (0, _error_handler.handleError)(error, response, context.assistant_plugin.logger);
    }
  });
  router.delete(deleteConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.deleteConversation(request.params.conversationId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(updateConversationRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.updateConversation(request.params.conversationId, request.body.title);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(getTracesRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    try {
      const getResponse = await storageService.getTraces(request.params.interactionId);
      return response.ok({
        body: getResponse
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post(abortAgentExecutionRoute, async (context, request, response) => {
    const chatService = await createChatService(context, request.query.dataSourceId);
    try {
      chatService.abortAgentExecution(request.body.conversationId);
      context.assistant_plugin.logger.info(`Abort agent execution: ${request.body.conversationId}`);
      return response.ok();
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(regenerateRoute, async (context, request, response) => {
    const {
      conversationId,
      interactionId
    } = request.body;
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const chatService = await createChatService(context, request.query.dataSourceId);
    let outputs;

    /**
     * Get final answer from Agent framework
     */
    try {
      outputs = await chatService.regenerate({
        conversationId,
        interactionId
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }

    /**
     * Retrieve latest interactions from memory
     */
    try {
      var _outputs3;
      const interaction = await storageService.getInteraction(conversationId, ((_outputs3 = outputs) === null || _outputs3 === void 0 ? void 0 : _outputs3.interactionId) || '');
      const finalInteractions = [interaction].filter(item => item);
      const messages = finalInteractions.length ? await storageService.getMessagesFromInteractions(finalInteractions) : [];
      return response.ok({
        body: {
          interactions: finalInteractions,
          messages,
          conversationId
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.put(feedbackRoute, async (context, request, response) => {
    const storageService = await createStorageService(context, request.query.dataSourceId);
    const {
      interactionId
    } = request.params;
    try {
      const updateResponse = await storageService.updateInteraction(interactionId, {
        feedback: request.body
      });
      return response.ok({
        body: {
          ...updateResponse,
          success: true
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.get(accountRoute, async (context, request, response) => {
    try {
      var _auth$state$authInfo$, _auth$state;
      const auth = routeOptions.auth.get(request);
      return response.ok({
        body: {
          user_name: (_auth$state$authInfo$ = auth === null || auth === void 0 || (_auth$state = auth.state) === null || _auth$state === void 0 || (_auth$state = _auth$state.authInfo) === null || _auth$state === void 0 ? void 0 : _auth$state.user_name) !== null && _auth$state$authInfo$ !== void 0 ? _auth$state$authInfo$ : _llm.DEFAULT_USER_NAME
        }
      });
    } catch (error) {
      context.assistant_plugin.logger.error(error);
      return response.ok({
        body: {
          user_name: _llm.DEFAULT_USER_NAME
        }
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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