"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AssistantPlugin = void 0;
var _operators = require("rxjs/operators");
var _basic_input_output_parser = require("./parsers/basic_input_output_parser");
var _visualization_card_parser = require("./parsers/visualization_card_parser");
var _chat_routes = require("./routes/chat_routes");
var _text2viz_routes = require("./routes/text2viz_routes");
var _assistant_service = require("./services/assistant_service");
var _agent_routes = require("./routes/agent_routes");
var _summary_routes = require("./routes/summary_routes");
var _capabilities_provider = require("./vis_type_nlq/capabilities_provider");
var _saved_object_type = require("./vis_type_nlq/saved_object_type");
var _capabilities = require("./capabilities");
var _constants = require("./utils/constants");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
class AssistantPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "messageParsers", []);
    _defineProperty(this, "assistantService", new _assistant_service.AssistantService());
    this.logger = initializerContext.logger.get();
  }
  async setup(core) {
    this.logger.debug('Assistant: Setup');
    const config = await this.initializerContext.config.create().pipe((0, _operators.first)()).toPromise();
    const assistantServiceSetup = this.assistantService.setup();
    const router = core.http.createRouter();
    core.http.registerRouteHandlerContext('assistant_plugin', () => {
      return {
        config,
        logger: this.logger
      };
    });
    (0, _agent_routes.registerAgentRoutes)(router, assistantServiceSetup);

    // Register server side APIs
    (0, _chat_routes.registerChatRoutes)(router, {
      messageParsers: this.messageParsers,
      auth: core.http.auth
    });

    // Register router for text to visualization
    if (config.text2viz.enabled) {
      (0, _text2viz_routes.registerText2VizRoutes)(router, assistantServiceSetup);
      core.capabilities.registerProvider(_capabilities_provider.capabilitiesProvider);
      core.savedObjects.registerType(_saved_object_type.visNLQSavedObjectType);
    }

    // Register router for alert insight
    if (config.alertInsight.enabled) {
      (0, _summary_routes.registerSummaryAssistantRoutes)(router, assistantServiceSetup);
    }
    core.capabilities.registerProvider(_capabilities.capabilitiesProvider);
    // register UI capabilities from dynamic config service
    core.capabilities.registerSwitcher(async opensearchDashboardsRequest => {
      const dynamicConfigServiceStart = await core.dynamicConfigService.getStartService();
      const store = dynamicConfigServiceStart.getAsyncLocalStore();
      const client = dynamicConfigServiceStart.getClient();
      try {
        const dynamicConfig = await client.getConfig({
          pluginConfigPath: 'assistant'
        }, {
          asyncLocalStorageContext: store
        });
        const [coreStart] = await core.getStartServices();
        const savedObjectsClient = coreStart.savedObjects.getScopedClient(opensearchDashboardsRequest);
        const uiSettingsClient = coreStart.uiSettings.asScopedToClient(savedObjectsClient);
        const isAssistantEnabledBySetting = Boolean(await uiSettingsClient.get(_constants.ENABLE_AI_FEATURES).catch(() => false));
        return {
          assistant: {
            enabled: dynamicConfig.enabled && isAssistantEnabledBySetting,
            chatEnabled: dynamicConfig.chat.enabled && isAssistantEnabledBySetting
          }
        };
      } catch (e) {
        this.logger.error(e);
        return {};
      }
    });

    // Register router for discovery summary
    (0, _summary_routes.registerData2SummaryRoutes)(router, assistantServiceSetup);
    const registerMessageParser = messageParser => {
      const findItem = this.messageParsers.find(item => item.id === messageParser.id);
      if (findItem) {
        throw new Error(`There is already a messageParser whose id is ${messageParser.id}`);
      }
      this.messageParsers.push(messageParser);
    };
    registerMessageParser(_basic_input_output_parser.BasicInputOutputParser);
    registerMessageParser(_visualization_card_parser.VisualizationCardParser);
    return {
      assistantService: assistantServiceSetup,
      registerMessageParser,
      removeMessageParser: parserId => {
        const findIndex = this.messageParsers.findIndex(item => item.id === parserId);
        if (findIndex < 0) {
          this.logger.error(`There is not a messageParser whose id is ${parserId}`);
        }
        this.messageParsers.splice(findIndex, 1);
      }
    };
  }
  start(core) {
    this.logger.debug('Assistant: Started');
    this.assistantService.start();
    return {};
  }
  stop() {
    this.assistantService.stop();
  }
}
exports.AssistantPlugin = AssistantPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJuYW1lcyI6WyJfb3BlcmF0b3JzIiwicmVxdWlyZSIsIl9iYXNpY19pbnB1dF9vdXRwdXRfcGFyc2VyIiwiX3Zpc3VhbGl6YXRpb25fY2FyZF9wYXJzZXIiLCJfY2hhdF9yb3V0ZXMiLCJfdGV4dDJ2aXpfcm91dGVzIiwiX2Fzc2lzdGFudF9zZXJ2aWNlIiwiX2FnZW50X3JvdXRlcyIsIl9zdW1tYXJ5X3JvdXRlcyIsIl9jYXBhYmlsaXRpZXNfcHJvdmlkZXIiLCJfc2F2ZWRfb2JqZWN0X3R5cGUiLCJfY2FwYWJpbGl0aWVzIiwiX2NvbnN0YW50cyIsIl9kZWZpbmVQcm9wZXJ0eSIsImUiLCJyIiwidCIsIl90b1Byb3BlcnR5S2V5IiwiT2JqZWN0IiwiZGVmaW5lUHJvcGVydHkiLCJ2YWx1ZSIsImVudW1lcmFibGUiLCJjb25maWd1cmFibGUiLCJ3cml0YWJsZSIsImkiLCJfdG9QcmltaXRpdmUiLCJTeW1ib2wiLCJ0b1ByaW1pdGl2ZSIsImNhbGwiLCJUeXBlRXJyb3IiLCJTdHJpbmciLCJOdW1iZXIiLCJBc3Npc3RhbnRQbHVnaW4iLCJjb25zdHJ1Y3RvciIsImluaXRpYWxpemVyQ29udGV4dCIsIkFzc2lzdGFudFNlcnZpY2UiLCJsb2dnZXIiLCJnZXQiLCJzZXR1cCIsImNvcmUiLCJkZWJ1ZyIsImNvbmZpZyIsImNyZWF0ZSIsInBpcGUiLCJmaXJzdCIsInRvUHJvbWlzZSIsImFzc2lzdGFudFNlcnZpY2VTZXR1cCIsImFzc2lzdGFudFNlcnZpY2UiLCJyb3V0ZXIiLCJodHRwIiwiY3JlYXRlUm91dGVyIiwicmVnaXN0ZXJSb3V0ZUhhbmRsZXJDb250ZXh0IiwicmVnaXN0ZXJBZ2VudFJvdXRlcyIsInJlZ2lzdGVyQ2hhdFJvdXRlcyIsIm1lc3NhZ2VQYXJzZXJzIiwiYXV0aCIsInRleHQydml6IiwiZW5hYmxlZCIsInJlZ2lzdGVyVGV4dDJWaXpSb3V0ZXMiLCJjYXBhYmlsaXRpZXMiLCJyZWdpc3RlclByb3ZpZGVyIiwidmlzTkxRQ2FwYWJpbGl0aWVzUHJvdmlkZXIiLCJzYXZlZE9iamVjdHMiLCJyZWdpc3RlclR5cGUiLCJ2aXNOTFFTYXZlZE9iamVjdFR5cGUiLCJhbGVydEluc2lnaHQiLCJyZWdpc3RlclN1bW1hcnlBc3Npc3RhbnRSb3V0ZXMiLCJjYXBhYmlsaXRpZXNQcm92aWRlciIsInJlZ2lzdGVyU3dpdGNoZXIiLCJvcGVuc2VhcmNoRGFzaGJvYXJkc1JlcXVlc3QiLCJkeW5hbWljQ29uZmlnU2VydmljZVN0YXJ0IiwiZHluYW1pY0NvbmZpZ1NlcnZpY2UiLCJnZXRTdGFydFNlcnZpY2UiLCJzdG9yZSIsImdldEFzeW5jTG9jYWxTdG9yZSIsImNsaWVudCIsImdldENsaWVudCIsImR5bmFtaWNDb25maWciLCJnZXRDb25maWciLCJwbHVnaW5Db25maWdQYXRoIiwiYXN5bmNMb2NhbFN0b3JhZ2VDb250ZXh0IiwiY29yZVN0YXJ0IiwiZ2V0U3RhcnRTZXJ2aWNlcyIsInNhdmVkT2JqZWN0c0NsaWVudCIsImdldFNjb3BlZENsaWVudCIsInVpU2V0dGluZ3NDbGllbnQiLCJ1aVNldHRpbmdzIiwiYXNTY29wZWRUb0NsaWVudCIsImlzQXNzaXN0YW50RW5hYmxlZEJ5U2V0dGluZyIsIkJvb2xlYW4iLCJFTkFCTEVfQUlfRkVBVFVSRVMiLCJjYXRjaCIsImFzc2lzdGFudCIsImNoYXRFbmFibGVkIiwiY2hhdCIsImVycm9yIiwicmVnaXN0ZXJEYXRhMlN1bW1hcnlSb3V0ZXMiLCJyZWdpc3Rlck1lc3NhZ2VQYXJzZXIiLCJtZXNzYWdlUGFyc2VyIiwiZmluZEl0ZW0iLCJmaW5kIiwiaXRlbSIsImlkIiwiRXJyb3IiLCJwdXNoIiwiQmFzaWNJbnB1dE91dHB1dFBhcnNlciIsIlZpc3VhbGl6YXRpb25DYXJkUGFyc2VyIiwicmVtb3ZlTWVzc2FnZVBhcnNlciIsInBhcnNlcklkIiwiZmluZEluZGV4Iiwic3BsaWNlIiwic3RhcnQiLCJzdG9wIiwiZXhwb3J0cyJdLCJzb3VyY2VzIjpbInBsdWdpbi50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvKlxyXG4gKiBDb3B5cmlnaHQgT3BlblNlYXJjaCBDb250cmlidXRvcnNcclxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcclxuICovXHJcblxyXG5pbXBvcnQgeyBmaXJzdCB9IGZyb20gJ3J4anMvb3BlcmF0b3JzJztcclxuaW1wb3J0IHsgQ29uZmlnU2NoZW1hIH0gZnJvbSAnLi4vY29tbW9uL3R5cGVzL2NvbmZpZyc7XHJcbmltcG9ydCB7XHJcbiAgQ29yZVNldHVwLFxyXG4gIENvcmVTdGFydCxcclxuICBMb2dnZXIsXHJcbiAgT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0LFxyXG4gIFBsdWdpbixcclxuICBQbHVnaW5Jbml0aWFsaXplckNvbnRleHQsXHJcbn0gZnJvbSAnLi4vLi4vLi4vc3JjL2NvcmUvc2VydmVyJztcclxuaW1wb3J0IHsgQXNzaXN0YW50UGx1Z2luU2V0dXAsIEFzc2lzdGFudFBsdWdpblN0YXJ0LCBNZXNzYWdlUGFyc2VyIH0gZnJvbSAnLi90eXBlcyc7XHJcbmltcG9ydCB7IEJhc2ljSW5wdXRPdXRwdXRQYXJzZXIgfSBmcm9tICcuL3BhcnNlcnMvYmFzaWNfaW5wdXRfb3V0cHV0X3BhcnNlcic7XHJcbmltcG9ydCB7IFZpc3VhbGl6YXRpb25DYXJkUGFyc2VyIH0gZnJvbSAnLi9wYXJzZXJzL3Zpc3VhbGl6YXRpb25fY2FyZF9wYXJzZXInO1xyXG5pbXBvcnQgeyByZWdpc3RlckNoYXRSb3V0ZXMgfSBmcm9tICcuL3JvdXRlcy9jaGF0X3JvdXRlcyc7XHJcbmltcG9ydCB7IHJlZ2lzdGVyVGV4dDJWaXpSb3V0ZXMgfSBmcm9tICcuL3JvdXRlcy90ZXh0MnZpel9yb3V0ZXMnO1xyXG5pbXBvcnQgeyBBc3Npc3RhbnRTZXJ2aWNlIH0gZnJvbSAnLi9zZXJ2aWNlcy9hc3Npc3RhbnRfc2VydmljZSc7XHJcbmltcG9ydCB7IHJlZ2lzdGVyQWdlbnRSb3V0ZXMgfSBmcm9tICcuL3JvdXRlcy9hZ2VudF9yb3V0ZXMnO1xyXG5pbXBvcnQge1xyXG4gIHJlZ2lzdGVyU3VtbWFyeUFzc2lzdGFudFJvdXRlcyxcclxuICByZWdpc3RlckRhdGEyU3VtbWFyeVJvdXRlcyxcclxufSBmcm9tICcuL3JvdXRlcy9zdW1tYXJ5X3JvdXRlcyc7XHJcbmltcG9ydCB7IGNhcGFiaWxpdGllc1Byb3ZpZGVyIGFzIHZpc05MUUNhcGFiaWxpdGllc1Byb3ZpZGVyIH0gZnJvbSAnLi92aXNfdHlwZV9ubHEvY2FwYWJpbGl0aWVzX3Byb3ZpZGVyJztcclxuaW1wb3J0IHsgdmlzTkxRU2F2ZWRPYmplY3RUeXBlIH0gZnJvbSAnLi92aXNfdHlwZV9ubHEvc2F2ZWRfb2JqZWN0X3R5cGUnO1xyXG5pbXBvcnQgeyBjYXBhYmlsaXRpZXNQcm92aWRlciB9IGZyb20gJy4vY2FwYWJpbGl0aWVzJztcclxuaW1wb3J0IHsgRU5BQkxFX0FJX0ZFQVRVUkVTIH0gZnJvbSAnLi91dGlscy9jb25zdGFudHMnO1xyXG5cclxuZXhwb3J0IGNsYXNzIEFzc2lzdGFudFBsdWdpbiBpbXBsZW1lbnRzIFBsdWdpbjxBc3Npc3RhbnRQbHVnaW5TZXR1cCwgQXNzaXN0YW50UGx1Z2luU3RhcnQ+IHtcclxuICBwcml2YXRlIHJlYWRvbmx5IGxvZ2dlcjogTG9nZ2VyO1xyXG4gIHByaXZhdGUgbWVzc2FnZVBhcnNlcnM6IE1lc3NhZ2VQYXJzZXJbXSA9IFtdO1xyXG4gIHByaXZhdGUgYXNzaXN0YW50U2VydmljZSA9IG5ldyBBc3Npc3RhbnRTZXJ2aWNlKCk7XHJcblxyXG4gIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgaW5pdGlhbGl6ZXJDb250ZXh0OiBQbHVnaW5Jbml0aWFsaXplckNvbnRleHQpIHtcclxuICAgIHRoaXMubG9nZ2VyID0gaW5pdGlhbGl6ZXJDb250ZXh0LmxvZ2dlci5nZXQoKTtcclxuICB9XHJcblxyXG4gIHB1YmxpYyBhc3luYyBzZXR1cChjb3JlOiBDb3JlU2V0dXApOiBQcm9taXNlPEFzc2lzdGFudFBsdWdpblNldHVwPiB7XHJcbiAgICB0aGlzLmxvZ2dlci5kZWJ1ZygnQXNzaXN0YW50OiBTZXR1cCcpO1xyXG4gICAgY29uc3QgY29uZmlnID0gYXdhaXQgdGhpcy5pbml0aWFsaXplckNvbnRleHQuY29uZmlnXHJcbiAgICAgIC5jcmVhdGU8Q29uZmlnU2NoZW1hPigpXHJcbiAgICAgIC5waXBlKGZpcnN0KCkpXHJcbiAgICAgIC50b1Byb21pc2UoKTtcclxuXHJcbiAgICBjb25zdCBhc3Npc3RhbnRTZXJ2aWNlU2V0dXAgPSB0aGlzLmFzc2lzdGFudFNlcnZpY2Uuc2V0dXAoKTtcclxuXHJcbiAgICBjb25zdCByb3V0ZXIgPSBjb3JlLmh0dHAuY3JlYXRlUm91dGVyKCk7XHJcblxyXG4gICAgY29yZS5odHRwLnJlZ2lzdGVyUm91dGVIYW5kbGVyQ29udGV4dCgnYXNzaXN0YW50X3BsdWdpbicsICgpID0+IHtcclxuICAgICAgcmV0dXJuIHtcclxuICAgICAgICBjb25maWcsXHJcbiAgICAgICAgbG9nZ2VyOiB0aGlzLmxvZ2dlcixcclxuICAgICAgfTtcclxuICAgIH0pO1xyXG5cclxuICAgIHJlZ2lzdGVyQWdlbnRSb3V0ZXMocm91dGVyLCBhc3Npc3RhbnRTZXJ2aWNlU2V0dXApO1xyXG5cclxuICAgIC8vIFJlZ2lzdGVyIHNlcnZlciBzaWRlIEFQSXNcclxuICAgIHJlZ2lzdGVyQ2hhdFJvdXRlcyhyb3V0ZXIsIHtcclxuICAgICAgbWVzc2FnZVBhcnNlcnM6IHRoaXMubWVzc2FnZVBhcnNlcnMsXHJcbiAgICAgIGF1dGg6IGNvcmUuaHR0cC5hdXRoLFxyXG4gICAgfSk7XHJcblxyXG4gICAgLy8gUmVnaXN0ZXIgcm91dGVyIGZvciB0ZXh0IHRvIHZpc3VhbGl6YXRpb25cclxuICAgIGlmIChjb25maWcudGV4dDJ2aXouZW5hYmxlZCkge1xyXG4gICAgICByZWdpc3RlclRleHQyVml6Um91dGVzKHJvdXRlciwgYXNzaXN0YW50U2VydmljZVNldHVwKTtcclxuICAgICAgY29yZS5jYXBhYmlsaXRpZXMucmVnaXN0ZXJQcm92aWRlcih2aXNOTFFDYXBhYmlsaXRpZXNQcm92aWRlcik7XHJcbiAgICAgIGNvcmUuc2F2ZWRPYmplY3RzLnJlZ2lzdGVyVHlwZSh2aXNOTFFTYXZlZE9iamVjdFR5cGUpO1xyXG4gICAgfVxyXG5cclxuICAgIC8vIFJlZ2lzdGVyIHJvdXRlciBmb3IgYWxlcnQgaW5zaWdodFxyXG4gICAgaWYgKGNvbmZpZy5hbGVydEluc2lnaHQuZW5hYmxlZCkge1xyXG4gICAgICByZWdpc3RlclN1bW1hcnlBc3Npc3RhbnRSb3V0ZXMocm91dGVyLCBhc3Npc3RhbnRTZXJ2aWNlU2V0dXApO1xyXG4gICAgfVxyXG5cclxuICAgIGNvcmUuY2FwYWJpbGl0aWVzLnJlZ2lzdGVyUHJvdmlkZXIoY2FwYWJpbGl0aWVzUHJvdmlkZXIpO1xyXG4gICAgLy8gcmVnaXN0ZXIgVUkgY2FwYWJpbGl0aWVzIGZyb20gZHluYW1pYyBjb25maWcgc2VydmljZVxyXG4gICAgY29yZS5jYXBhYmlsaXRpZXMucmVnaXN0ZXJTd2l0Y2hlcihcclxuICAgICAgYXN5bmMgKG9wZW5zZWFyY2hEYXNoYm9hcmRzUmVxdWVzdDogT3BlblNlYXJjaERhc2hib2FyZHNSZXF1ZXN0KSA9PiB7XHJcbiAgICAgICAgY29uc3QgZHluYW1pY0NvbmZpZ1NlcnZpY2VTdGFydCA9IGF3YWl0IGNvcmUuZHluYW1pY0NvbmZpZ1NlcnZpY2UuZ2V0U3RhcnRTZXJ2aWNlKCk7XHJcbiAgICAgICAgY29uc3Qgc3RvcmUgPSBkeW5hbWljQ29uZmlnU2VydmljZVN0YXJ0LmdldEFzeW5jTG9jYWxTdG9yZSgpO1xyXG4gICAgICAgIGNvbnN0IGNsaWVudCA9IGR5bmFtaWNDb25maWdTZXJ2aWNlU3RhcnQuZ2V0Q2xpZW50KCk7XHJcblxyXG4gICAgICAgIHRyeSB7XHJcbiAgICAgICAgICBjb25zdCBkeW5hbWljQ29uZmlnID0gYXdhaXQgY2xpZW50LmdldENvbmZpZyhcclxuICAgICAgICAgICAgeyBwbHVnaW5Db25maWdQYXRoOiAnYXNzaXN0YW50JyB9LFxyXG4gICAgICAgICAgICB7IGFzeW5jTG9jYWxTdG9yYWdlQ29udGV4dDogc3RvcmUhIH1cclxuICAgICAgICAgICk7XHJcblxyXG4gICAgICAgICAgY29uc3QgW2NvcmVTdGFydF0gPSBhd2FpdCBjb3JlLmdldFN0YXJ0U2VydmljZXMoKTtcclxuICAgICAgICAgIGNvbnN0IHNhdmVkT2JqZWN0c0NsaWVudCA9IGNvcmVTdGFydC5zYXZlZE9iamVjdHMuZ2V0U2NvcGVkQ2xpZW50KFxyXG4gICAgICAgICAgICBvcGVuc2VhcmNoRGFzaGJvYXJkc1JlcXVlc3RcclxuICAgICAgICAgICk7XHJcbiAgICAgICAgICBjb25zdCB1aVNldHRpbmdzQ2xpZW50ID0gY29yZVN0YXJ0LnVpU2V0dGluZ3MuYXNTY29wZWRUb0NsaWVudChzYXZlZE9iamVjdHNDbGllbnQpO1xyXG4gICAgICAgICAgY29uc3QgaXNBc3Npc3RhbnRFbmFibGVkQnlTZXR0aW5nID0gQm9vbGVhbihcclxuICAgICAgICAgICAgYXdhaXQgdWlTZXR0aW5nc0NsaWVudC5nZXQoRU5BQkxFX0FJX0ZFQVRVUkVTKS5jYXRjaCgoKSA9PiBmYWxzZSlcclxuICAgICAgICAgICk7XHJcblxyXG4gICAgICAgICAgcmV0dXJuIHtcclxuICAgICAgICAgICAgYXNzaXN0YW50OiB7XHJcbiAgICAgICAgICAgICAgZW5hYmxlZDogZHluYW1pY0NvbmZpZy5lbmFibGVkICYmIGlzQXNzaXN0YW50RW5hYmxlZEJ5U2V0dGluZyxcclxuICAgICAgICAgICAgICBjaGF0RW5hYmxlZDogZHluYW1pY0NvbmZpZy5jaGF0LmVuYWJsZWQgJiYgaXNBc3Npc3RhbnRFbmFibGVkQnlTZXR0aW5nLFxyXG4gICAgICAgICAgICB9LFxyXG4gICAgICAgICAgfTtcclxuICAgICAgICB9IGNhdGNoIChlKSB7XHJcbiAgICAgICAgICB0aGlzLmxvZ2dlci5lcnJvcihlKTtcclxuICAgICAgICAgIHJldHVybiB7fTtcclxuICAgICAgICB9XHJcbiAgICAgIH1cclxuICAgICk7XHJcblxyXG4gICAgLy8gUmVnaXN0ZXIgcm91dGVyIGZvciBkaXNjb3Zlcnkgc3VtbWFyeVxyXG4gICAgcmVnaXN0ZXJEYXRhMlN1bW1hcnlSb3V0ZXMocm91dGVyLCBhc3Npc3RhbnRTZXJ2aWNlU2V0dXApO1xyXG5cclxuICAgIGNvbnN0IHJlZ2lzdGVyTWVzc2FnZVBhcnNlciA9IChtZXNzYWdlUGFyc2VyOiBNZXNzYWdlUGFyc2VyKSA9PiB7XHJcbiAgICAgIGNvbnN0IGZpbmRJdGVtID0gdGhpcy5tZXNzYWdlUGFyc2Vycy5maW5kKChpdGVtKSA9PiBpdGVtLmlkID09PSBtZXNzYWdlUGFyc2VyLmlkKTtcclxuICAgICAgaWYgKGZpbmRJdGVtKSB7XHJcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGVyZSBpcyBhbHJlYWR5IGEgbWVzc2FnZVBhcnNlciB3aG9zZSBpZCBpcyAke21lc3NhZ2VQYXJzZXIuaWR9YCk7XHJcbiAgICAgIH1cclxuXHJcbiAgICAgIHRoaXMubWVzc2FnZVBhcnNlcnMucHVzaChtZXNzYWdlUGFyc2VyKTtcclxuICAgIH07XHJcblxyXG4gICAgcmVnaXN0ZXJNZXNzYWdlUGFyc2VyKEJhc2ljSW5wdXRPdXRwdXRQYXJzZXIpO1xyXG4gICAgcmVnaXN0ZXJNZXNzYWdlUGFyc2VyKFZpc3VhbGl6YXRpb25DYXJkUGFyc2VyKTtcclxuXHJcbiAgICByZXR1cm4ge1xyXG4gICAgICBhc3Npc3RhbnRTZXJ2aWNlOiBhc3Npc3RhbnRTZXJ2aWNlU2V0dXAsXHJcbiAgICAgIHJlZ2lzdGVyTWVzc2FnZVBhcnNlcixcclxuICAgICAgcmVtb3ZlTWVzc2FnZVBhcnNlcjogKHBhcnNlcklkOiBNZXNzYWdlUGFyc2VyWydpZCddKSA9PiB7XHJcbiAgICAgICAgY29uc3QgZmluZEluZGV4ID0gdGhpcy5tZXNzYWdlUGFyc2Vycy5maW5kSW5kZXgoKGl0ZW0pID0+IGl0ZW0uaWQgPT09IHBhcnNlcklkKTtcclxuICAgICAgICBpZiAoZmluZEluZGV4IDwgMCkge1xyXG4gICAgICAgICAgdGhpcy5sb2dnZXIuZXJyb3IoYFRoZXJlIGlzIG5vdCBhIG1lc3NhZ2VQYXJzZXIgd2hvc2UgaWQgaXMgJHtwYXJzZXJJZH1gKTtcclxuICAgICAgICB9XHJcblxyXG4gICAgICAgIHRoaXMubWVzc2FnZVBhcnNlcnMuc3BsaWNlKGZpbmRJbmRleCwgMSk7XHJcbiAgICAgIH0sXHJcbiAgICB9O1xyXG4gIH1cclxuXHJcbiAgcHVibGljIHN0YXJ0KGNvcmU6IENvcmVTdGFydCkge1xyXG4gICAgdGhpcy5sb2dnZXIuZGVidWcoJ0Fzc2lzdGFudDogU3RhcnRlZCcpO1xyXG4gICAgdGhpcy5hc3Npc3RhbnRTZXJ2aWNlLnN0YXJ0KCk7XHJcbiAgICByZXR1cm4ge307XHJcbiAgfVxyXG5cclxuICBwdWJsaWMgc3RvcCgpIHtcclxuICAgIHRoaXMuYXNzaXN0YW50U2VydmljZS5zdG9wKCk7XHJcbiAgfVxyXG59XHJcbiJdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBS0EsSUFBQUEsVUFBQSxHQUFBQyxPQUFBO0FBV0EsSUFBQUMsMEJBQUEsR0FBQUQsT0FBQTtBQUNBLElBQUFFLDBCQUFBLEdBQUFGLE9BQUE7QUFDQSxJQUFBRyxZQUFBLEdBQUFILE9BQUE7QUFDQSxJQUFBSSxnQkFBQSxHQUFBSixPQUFBO0FBQ0EsSUFBQUssa0JBQUEsR0FBQUwsT0FBQTtBQUNBLElBQUFNLGFBQUEsR0FBQU4sT0FBQTtBQUNBLElBQUFPLGVBQUEsR0FBQVAsT0FBQTtBQUlBLElBQUFRLHNCQUFBLEdBQUFSLE9BQUE7QUFDQSxJQUFBUyxrQkFBQSxHQUFBVCxPQUFBO0FBQ0EsSUFBQVUsYUFBQSxHQUFBVixPQUFBO0FBQ0EsSUFBQVcsVUFBQSxHQUFBWCxPQUFBO0FBQXVELFNBQUFZLGdCQUFBQyxDQUFBLEVBQUFDLENBQUEsRUFBQUMsQ0FBQSxZQUFBRCxDQUFBLEdBQUFFLGNBQUEsQ0FBQUYsQ0FBQSxNQUFBRCxDQUFBLEdBQUFJLE1BQUEsQ0FBQUMsY0FBQSxDQUFBTCxDQUFBLEVBQUFDLENBQUEsSUFBQUssS0FBQSxFQUFBSixDQUFBLEVBQUFLLFVBQUEsTUFBQUMsWUFBQSxNQUFBQyxRQUFBLFVBQUFULENBQUEsQ0FBQUMsQ0FBQSxJQUFBQyxDQUFBLEVBQUFGLENBQUE7QUFBQSxTQUFBRyxlQUFBRCxDQUFBLFFBQUFRLENBQUEsR0FBQUMsWUFBQSxDQUFBVCxDQUFBLHVDQUFBUSxDQUFBLEdBQUFBLENBQUEsR0FBQUEsQ0FBQTtBQUFBLFNBQUFDLGFBQUFULENBQUEsRUFBQUQsQ0FBQSwyQkFBQUMsQ0FBQSxLQUFBQSxDQUFBLFNBQUFBLENBQUEsTUFBQUYsQ0FBQSxHQUFBRSxDQUFBLENBQUFVLE1BQUEsQ0FBQUMsV0FBQSxrQkFBQWIsQ0FBQSxRQUFBVSxDQUFBLEdBQUFWLENBQUEsQ0FBQWMsSUFBQSxDQUFBWixDQUFBLEVBQUFELENBQUEsdUNBQUFTLENBQUEsU0FBQUEsQ0FBQSxZQUFBSyxTQUFBLHlFQUFBZCxDQUFBLEdBQUFlLE1BQUEsR0FBQUMsTUFBQSxFQUFBZixDQUFBLEtBN0J2RDtBQUNBO0FBQ0E7QUFDQTtBQTRCTyxNQUFNZ0IsZUFBZSxDQUErRDtFQUt6RkMsV0FBV0EsQ0FBa0JDLGtCQUE0QyxFQUFFO0lBQUEsS0FBOUNBLGtCQUE0QyxHQUE1Q0Esa0JBQTRDO0lBQUFyQixlQUFBO0lBQUFBLGVBQUEseUJBSC9CLEVBQUU7SUFBQUEsZUFBQSwyQkFDakIsSUFBSXNCLG1DQUFnQixDQUFDLENBQUM7SUFHL0MsSUFBSSxDQUFDQyxNQUFNLEdBQUdGLGtCQUFrQixDQUFDRSxNQUFNLENBQUNDLEdBQUcsQ0FBQyxDQUFDO0VBQy9DO0VBRUEsTUFBYUMsS0FBS0EsQ0FBQ0MsSUFBZSxFQUFpQztJQUNqRSxJQUFJLENBQUNILE1BQU0sQ0FBQ0ksS0FBSyxDQUFDLGtCQUFrQixDQUFDO0lBQ3JDLE1BQU1DLE1BQU0sR0FBRyxNQUFNLElBQUksQ0FBQ1Asa0JBQWtCLENBQUNPLE1BQU0sQ0FDaERDLE1BQU0sQ0FBZSxDQUFDLENBQ3RCQyxJQUFJLENBQUMsSUFBQUMsZ0JBQUssRUFBQyxDQUFDLENBQUMsQ0FDYkMsU0FBUyxDQUFDLENBQUM7SUFFZCxNQUFNQyxxQkFBcUIsR0FBRyxJQUFJLENBQUNDLGdCQUFnQixDQUFDVCxLQUFLLENBQUMsQ0FBQztJQUUzRCxNQUFNVSxNQUFNLEdBQUdULElBQUksQ0FBQ1UsSUFBSSxDQUFDQyxZQUFZLENBQUMsQ0FBQztJQUV2Q1gsSUFBSSxDQUFDVSxJQUFJLENBQUNFLDJCQUEyQixDQUFDLGtCQUFrQixFQUFFLE1BQU07TUFDOUQsT0FBTztRQUNMVixNQUFNO1FBQ05MLE1BQU0sRUFBRSxJQUFJLENBQUNBO01BQ2YsQ0FBQztJQUNILENBQUMsQ0FBQztJQUVGLElBQUFnQixpQ0FBbUIsRUFBQ0osTUFBTSxFQUFFRixxQkFBcUIsQ0FBQzs7SUFFbEQ7SUFDQSxJQUFBTywrQkFBa0IsRUFBQ0wsTUFBTSxFQUFFO01BQ3pCTSxjQUFjLEVBQUUsSUFBSSxDQUFDQSxjQUFjO01BQ25DQyxJQUFJLEVBQUVoQixJQUFJLENBQUNVLElBQUksQ0FBQ007SUFDbEIsQ0FBQyxDQUFDOztJQUVGO0lBQ0EsSUFBSWQsTUFBTSxDQUFDZSxRQUFRLENBQUNDLE9BQU8sRUFBRTtNQUMzQixJQUFBQyx1Q0FBc0IsRUFBQ1YsTUFBTSxFQUFFRixxQkFBcUIsQ0FBQztNQUNyRFAsSUFBSSxDQUFDb0IsWUFBWSxDQUFDQyxnQkFBZ0IsQ0FBQ0MsMkNBQTBCLENBQUM7TUFDOUR0QixJQUFJLENBQUN1QixZQUFZLENBQUNDLFlBQVksQ0FBQ0Msd0NBQXFCLENBQUM7SUFDdkQ7O0lBRUE7SUFDQSxJQUFJdkIsTUFBTSxDQUFDd0IsWUFBWSxDQUFDUixPQUFPLEVBQUU7TUFDL0IsSUFBQVMsOENBQThCLEVBQUNsQixNQUFNLEVBQUVGLHFCQUFxQixDQUFDO0lBQy9EO0lBRUFQLElBQUksQ0FBQ29CLFlBQVksQ0FBQ0MsZ0JBQWdCLENBQUNPLGtDQUFvQixDQUFDO0lBQ3hEO0lBQ0E1QixJQUFJLENBQUNvQixZQUFZLENBQUNTLGdCQUFnQixDQUNoQyxNQUFPQywyQkFBd0QsSUFBSztNQUNsRSxNQUFNQyx5QkFBeUIsR0FBRyxNQUFNL0IsSUFBSSxDQUFDZ0Msb0JBQW9CLENBQUNDLGVBQWUsQ0FBQyxDQUFDO01BQ25GLE1BQU1DLEtBQUssR0FBR0gseUJBQXlCLENBQUNJLGtCQUFrQixDQUFDLENBQUM7TUFDNUQsTUFBTUMsTUFBTSxHQUFHTCx5QkFBeUIsQ0FBQ00sU0FBUyxDQUFDLENBQUM7TUFFcEQsSUFBSTtRQUNGLE1BQU1DLGFBQWEsR0FBRyxNQUFNRixNQUFNLENBQUNHLFNBQVMsQ0FDMUM7VUFBRUMsZ0JBQWdCLEVBQUU7UUFBWSxDQUFDLEVBQ2pDO1VBQUVDLHdCQUF3QixFQUFFUDtRQUFPLENBQ3JDLENBQUM7UUFFRCxNQUFNLENBQUNRLFNBQVMsQ0FBQyxHQUFHLE1BQU0xQyxJQUFJLENBQUMyQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2pELE1BQU1DLGtCQUFrQixHQUFHRixTQUFTLENBQUNuQixZQUFZLENBQUNzQixlQUFlLENBQy9EZiwyQkFDRixDQUFDO1FBQ0QsTUFBTWdCLGdCQUFnQixHQUFHSixTQUFTLENBQUNLLFVBQVUsQ0FBQ0MsZ0JBQWdCLENBQUNKLGtCQUFrQixDQUFDO1FBQ2xGLE1BQU1LLDJCQUEyQixHQUFHQyxPQUFPLENBQ3pDLE1BQU1KLGdCQUFnQixDQUFDaEQsR0FBRyxDQUFDcUQsNkJBQWtCLENBQUMsQ0FBQ0MsS0FBSyxDQUFDLE1BQU0sS0FBSyxDQUNsRSxDQUFDO1FBRUQsT0FBTztVQUNMQyxTQUFTLEVBQUU7WUFDVG5DLE9BQU8sRUFBRW9CLGFBQWEsQ0FBQ3BCLE9BQU8sSUFBSStCLDJCQUEyQjtZQUM3REssV0FBVyxFQUFFaEIsYUFBYSxDQUFDaUIsSUFBSSxDQUFDckMsT0FBTyxJQUFJK0I7VUFDN0M7UUFDRixDQUFDO01BQ0gsQ0FBQyxDQUFDLE9BQU8xRSxDQUFDLEVBQUU7UUFDVixJQUFJLENBQUNzQixNQUFNLENBQUMyRCxLQUFLLENBQUNqRixDQUFDLENBQUM7UUFDcEIsT0FBTyxDQUFDLENBQUM7TUFDWDtJQUNGLENBQ0YsQ0FBQzs7SUFFRDtJQUNBLElBQUFrRiwwQ0FBMEIsRUFBQ2hELE1BQU0sRUFBRUYscUJBQXFCLENBQUM7SUFFekQsTUFBTW1ELHFCQUFxQixHQUFJQyxhQUE0QixJQUFLO01BQzlELE1BQU1DLFFBQVEsR0FBRyxJQUFJLENBQUM3QyxjQUFjLENBQUM4QyxJQUFJLENBQUVDLElBQUksSUFBS0EsSUFBSSxDQUFDQyxFQUFFLEtBQUtKLGFBQWEsQ0FBQ0ksRUFBRSxDQUFDO01BQ2pGLElBQUlILFFBQVEsRUFBRTtRQUNaLE1BQU0sSUFBSUksS0FBSyxDQUFFLGdEQUErQ0wsYUFBYSxDQUFDSSxFQUFHLEVBQUMsQ0FBQztNQUNyRjtNQUVBLElBQUksQ0FBQ2hELGNBQWMsQ0FBQ2tELElBQUksQ0FBQ04sYUFBYSxDQUFDO0lBQ3pDLENBQUM7SUFFREQscUJBQXFCLENBQUNRLGlEQUFzQixDQUFDO0lBQzdDUixxQkFBcUIsQ0FBQ1Msa0RBQXVCLENBQUM7SUFFOUMsT0FBTztNQUNMM0QsZ0JBQWdCLEVBQUVELHFCQUFxQjtNQUN2Q21ELHFCQUFxQjtNQUNyQlUsbUJBQW1CLEVBQUdDLFFBQTZCLElBQUs7UUFDdEQsTUFBTUMsU0FBUyxHQUFHLElBQUksQ0FBQ3ZELGNBQWMsQ0FBQ3VELFNBQVMsQ0FBRVIsSUFBSSxJQUFLQSxJQUFJLENBQUNDLEVBQUUsS0FBS00sUUFBUSxDQUFDO1FBQy9FLElBQUlDLFNBQVMsR0FBRyxDQUFDLEVBQUU7VUFDakIsSUFBSSxDQUFDekUsTUFBTSxDQUFDMkQsS0FBSyxDQUFFLDRDQUEyQ2EsUUFBUyxFQUFDLENBQUM7UUFDM0U7UUFFQSxJQUFJLENBQUN0RCxjQUFjLENBQUN3RCxNQUFNLENBQUNELFNBQVMsRUFBRSxDQUFDLENBQUM7TUFDMUM7SUFDRixDQUFDO0VBQ0g7RUFFT0UsS0FBS0EsQ0FBQ3hFLElBQWUsRUFBRTtJQUM1QixJQUFJLENBQUNILE1BQU0sQ0FBQ0ksS0FBSyxDQUFDLG9CQUFvQixDQUFDO0lBQ3ZDLElBQUksQ0FBQ08sZ0JBQWdCLENBQUNnRSxLQUFLLENBQUMsQ0FBQztJQUM3QixPQUFPLENBQUMsQ0FBQztFQUNYO0VBRU9DLElBQUlBLENBQUEsRUFBRztJQUNaLElBQUksQ0FBQ2pFLGdCQUFnQixDQUFDaUUsSUFBSSxDQUFDLENBQUM7RUFDOUI7QUFDRjtBQUFDQyxPQUFBLENBQUFqRixlQUFBLEdBQUFBLGVBQUEifQ==