"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadSampleData = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _readline = _interopRequireDefault(require("readline"));
var _fs = _interopRequireDefault(require("fs"));
var _zlib = require("zlib");
var _lodash = require("lodash");
var _helpers = require("../../utils/helpers");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

//@ts-ignore

const BULK_INSERT_SIZE = 500;
const loadSampleData = (filePath, indexName, client, request, context, dataSourceEnabled) => {
  const {
    dataSourceId = ''
  } = request.params;
  return new Promise((resolve, reject) => {
    let count = 0;
    let docs = [];
    let isPaused = false;
    let offset = 0;
    const startTime = (0, _moment.default)(new Date().getTime()).subtract({
      days: 7
    });

    // Create the read stream for the file. Set a smaller buffer size here to prevent it from
    // getting too large, to prevent inserting too many docs at once into the index.
    const readStream = _fs.default.createReadStream(filePath, {
      highWaterMark: 1024 * 4
    });
    const lineStream = _readline.default.createInterface({
      input: readStream.pipe((0, _zlib.createUnzip)())
    });

    // This is only ran when the end of lineStream closes normally. It is used to
    // bulk insert the final batch of lines that are < BULK_INSERT_SIZE
    const onClose = async () => {
      if (docs.length > 0) {
        try {
          await bulkInsert(docs);
        } catch (err) {
          reject(err);
          return;
        }
      }
      resolve(count);
    };
    lineStream.on('close', onClose);
    lineStream.on('pause', async () => {
      isPaused = true;
    });
    lineStream.on('resume', async () => {
      isPaused = false;
    });
    lineStream.on('line', async doc => {
      // for the initial doc, get the timestamp to properly set an offset
      if (count === 0) {
        const initialTime = (0, _moment.default)(JSON.parse(doc).timestamp);
        offset = startTime.diff(initialTime);
      }
      count++;
      docs.push(doc);

      // If not currently paused: pause the stream to prevent concurrent bulk inserts
      // on the cluster which could cause performance issues.
      // Also, empty the current docs[] before performing the bulk insert to prevent
      // buffered docs from being dropped.
      if (docs.length >= BULK_INSERT_SIZE && !isPaused) {
        lineStream.pause();

        // save the docs to insert, and empty out current docs list
        const docsToInsert = docs.slice();
        docs = [];
        try {
          await bulkInsert(docsToInsert);
          lineStream.resume();
        } catch (err) {
          lineStream.removeListener('close', onClose);
          lineStream.close();
          reject(err);
        }
      }
    });
    const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, dataSourceEnabled, request, dataSourceId, client);
    const bulkInsert = async docs => {
      try {
        const bulkBody = prepareBody(docs, offset);
        const resp = await callWithRequest('bulk', {
          body: bulkBody
        });
        if (resp.errors) {
          const errorItems = resp.items;
          const firstErrorReason = (0, _lodash.isEmpty)(errorItems) ? 'Error while bulk inserting. Please try again.' : (0, _helpers.prettifyErrorMessage)(errorItems[0].index.error.reason);
          return Promise.reject(new Error(firstErrorReason));
        }
      } catch (err) {
        console.log('Error while bulk inserting. ', err);
        return Promise.reject(new Error('Error while bulk inserting. Please try again.'));
      }
    };
    const prepareBody = (docs, offset) => {
      const bulkBody = [];
      let docIdCount = count - docs.length;
      docs.forEach(doc => {
        bulkBody.push(getDocDetails(docIdCount));
        bulkBody.push(updateTimestamp(doc, offset));
        docIdCount++;
      });
      return bulkBody;
    };
    const updateTimestamp = (doc, offset) => {
      let docAsJSON = JSON.parse(doc);
      const updatedTimestamp = docAsJSON.timestamp + offset;
      docAsJSON.timestamp = updatedTimestamp;
      return docAsJSON;
    };
    const getDocDetails = docIdCount => {
      return `{ "index": { "_index": "${indexName}", "_id": ${docIdCount} } }`;
    };
  });
};
exports.loadSampleData = loadSampleData;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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