"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processTaskError = exports.isIndexNotFoundError = exports.getTaskState = exports.getTaskInitProgress = exports.getLatestTaskForForecasterQuery = exports.getLatestForecasterTasksQuery = exports.getErrorMessage = exports.convertTaskAndJobFieldsToCamelCase = exports.convertStaticFieldsToCamelCase = exports.convertForecastKeysToSnakeCase = exports.convertForecastKeysToCamelCase = exports.combineTaskState = exports.buildEntityListQuery = void 0;
var _lodash = require("lodash");
var _constants = require("../../utils/constants");
var _helpers = require("../../utils/helpers");
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

const convertForecastKeysToSnakeCase = payload => {
  return {
    ...(0, _helpers.mapKeysDeep)({
      ...(0, _lodash.omit)(payload, ['filterQuery', 'featureAttributes']) // Exclude the filterQuery,
    }, _helpers.toSnake),
    filter_query: (0, _lodash.get)(payload, 'filterQuery', {}),
    ui_metadata: (0, _lodash.get)(payload, 'uiMetadata', {}),
    feature_attributes: (0, _lodash.get)(payload, 'featureAttributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregationQuery'])
      }, _helpers.toSnake),
      aggregation_query: feature.aggregationQuery
    }))
  };
};
exports.convertForecastKeysToSnakeCase = convertForecastKeysToSnakeCase;
const processTaskError = error => {
  const errorWithPrefixRemoved = error.replace(_constants.OPENSEARCH_EXCEPTION_PREFIX, '');
  return (0, _lodash.isEmpty)(errorWithPrefixRemoved) || errorWithPrefixRemoved.endsWith('.') ? errorWithPrefixRemoved : errorWithPrefixRemoved + '.';
};

// Following checks/transformations need to be made here:
// - set to INACTIVE_NOT_STARTED if there is no existing task for this forecaster
// - set to UNEXPECTED_FAILURE if the task is in a FAILED state & the error message is unreadable / is a stack trace
// - set to INITIALIZING_FORECAST if the task is in a CREATED state
// - set to INACTIVE_STOPPED if the task is in a STOPPED state
// - override INACTIVE states (INACTIVE_STOPPED or INACTIVE_NOT_STARTED) with test-related states
//   (INITIALIZING_TEST, TEST_COMPLETE, or INIT_TEST_FAILURE) if runOnceTask exists with these states
exports.processTaskError = processTaskError;
const combineTaskState = (realTimeTask, runOnceTask) => {
  const realTimeState = (0, _lodash.get)(realTimeTask, 'state', 'INACTIVE_NOT_STARTED');
  const updatedStateString = realTimeState === 'CREATED' ? 'INITIALIZING_FORECAST' : realTimeState === 'STOPPED' ? 'INACTIVE_STOPPED' : realTimeState;
  //@ts-ignore
  let updatedState = _constants.FORECASTER_STATE[updatedStateString];

  // at the beginning, runOnceTask is inactive before going into initializing test state
  // if runOnceTask is not empty and the error is empty, set the state to INIT_TEST
  const runOnceState = (0, _lodash.get)(runOnceTask, 'state', 'INACTIVE_NOT_STARTED');
  const runOnceStateError = (0, _lodash.get)(runOnceTask, 'error', '');
  const updatedRunOnceStateString = runOnceState === 'INACTIVE' && runOnceStateError === '' ? 'INIT_TEST' : runOnceState;
  const realTimeLastUpdateTime = (0, _lodash.get)(realTimeTask, 'last_update_time', 0);
  const runOnceLastUpdateTime = (0, _lodash.get)(runOnceTask, 'last_update_time', 0);

  // Check if current state is INACTIVE and runOnceTask has a priority state
  // when realTimeTask is not updated for a while
  if ((updatedState === _constants.FORECASTER_STATE.INACTIVE_STOPPED || updatedState === _constants.FORECASTER_STATE.INACTIVE_NOT_STARTED) && runOnceTask && runOnceLastUpdateTime > realTimeLastUpdateTime) {
    // Convert runOnceState to enum value before comparison
    const runOnceStateEnum = _constants.FORECASTER_STATE[updatedRunOnceStateString];
    if (runOnceStateEnum === _constants.FORECASTER_STATE.INIT_TEST || runOnceStateEnum === _constants.FORECASTER_STATE.TEST_COMPLETE || runOnceStateEnum === _constants.FORECASTER_STATE.INIT_TEST_FAILED) {
      updatedState = runOnceStateEnum;
    }
  }
  return updatedState;
};
exports.combineTaskState = combineTaskState;
const getTaskState = realTimeTask => {
  const state = (0, _lodash.get)(realTimeTask, 'state', 'INACTIVE_NOT_STARTED');
  const updatedStateString = state === 'CREATED' ? 'INITIALIZING_FORECAST' : state === 'STOPPED' ? 'INACTIVE_STOPPED' : state;
  //@ts-ignore
  return _constants.FORECASTER_STATE[updatedStateString];
};
exports.getTaskState = getTaskState;
const convertForecastKeysToCamelCase = response => {
  let camelCaseResponse = {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'ui_metadata', 'feature_query', 'feature_attributes', 'forecasterJob', 'runOnceTask']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    enabled: (0, _lodash.get)(response, 'adJob.enabled', false),
    enabledTime: (0, _lodash.get)(response, 'adJob.enabled_time'),
    disabledTime: (0, _lodash.get)(response, 'adJob.disabled_time'),
    categoryField: (0, _lodash.get)(response, 'category_field')
  };
  if (!(0, _lodash.isEmpty)((0, _lodash.get)(response, 'runOnceTask', {}))) {
    camelCaseResponse = {
      ...camelCaseResponse,
      //@ts-ignore
      taskId: (0, _lodash.get)(response, 'runOnceTask.task_id'),
      taskState: getTaskState((0, _lodash.get)(response, 'runOnceTask', {})),
      taskProgress: (0, _lodash.get)(response, 'runOnceTask.task_progress'),
      taskError: processTaskError((0, _lodash.get)(response, 'runOnceTask.error', '')),
      detectionDateRange: {
        startTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.start_time'),
        endTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.end_time')
      }
    };
  }
  return camelCaseResponse;
};
exports.convertForecastKeysToCamelCase = convertForecastKeysToCamelCase;
const isIndexNotFoundError = err => {
  return err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception';
};
exports.isIndexNotFoundError = isIndexNotFoundError;
const getErrorMessage = err => {
  return !(0, _lodash.isEmpty)((0, _lodash.get)(err, 'body.error.reason')) ? (0, _lodash.get)(err, 'body.error.reason') : (0, _lodash.get)(err, 'message');
};

// Filtering by 'is_latest=true' is not enough here. It is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.getErrorMessage = getErrorMessage;
const getLatestForecasterTasksQuery = realtime => {
  const taskTypes = realtime ? _constants.FORECAST_REALTIME_TASK_TYPES : _constants.FORECAST_RUN_ONCE_TASK_TYPES;
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    },
    aggs: {
      forecasters: {
        terms: {
          field: 'forecaster_id',
          size: _constants.MAX_FORECASTER
        },
        aggs: {
          latest_tasks: {
            top_hits: {
              size: 1,
              sort: {
                execution_start_time: _constants.SORT_DIRECTION.DESC
              }
            }
          }
        }
      }
    }
  };
};

// Converts the static forecaster fields into camelcase. Ignores any job or task-related fields
exports.getLatestForecasterTasksQuery = getLatestForecasterTasksQuery;
const convertStaticFieldsToCamelCase = response => {
  return {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'feature_query', 'feature_attributes', 'ui_metadata', 'forecaster_job', 'realtime_task', 'run_once_task']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    lastUiBreakingChangeTime: (0, _lodash.get)(response, 'last_ui_breaking_change_time'),
    lastUpdateTime: (0, _lodash.get)(response, 'last_update_time')
  };
};

// Filtering by 'is_latest=true' is not enough here. It is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.convertStaticFieldsToCamelCase = convertStaticFieldsToCamelCase;
const getLatestTaskForForecasterQuery = (forecasterId, realtime) => {
  const taskTypes = realtime ? _constants.FORECAST_REALTIME_TASK_TYPES : _constants.FORECAST_RUN_ONCE_TASK_TYPES;
  return {
    size: 1,
    sort: {
      execution_start_time: _constants.SORT_DIRECTION.DESC
    },
    query: {
      bool: {
        filter: [{
          term: {
            forecaster_id: forecasterId
          }
        }, {
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    }
  };
};
exports.getLatestTaskForForecasterQuery = getLatestTaskForForecasterQuery;
const getTaskInitProgress = task => {
  if ((task === null || task === void 0 ? void 0 : task.init_progress) !== undefined) {
    return {
      percentageStr: `${((0, _lodash.get)(task, 'init_progress', 0) * 100).toFixed(0)}%`,
      estimatedMinutesLeft: task.estimated_minutes_left
    };
  }
  return undefined;
};

// Converts the task-related detector fields into camelcase
exports.getTaskInitProgress = getTaskInitProgress;
const convertTaskAndJobFieldsToCamelCase = (realtimeTask, runOnceTask, forecasterJob) => {
  // Populate Forecaster job fields
  return {
    enabled: (0, _lodash.get)(forecasterJob, 'enabled', false),
    enabledTime: (0, _lodash.get)(forecasterJob, 'enabled_time'),
    disabledTime: (0, _lodash.get)(forecasterJob, 'disabled_time'),
    curState: combineTaskState(realtimeTask, runOnceTask),
    stateError: (0, _lodash.get)(realtimeTask, 'error') || (0, _lodash.get)(runOnceTask, 'error'),
    initProgress: getTaskInitProgress(realtimeTask),
    realTimeLastUpdateTime: (0, _lodash.get)(realtimeTask, 'last_update_time'),
    runOnceLastUpdateTime: (0, _lodash.get)(runOnceTask, 'last_update_time'),
    taskId: (0, _lodash.get)(runOnceTask, 'task_id'),
    taskState: getTaskState(runOnceTask),
    taskProgress: (0, _lodash.get)(runOnceTask, 'task_progress'),
    taskError: processTaskError((0, _lodash.get)(runOnceTask, 'error', ''))
  };
};

/**
 * Builds an OpenSearch query for matching multiple ForecastEntity objects
 * Each entity's key-value pairs are grouped with "must" clauses
 * Different entities are combined with "should" clauses
 */
exports.convertTaskAndJobFieldsToCamelCase = convertTaskAndJobFieldsToCamelCase;
const buildEntityListQuery = entityList => {
  if (!entityList || entityList.length === 0) {
    return undefined;
  }

  // Create an array of bool queries - one for each entity
  const shouldClauses = entityList.map(entity => {
    // For each entity, create nested queries for each key-value pair
    const mustClauses = Object.entries(entity).map(([name, value]) => {
      return {
        nested: {
          path: "entity",
          query: {
            bool: {
              must: [{
                term: {
                  "entity.name": name
                }
              }, {
                term: {
                  "entity.value": value
                }
              }]
            }
          },
          ignore_unmapped: false,
          score_mode: "avg"
        }
      };
    });

    // All key-value pairs for this entity must match
    return {
      bool: {
        must: mustClauses
      }
    };
  });

  // At least one entity should match
  return {
    bool: {
      should: shouldClauses,
      minimum_should_match: 1
    }
  };
};
exports.buildEntityListQuery = buildEntityListQuery;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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