"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.processTaskError = exports.isRealTimeTask = exports.isIndexNotFoundError = exports.getTaskState = exports.getTaskInitProgress = exports.getResultAggregationQuery = exports.getLatestTaskForDetectorQuery = exports.getLatestDetectorTasksQuery = exports.getFinalDetectorStates = exports.getFiltersFromEntityList = exports.getErrorMessage = exports.getDetectorsWithJob = exports.getDetectorTasks = exports.getDetectorResults = exports.convertTaskAndJobFieldsToCamelCase = exports.convertStaticFieldsToCamelCase = exports.convertPreviewInputKeysToSnakeCase = exports.convertDetectorKeysToSnakeCase = exports.convertDetectorKeysToCamelCase = exports.appendTaskInfo = exports.anomalyResultMapper = void 0;
var _lodash = require("lodash");
var _helpers = require("../../utils/helpers");
var _constants = require("../../utils/constants");
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

const convertDetectorKeysToSnakeCase = payload => {
  return {
    ...(0, _helpers.mapKeysDeep)({
      ...(0, _lodash.omit)(payload, ['filterQuery', 'featureAttributes']) // Exclude the filterQuery,
    }, _helpers.toSnake),
    filter_query: (0, _lodash.get)(payload, 'filterQuery', {}),
    ui_metadata: (0, _lodash.get)(payload, 'uiMetadata', {}),
    feature_attributes: (0, _lodash.get)(payload, 'featureAttributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregationQuery'])
      }, _helpers.toSnake),
      aggregation_query: feature.aggregationQuery
    }))
  };
};
exports.convertDetectorKeysToSnakeCase = convertDetectorKeysToSnakeCase;
const convertPreviewInputKeysToSnakeCase = payload => {
  return {
    ...(0, _helpers.mapKeysDeep)({
      ...(0, _lodash.omit)(payload, ['detector']) // Exclude the detector,
    }, _helpers.toSnake),
    detector: convertDetectorKeysToSnakeCase((0, _lodash.get)(payload, 'detector', {}))
  };
};
exports.convertPreviewInputKeysToSnakeCase = convertPreviewInputKeysToSnakeCase;
const convertDetectorKeysToCamelCase = response => {
  let camelCaseResponse = {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'ui_metadata', 'feature_query', 'feature_attributes', 'adJob', 'historicalTask']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {}),
    enabled: (0, _lodash.get)(response, 'adJob.enabled', false),
    enabledTime: (0, _lodash.get)(response, 'adJob.enabled_time'),
    disabledTime: (0, _lodash.get)(response, 'adJob.disabled_time'),
    categoryField: (0, _lodash.get)(response, 'category_field')
  };
  if (!(0, _lodash.isEmpty)((0, _lodash.get)(response, 'historicalTask', {}))) {
    camelCaseResponse = {
      ...camelCaseResponse,
      //@ts-ignore
      taskId: (0, _lodash.get)(response, 'historicalTask.task_id'),
      taskState: getTaskState((0, _lodash.get)(response, 'historicalTask', {})),
      taskProgress: (0, _lodash.get)(response, 'historicalTask.task_progress'),
      taskError: processTaskError((0, _lodash.get)(response, 'historicalTask.error', '')),
      detectionDateRange: {
        startTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.start_time'),
        endTime: (0, _lodash.get)(response, 'historicalTask.detection_date_range.end_time')
      }
    };
  }
  return camelCaseResponse;
};

// Converts the static detector fields into camelcase. Ignores any job or task-related fields
exports.convertDetectorKeysToCamelCase = convertDetectorKeysToCamelCase;
const convertStaticFieldsToCamelCase = response => {
  return {
    ...(0, _helpers.mapKeysDeep)((0, _lodash.omit)(response, ['filter_query', 'feature_query', 'feature_attributes', 'ui_metadata', 'anomaly_detector_job', 'anomaly_detection_task', 'realtime_detection_task', 'historical_analysis_task']), _helpers.toCamel),
    filterQuery: (0, _lodash.get)(response, 'filter_query', {}),
    featureAttributes: (0, _lodash.get)(response, 'feature_attributes', []).map(feature => ({
      ...(0, _helpers.mapKeysDeep)({
        ...(0, _lodash.omit)(feature, ['aggregation_query'])
      }, _helpers.toCamel),
      aggregationQuery: feature.aggregation_query
    })),
    uiMetadata: (0, _lodash.get)(response, 'ui_metadata', {})
  };
};

// Converts the task-related detector fields into camelcase
exports.convertStaticFieldsToCamelCase = convertStaticFieldsToCamelCase;
const convertTaskAndJobFieldsToCamelCase = (realtimeTask, historicalTask, detectorJob) => {
  let response = {};

  // Populate AD job fields
  response = {
    ...response,
    enabled: (0, _lodash.get)(detectorJob, 'enabled', false),
    enabledTime: (0, _lodash.get)(detectorJob, 'enabled_time'),
    disabledTime: (0, _lodash.get)(detectorJob, 'disabled_time')
  };

  // Populate RT-task-related fields
  response = realtimeTask !== undefined ? {
    ...response,
    curState: getTaskState(realtimeTask),
    stateError: processTaskError((0, _lodash.get)(realtimeTask, 'error', '')),
    initProgress: getTaskInitProgress(realtimeTask)
  } : {
    ...response,
    curState: (0, _lodash.get)(detectorJob, 'enabled', false) ? _constants.DETECTOR_STATE.RUNNING : _constants.DETECTOR_STATE.DISABLED
  };

  // Detection date range field is stored under the 'detector' field in legacy historical tasks.
  // To handle this, need to add a check to fetch the date range from the correct place
  const isLegacyHistorical = (0, _lodash.get)(historicalTask, 'detection_date_range') === undefined && (0, _lodash.get)(historicalTask, 'detector.detection_date_range') !== undefined;
  const legacyDateRangePrefix = isLegacyHistorical ? 'detector.' : '';

  // Populate historical-task-related fields
  response = {
    ...response,
    taskId: (0, _lodash.get)(historicalTask, 'id'),
    taskState: getTaskState(historicalTask),
    taskProgress: (0, _lodash.get)(historicalTask, 'task_progress'),
    taskError: processTaskError((0, _lodash.get)(historicalTask, 'error', '')),
    detectionDateRange: {
      startTime: (0, _lodash.get)(historicalTask, `${legacyDateRangePrefix}detection_date_range.start_time`),
      endTime: (0, _lodash.get)(historicalTask, `${legacyDateRangePrefix}detection_date_range.end_time`)
    }
  };
  if ((0, _lodash.isEmpty)(historicalTask)) {
    //@ts-ignore
    delete response.detectionDateRange;
  }
  return response;
};
exports.convertTaskAndJobFieldsToCamelCase = convertTaskAndJobFieldsToCamelCase;
const getResultAggregationQuery = (detectors, queryParams) => {
  const aggregationSort = {
    totalAnomalies: 'total_anomalies_in_24hr',
    latestAnomalyTime: 'latest_anomaly_time'
  };
  let aggsSortingOrder = {};
  if (aggregationSort[queryParams.sortField]) {
    aggsSortingOrder = {
      order: {
        [aggregationSort[queryParams.sortField]]: queryParams.sortDirection
      }
    };
  }
  return {
    size: 0,
    query: {
      bool: {
        must: [{
          terms: {
            detector_id: detectors
          }
        }, {
          range: {
            anomaly_grade: {
              gt: 0
            }
          }
        }],
        must_not: {
          exists: {
            field: 'task_id'
          }
        }
      }
    },
    aggs: {
      unique_detectors: {
        terms: {
          field: 'detector_id',
          size: queryParams.from + queryParams.size,
          ...aggsSortingOrder
        },
        aggs: {
          total_anomalies_in_24hr: {
            filter: {
              range: {
                data_start_time: {
                  gte: 'now-24h',
                  lte: 'now'
                }
              }
            }
          },
          latest_anomaly_time: {
            max: {
              field: 'data_start_time'
            }
          }
        }
      }
    }
  };
};
exports.getResultAggregationQuery = getResultAggregationQuery;
const anomalyResultMapper = anomalyResults => {
  let resultData = {
    anomalies: [],
    featureData: {}
  };
  if (anomalyResults.length === 0) return resultData;
  //initialize features list.
  const firstAnomaly = anomalyResults[0];
  Object.values(firstAnomaly.featureData).forEach(feature => {
    resultData.featureData[feature.featureId] = [];
  });
  anomalyResults.forEach(({
    featureData,
    ...rest
  }) => {
    const {
      dataStartTime,
      dataEndTime,
      ...others
    } = rest;
    resultData.anomalies.push({
      ...others,
      anomalyGrade: rest.anomalyGrade != null && rest.anomalyGrade > 0 ? Number.parseFloat(rest.anomalyGrade).toFixed(2) : 0,
      confidence: rest.anomalyGrade != null && rest.anomalyGrade > 0 ? Number.parseFloat(rest.confidence).toFixed(2) : 0,
      startTime: rest.dataStartTime,
      endTime: rest.dataEndTime,
      plotTime: rest.dataEndTime,
      ...(rest.entity !== undefined ? {
        entity: rest.entity
      } : {})
    });
    featureData.forEach(feature => {
      resultData.featureData[feature.featureId].push({
        startTime: rest.dataStartTime,
        endTime: rest.dataEndTime,
        plotTime: rest.dataEndTime,
        data: feature.data
      });
    });
  });
  return resultData;
};
exports.anomalyResultMapper = anomalyResultMapper;
const getTaskInitProgress = task => {
  if ((task === null || task === void 0 ? void 0 : task.init_progress) !== undefined) {
    return {
      percentageStr: `${((0, _lodash.get)(task, 'init_progress', 0) * 100).toFixed(0)}%`,
      estimatedMinutesLeft: task.estimated_minutes_left
    };
  }
  return undefined;
};
exports.getTaskInitProgress = getTaskInitProgress;
const getFinalDetectorStates = (detectorStateResponses, finalDetectors) => {
  let finalDetectorStates = (0, _lodash.cloneDeep)(detectorStateResponses);
  finalDetectorStates.forEach(detectorState => {
    //@ts-ignore
    detectorState.state = _constants.DETECTOR_STATE[detectorState.state];
  });

  // check if there was any failures / detectors that are unable to start
  finalDetectorStates.forEach((detectorState, i) => {
    /*
      If the error starts with 'Stopped detector', then an EndRunException was thrown.
      All EndRunExceptions are related to initialization failures except for the
      unknown prediction error which contains the message "We might have bugs".
    */
    if (detectorState.state === _constants.DETECTOR_STATE.DISABLED && detectorState.error !== undefined && detectorState.error.includes('Stopped detector')) {
      detectorState.state = detectorState.error.includes('We might have bugs') ? _constants.DETECTOR_STATE.UNEXPECTED_FAILURE : _constants.DETECTOR_STATE.INIT_FAILURE;
    }

    /*
      If a detector is disabled and has no features, set to
      a feature required state
    */
    if (detectorState.state === _constants.DETECTOR_STATE.DISABLED && finalDetectors[i].featureAttributes.length === 0) {
      detectorState.state = _constants.DETECTOR_STATE.FEATURE_REQUIRED;
    }
  });
  return finalDetectorStates;
};
exports.getFinalDetectorStates = getFinalDetectorStates;
const getDetectorsWithJob = detectorsWithJobResponses => {
  const finalDetectorsWithJobResponses = (0, _lodash.cloneDeep)(detectorsWithJobResponses);
  const resultDetectorWithJobs = [];
  finalDetectorsWithJobResponses.forEach(detectorWithJobResponse => {
    const resp = {
      ...detectorWithJobResponse.anomaly_detector,
      id: detectorWithJobResponse._id,
      primaryTerm: detectorWithJobResponse._primary_term,
      seqNo: detectorWithJobResponse._seq_no,
      adJob: {
        ...detectorWithJobResponse.anomaly_detector_job
      },
      historicalTask: {
        ...detectorWithJobResponse.anomaly_detection_task
      }
    };
    resultDetectorWithJobs.push(convertDetectorKeysToCamelCase(resp));
  });
  return resultDetectorWithJobs;
};
exports.getDetectorsWithJob = getDetectorsWithJob;
const isIndexNotFoundError = err => {
  return err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception';
};
exports.isIndexNotFoundError = isIndexNotFoundError;
const getErrorMessage = err => {
  return !(0, _lodash.isEmpty)((0, _lodash.get)(err, 'body.error.reason')) ? (0, _lodash.get)(err, 'body.error.reason') : (0, _lodash.get)(err, 'message');
};
exports.getErrorMessage = getErrorMessage;
const getDetectorTasks = detectorTaskResponses => {
  const detectorToTaskMap = {};
  detectorTaskResponses.forEach(response => {
    const detectorId = (0, _lodash.get)(response, '_id', '');
    const detectorTask = (0, _lodash.get)(response, 'anomaly_detection_task', null);
    if (detectorTask !== null) {
      detectorToTaskMap[detectorId] = detectorTask;
    }
  });
  return detectorToTaskMap;
};
exports.getDetectorTasks = getDetectorTasks;
const getDetectorResults = detectorResultsResponses => {
  const detectorToResultsMap = {};
  detectorResultsResponses.forEach(response => {
    const detectorId = (0, _lodash.get)(response, 'hits.hits.0._source.detector_id', null);
    if (detectorId !== null) {
      detectorToResultsMap[detectorId] = response;
    }
  });
  return detectorToResultsMap;
};

// Append task-related info - task state & anomaly results of the task.
// If there is no related task info for a detector: set to default values of DISABLED state and 0 anomalies
exports.getDetectorResults = getDetectorResults;
const appendTaskInfo = (detectorMap, detectorToTaskMap, detectorToResultsMap) => {
  const detectorMapWithTaskInfo = {};
  const detectorsWithTasks = Object.keys(detectorToTaskMap);
  Object.keys(detectorMap).forEach((detectorId, index) => {
    if (!detectorsWithTasks.includes(detectorId)) {
      detectorMapWithTaskInfo[detectorId] = {
        ...detectorMap[detectorId],
        curState: _constants.DETECTOR_STATE.DISABLED,
        totalAnomalies: 0
      };
    } else {
      const task = detectorToTaskMap[detectorId];
      const state = getTaskState(task);
      const totalAnomalies = (0, _lodash.get)(detectorToResultsMap[detectorId], 'hits.total.value', 0);
      detectorMapWithTaskInfo[detectorId] = {
        ...detectorMap[detectorId],
        curState: state,
        totalAnomalies: totalAnomalies
      };
    }
  });
  return detectorMapWithTaskInfo;
};

// Following checks/transformations need to be made here:
// - set to DISABLED if there is no existing task for this detector
// - set to UNEXPECTED_FAILURE if the task is in a FAILED state & the error message is unreadable / is a stack trace
// - set to INIT if the task is in a CREATED state
// - set to DISABLED if the task is in a STOPPED state
exports.appendTaskInfo = appendTaskInfo;
const getTaskState = task => {
  const state = (0, _lodash.get)(task, 'state', 'DISABLED');
  const errorMessage = processTaskError((0, _lodash.get)(task, 'error', ''));
  const updatedState = state === 'FAILED' && errorMessage.includes(_constants.STACK_TRACE_PATTERN) ? 'UNEXPECTED_FAILURE' : state === 'CREATED' ? 'INIT' : state === 'STOPPED' ? 'DISABLED' : state;
  //@ts-ignore
  return _constants.DETECTOR_STATE[updatedState];
};
exports.getTaskState = getTaskState;
const processTaskError = error => {
  const errorWithPrefixRemoved = error.replace(_constants.OPENSEARCH_EXCEPTION_PREFIX, '');
  return (0, _lodash.isEmpty)(errorWithPrefixRemoved) || errorWithPrefixRemoved.endsWith('.') ? errorWithPrefixRemoved : errorWithPrefixRemoved + '.';
};

// Filtering by 'is_latest=true' is not enough here. During backfilling of legacy
// realtime detectors on the backend, it is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.processTaskError = processTaskError;
const getLatestDetectorTasksQuery = realtime => {
  const taskTypes = realtime ? _constants.REALTIME_TASK_TYPES : _constants.HISTORICAL_TASK_TYPES;
  return {
    size: 0,
    query: {
      bool: {
        filter: [{
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    },
    aggs: {
      detectors: {
        terms: {
          field: 'detector_id',
          size: _constants.MAX_DETECTORS
        },
        aggs: {
          latest_tasks: {
            top_hits: {
              size: 1,
              sort: {
                execution_start_time: _constants.SORT_DIRECTION.DESC
              }
            }
          }
        }
      }
    }
  };
};
exports.getLatestDetectorTasksQuery = getLatestDetectorTasksQuery;
const isRealTimeTask = task => {
  return (0, _lodash.get)(task, 'task_type', '').includes(_constants.REALTIME_TASK_TYPE_PREFIX);
};
exports.isRealTimeTask = isRealTimeTask;
const getFiltersFromEntityList = entityListAsObj => {
  let filters = [];
  Object.values(entityListAsObj).forEach(entity => {
    filters.push({
      nested: {
        path: _constants.ENTITY_FIELD,
        query: {
          term: {
            [_constants.ENTITY_NAME_PATH_FIELD]: {
              value: entity.name
            }
          }
        }
      }
    });
    filters.push({
      nested: {
        path: _constants.ENTITY_FIELD,
        query: {
          term: {
            [_constants.ENTITY_VALUE_PATH_FIELD]: {
              value: entity.value
            }
          }
        }
      }
    });
  });
  return filters;
};

// Filtering by 'is_latest=true' is not enough here. During backfilling of legacy
// realtime detectors on the backend, it is possible that multiple realtime
// tasks with 'is_latest=true' are created. We sort by latest execution_start_time
// (which is equivalent to it's creation timestamp), and only return the latest one.
exports.getFiltersFromEntityList = getFiltersFromEntityList;
const getLatestTaskForDetectorQuery = (detectorId, realtime) => {
  const taskTypes = realtime ? _constants.REALTIME_TASK_TYPES : _constants.HISTORICAL_TASK_TYPES;
  return {
    size: 1,
    sort: {
      execution_start_time: _constants.SORT_DIRECTION.DESC
    },
    query: {
      bool: {
        filter: [{
          term: {
            detector_id: detectorId
          }
        }, {
          term: {
            is_latest: 'true'
          }
        }, {
          terms: {
            task_type: taskTypes
          }
        }]
      }
    }
  };
};
exports.getLatestTaskForDetectorQuery = getLatestTaskForDetectorQuery;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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