"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _lodash = require("lodash");
var _adHelpers = require("./utils/adHelpers");
var _helpers = require("../utils/helpers");
var _opensearchHelpers = require("./utils/opensearchHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_indices/{dataSourceId}', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_aliases/{dataSourceId}', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.get('/_mappings/{dataSourceId}', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.put('/create_index/{dataSourceId}', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/bulk/{dataSourceId}', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
  apiRouter.get('/_remote/info', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/{dataSourceId}', opensearchService.getClustersInfo);
  apiRouter.get('/_indices_and_aliases', opensearchService.getIndicesAndAliases);
  apiRouter.get('/_indices_and_aliases/{dataSourceId}', opensearchService.getIndicesAndAliases);
}
class OpenSearchService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index,
        clusters
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indices = [];
        let resolve_resp;
        let response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        response = response.map(item => ({
          ...item,
          localCluster: true
        }));

        // only call cat indices
        if (clusters != '') {
          if (index == '') {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':*'
            });
          } else {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':' + index
            });
          }
          indices = resolve_resp.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          response = response.concat(indices);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;

      //@ts-ignore
      const index = request.body.index;
      //@ts-ignore
      const body = request.body.body;
      const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
      try {
        await callWithRequest('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
      try {
        const response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      const body = request.body;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;
      try {
        await callWithRequest('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err);
        // Ignore the error if it's an index_not_found_exception
        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      let {
        indices
      } = request.query;
      // If indices is not an array, convert it to an array, server framework auto converts single item in string array to a string
      if (!Array.isArray(indices)) {
        indices = [indices];
      }
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let mappings = {};
        let remoteMappings = {};
        let localIndices = indices.filter(index => !index.includes(':'));
        let remoteIndices = indices.filter(index => index.includes(':'));
        if (localIndices.length > 0) {
          mappings = await callWithRequest('indices.getMapping', {
            index: localIndices
          });
        }

        // make call to fields_caps
        if (remoteIndices.length) {
          const fieldCapsResponse = await callWithRequest('fieldCaps', {
            index: remoteIndices.toString(),
            fields: '*',
            include_unmapped: true
          });
          remoteMappings = (0, _opensearchHelpers.convertFieldCapsToMappingStructure)(fieldCapsResponse);
        }
        Object.assign(mappings, remoteMappings);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: mappings
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    // we use this to retrieve indices and aliases from both the local cluster and remote clusters
    // 3 different OS APIs are called here, _cat/indices, _cat/aliases and _resolve/index
    _defineProperty(this, "getIndicesAndAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        indexOrAliasQuery,
        clusters,
        queryForLocalCluster
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indicesResponse = [];
        let aliasesResponse = [];
        if (queryForLocalCluster == 'true') {
          indicesResponse = await callWithRequest('cat.indices', {
            index: indexOrAliasQuery,
            format: 'json',
            h: 'health,index'
          });
          indicesResponse = indicesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
          aliasesResponse = await callWithRequest('cat.aliases', {
            alias: indexOrAliasQuery,
            format: 'json',
            h: 'alias,index'
          });
          aliasesResponse = aliasesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
        }

        // only call cat indices and cat aliases
        if (clusters != '') {
          let remoteIndices = [];
          let remoteAliases = [];
          let resolveResponse;
          const resolveIndexQuery = indexOrAliasQuery == '' ? clusters.split(',').map(cluster => `${cluster}:*`).join(',') : clusters.split(',').map(cluster => `${cluster}:${indexOrAliasQuery}`).join(',');
          resolveResponse = await callWithRequest('transport.request', {
            method: 'GET',
            path: '/_resolve/index/' + resolveIndexQuery
          });
          remoteIndices = resolveResponse.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          remoteAliases = resolveResponse.aliases.map(item => ({
            alias: item.name,
            index: item.indices,
            format: 'json',
            localCluster: false
          }));
          indicesResponse = indicesResponse.concat(remoteIndices);
          aliasesResponse = aliasesResponse.concat(remoteAliases);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: aliasesResponse,
              indices: indicesResponse
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: [],
                aliases: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices and aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getClustersInfo", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let clustersResponse = [];
        try {
          const remoteInfo = await callWithRequest('transport.request', {
            method: 'GET',
            path: '/_remote/info'
          });
          clustersResponse = Object.keys(remoteInfo).map(key => ({
            name: key,
            localCluster: false
          }));
        } catch (remoteErr) {
          console.warn('Failed to fetch remote cluster info, proceeding with local datasource info only.', remoteErr);
        }
        const clusterHealth = await callWithRequest('cat.health', {
          format: 'json',
          h: 'cluster'
        });
        clustersResponse.push({
          name: clusterHealth[0].cluster,
          localCluster: true
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              clusters: clustersResponse
            }
          }
        });
      } catch (err) {
        console.error('Alerting - OpensearchService - getClusterHealth:', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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