"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerForecastRoutes = registerForecastRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _forecastHelpers = require("./utils/forecastHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerForecastRoutes(apiRouter, forecastService) {
  // create forecaster
  apiRouter.post('/forecasters', forecastService.putForecaster);
  apiRouter.post('/forecasters/{dataSourceId}', forecastService.putForecaster);

  // put forecaster
  apiRouter.put('/forecasters/{forecasterId}', forecastService.putForecaster);
  apiRouter.put('/forecasters/{forecasterId}/{dataSourceId}', forecastService.putForecaster);

  // FIXME: routes not used in the UI, therefore no data source id
  apiRouter.post('/forecasters/_search', forecastService.searchForecaster);

  /**
   * Search forecast results routes
   * 
   * We use 'by-source' and 'by-index' path segments to avoid route conflicts between
   * paths with different parameter types. Without these segments, routes like:
   *   /forecasters/results/_search/{resultIndex}
   *   /forecasters/results/_search/{dataSourceId}
   * would conflict because OpenSearch can't distinguish between parameter types in the same position.
   * 
   * Current route structure:
   * 1. Search by source (no params)     : /by-source/_search
   * 2. Search by source with ID         : /by-source/{dataSourceId}/_search
   * 3. Search by index pattern          : /by-index/{resultIndex}/_search
   * 4. Search by both index and source  : /by-index/{resultIndex}/by-source/{dataSourceId}/_search
   */

  // Search with no parameters
  apiRouter.post('/forecasters/results/by-source/_search', forecastService.searchResults);

  // Search by data source ID
  apiRouter.post('/forecasters/results/by-source/{dataSourceId}/_search', forecastService.searchResults);

  // Search by result index pattern
  apiRouter.post('/forecasters/results/by-index/{resultIndex}/_search', forecastService.searchResults);

  // Search by both result index and data source ID
  apiRouter.post('/forecasters/results/by-index/{resultIndex}/by-source/{dataSourceId}/_search', forecastService.searchResults);

  // list forecasters
  apiRouter.get('/forecasters/_list', forecastService.getForecasters);
  apiRouter.get('/forecasters/_list/{dataSourceId}', forecastService.getForecasters);

  // run once forecaster
  apiRouter.post('/forecasters/{forecasterId}/_run_once', forecastService.runOnceForecaster);
  apiRouter.post('/forecasters/{forecasterId}/_run_once/{dataSourceId}', forecastService.runOnceForecaster);

  // get forecaster forecast results
  apiRouter.get('/forecasters/{id}/results/{isRunOnce}/{resultIndex}', forecastService.getForecastResults);
  apiRouter.get('/forecasters/{id}/results/{isRunOnce}/{resultIndex}/{dataSourceId}', forecastService.getForecastResults);

  // delete forecaster
  apiRouter.delete('/forecasters/{forecasterId}', forecastService.deleteForecaster);
  apiRouter.delete('/forecasters/{forecasterId}/{dataSourceId}', forecastService.deleteForecaster);

  // start forecaster
  apiRouter.post('/forecasters/{forecasterId}/start', forecastService.startForecaster);
  apiRouter.post('/forecasters/{forecasterId}/start/{dataSourceId}', forecastService.startForecaster);

  // stop forecaster
  apiRouter.post('/forecasters/{forecasterId}/stop', forecastService.stopForecaster);
  apiRouter.post('/forecasters/{forecasterId}/stop/{dataSourceId}', forecastService.stopForecaster);
  apiRouter.get('/forecasters/{forecasterId}/_profile', forecastService.getForecasterProfile);

  // get forecaster
  apiRouter.get('/forecasters/{forecasterId}', forecastService.getForecaster);
  apiRouter.get('/forecasters/{forecasterId}/{dataSourceId}', forecastService.getForecaster);

  // match forecaster
  apiRouter.get('/forecasters/{forecasterName}/_match', forecastService.matchForecaster);
  apiRouter.get('/forecasters/{forecasterName}/_match/{dataSourceId}', forecastService.matchForecaster);

  // get forecaster count
  apiRouter.get('/forecasters/_count', forecastService.getForecasterCount);
  apiRouter.get('/forecasters/_count/{dataSourceId}', forecastService.getForecasterCount);

  // post get top forecast results
  apiRouter.post('/forecasters/{forecasterId}/_topForecasts/{isRunOnce}', forecastService.getTopForecastResults);
  apiRouter.post('/forecasters/{forecasterId}/_topForecasts/{isRunOnce}/{dataSourceId}', forecastService.getTopForecastResults);

  // validate forecaster
  apiRouter.post('/forecasters/_validate/{validationType}', forecastService.validateForecaster);
  apiRouter.post('/forecasters/_validate/{validationType}/{dataSourceId}', forecastService.validateForecaster);

  // suggest forecaster
  apiRouter.post('/forecasters/_suggest/{suggestType}', forecastService.suggestForecaster);
  apiRouter.post('/forecasters/_suggest/{suggestType}/{dataSourceId}', forecastService.suggestForecaster);
}
class ForecastService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "deleteForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        // dataSourceId will be "" and fall back to use the existing client for local cluster
        // On the other hand, in MDS world, the open search legacy client (this.client) will
        // be undefined and it will pickup the data source client 
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.deleteForecaster', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - deleteForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "runOnceForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId = '',
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.runOnceForecaster', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            // return taskId
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - runOnceForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;

        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        let params = {
          forecasterId: forecasterId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await callWithRequest('forecast.updateForecaster', params);
        } else {
          response = await callWithRequest('forecast.createForecaster', {
            body: params.body
          });
        }
        const resp = {
          ...response.forecaster,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _forecastHelpers.convertForecastKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Forecast - PutForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        const response = await callWithRequest('forecast.validateForecaster', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - validateForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "suggestForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          suggestType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        const response = await callWithRequest('forecast.suggestForecaster', {
          body: requestBody,
          suggestType: suggestType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - suggestForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const forecasterResponse = await callWithRequest('forecast.getForecaster', {
          forecasterId
        });

        // Populating static forecaster fields
        const staticFields = {
          id: forecasterResponse._id,
          primaryTerm: forecasterResponse._primary_term,
          seqNo: forecasterResponse._seq_no,
          // the backend returns a response with forecaster field.
          ...(0, _forecastHelpers.convertStaticFieldsToCamelCase)(forecasterResponse.forecaster)
        };

        // Get real-time and run-once task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no forecaster jobs have been ran on a new cluster.
        // let realtimeTasksResponse = {} as any;
        // let runOnceTasksResponse = {} as any;
        // try {
        //   const callWithRequest = getClientBasedOnDataSource(
        //     context,
        //     this.dataSourceEnabled,
        //     request,
        //     dataSourceId,
        //     this.client
        //   );

        //   realtimeTasksResponse = await callWithRequest('forecast.searchTasks', {
        //     body: getLatestTaskForForecasterQuery(forecasterId, true),
        //   });

        //   runOnceTasksResponse = await callWithRequest('forecast.searchTasks', {
        //     body: getLatestTaskForForecasterQuery(forecasterId, false),
        //   });
        // } catch (err) {
        //   if (!isIndexNotFoundError(err)) {
        //     throw err;
        //   }
        // }

        // const realtimeTask = get(
        //   get(realtimeTasksResponse, 'hits.hits', []).map((taskResponse: any) => {
        //     return {
        //       id: get(taskResponse, '_id'),
        //       ...get(taskResponse, '_source'),
        //     };
        //   }),
        //   0
        // );
        // const runOnceTask = get(
        //   get(runOnceTasksResponse, 'hits.hits', []).map(
        //     (taskResponse: any) => {
        //       return {
        //         id: get(taskResponse, '_id'),
        //         ...get(taskResponse, '_source'),
        //       };
        //     }
        //   ),
        //   0
        // );

        const taskAndJobFields = (0, _forecastHelpers.convertTaskAndJobFieldsToCamelCase)(forecasterResponse.realtime_task, forecasterResponse.run_once_task, forecasterResponse.forecaster_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        // if the forecaster is not found (e.g., deleted while on the detail page), return an empty response
        // this is to avoid the error message from the frontend where the forecaster is not found
        // the error is triggered by useEffect of useFetchForecasterInfo in ForecasterDetail
        if (err.statusCode === 404) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {}
            }
          });
        }
        console.log('Forecast - Unable to get forecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          forecasterId: forecasterId
        };
        let requestPath = 'forecast.startForecaster';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - startForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        // Extract required parameters with specific type assertion.
        // 'forecasterId' is expected to always be present in the route path.
        let {
          forecasterId
        } = request.params;
        // Extract optional parameters separately.
        // 'dataSourceId' might be missing from the route path (hence '?').
        // Provide a default value ('') if it's not present using destructuring default assignment.
        const {
          dataSourceId = ''
        } = request.params;
        const requestPath = 'forecast.stopForecaster';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          forecasterId: forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - stopForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasterProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('forecast.forecasterProfile', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Forecast - forecasterProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('forecast.searchForecaster', {
          body: requestBody
        });
        const totalForecasters = (0, _lodash.get)(response, 'hits.total.value', 0);
        const forecasters = (0, _lodash.get)(response, 'hits.hits', []).map(forecaster => ({
          ...(0, _forecastHelpers.convertForecastKeysToCamelCase)(forecaster._source),
          id: forecaster._id,
          seqNo: forecaster._seq_no,
          primaryTerm: forecaster._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalForecasters,
              forecasters
            }
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to search forecasters', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasters: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasters", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.searchForecaster', {
          body: {
            size: _constants.MAX_FORECASTER
          }
        });
        const totalForecasters = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all forecasters from search forecaster API
        const allForecasters = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, forecasterResponse) => ({
          ...acc,
          [forecasterResponse._id]: {
            id: forecasterResponse._id,
            primaryTerm: forecasterResponse._primary_term,
            seqNo: forecasterResponse._seq_no,
            ...(0, _forecastHelpers.convertStaticFieldsToCamelCase)(forecasterResponse._source)
          }
        }), {});

        // Fetch the latest realtime and runOnce tasks for all forecasters
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no forecaster jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let runOnceTasksResponse = {};
        try {
          realtimeTasksResponse = await callWithRequest('forecast.searchTasks', {
            body: (0, _forecastHelpers.getLatestForecasterTasksQuery)(true)
          });
          runOnceTasksResponse = await callWithRequest('forecast.searchTasks', {
            body: (0, _forecastHelpers.getLatestForecasterTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _forecastHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.forecasters.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const runOnceTasks = (0, _lodash.get)(runOnceTasksResponse, 'aggregations.forecasters.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              runOnceTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and runOnce task info by looping through each forecaster & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        const forecastersArray = Object.values(allForecasters);
        forecastersArray.forEach(forecaster => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[forecaster.id], 'realtimeTask._source');
          const runOnceTask = (0, _lodash.get)(runOnceTasks[forecaster.id], 'runOnceTask._source');
          forecaster.curState = (0, _forecastHelpers.combineTaskState)(realtimeTask, runOnceTask);
          forecaster.realTimeLastUpdateTime = (0, _lodash.get)(realtimeTask, 'last_update_time');
          forecaster.runOnceLastUpdateTime = (0, _lodash.get)(runOnceTask, 'last_update_time');
          forecaster.stateError = (0, _lodash.get)(realtimeTask, 'error') || (0, _lodash.get)(runOnceTask, 'error');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalForecasters: totalForecasters,
              forecasterList: forecastersArray
            }
          }
        });
      } catch (err) {
        console.log('Forecaster - Unable to search forecasters', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasterList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecastResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isRunOnce,
        resultIndex
      } = request.params;
      const {
        dataSourceId = ''
      } = request.params;
      if (!resultIndex) {
        // Not strictly querying custom ⇒ default to ''
        resultIndex = '';
      } else if (!resultIndex.startsWith(_constants.CUSTOM_FORECAST_RESULT_INDEX_PREFIX)) {
        // If resultIndex is given but not valid, revert to default
        resultIndex = '';
      }
      isRunOnce = JSON.parse(isRunOnce);

      // Search by task id if runOnce, or by forecaster id if realtime
      const searchTerm = isRunOnce ? {
        task_id: id
      } : {
        forecaster_id: id
      };
      try {
        const {
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.FORECASTER_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          entityList = '',
          dawnEpoch = 0,
          maxEntities = 0
        } = request.query;

        //Allowed sorting columns
        const sortQueryMap = {
          [_constants.FORECASTER_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.FORECASTER_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.FORECASTER_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.FORECASTER_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a runOnce result from a runOnce task.
        if (!isRunOnce) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          // Get current number of filters to determine the index for adding new date range filter
          // This includes the initial term filter and any entity filters that were added
          let filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
          filterSize = requestBody.query.bool.filter.length;

          // Add dawnEpoch filter if it exists
          if (dawnEpoch > 0) {
            (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${_constants.FORECASTER_DOC_FIELDS.EXECUTION_END_TIME}.gte`, dawnEpoch);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }

        // ─────────────────────────────────────────────────────────────
        // If maxEntities > 0, find top N entity_ids.
        // ─────────────────────────────────────────────────────────────
        let restrictedEntityIds = [];
        if (maxEntities > 0) {
          const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
          const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? {} : (0, _forecastHelpers.buildEntityListQuery)(entityListAsObj);

          // Only clone and modify requestBody if entityFilters exists and is not empty/null
          let queryForAggregation;
          if (entityFilters && typeof entityFilters === 'object' && Object.keys(entityFilters).length > 0) {
            // Create a deep clone of the request body
            const clonedRequestBody = JSON.parse(JSON.stringify(requestBody));

            // Add entity filters to the filter array of the cloned request body
            if (!clonedRequestBody.query) {
              clonedRequestBody.query = {
                bool: {
                  filter: []
                }
              };
            } else if (!clonedRequestBody.query.bool) {
              clonedRequestBody.query.bool = {
                filter: []
              };
            } else if (!clonedRequestBody.query.bool.filter) {
              clonedRequestBody.query.bool.filter = [];
            }

            // Add the entity filter object to the filter array
            clonedRequestBody.query.bool.filter.push(entityFilters);
            queryForAggregation = clonedRequestBody.query;
          } else {
            // Use the original requestBody if no entity filters to add
            queryForAggregation = requestBody.query;
          }

          // Example aggregatorRequestBody:
          // {
          //   "size": 0,
          //   "query": {
          //     "bool": {
          //       "filter": [
          //         {"term": {"task_id": "BsLQbZUBxkwQb14j93bF"}},
          //         {"range": {"execution_end_time": {"gte": "0"}}},
          //         {
          //           "bool": {
          //             "should": [
          //               {
          //                 "bool": {
          //                   "must": [
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "service"}}, {"term": {"entity.value": "app_6"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     },
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "host"}}, {"term": {"entity.value": "server_3"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     }
          //                   ]
          //                 }
          //               },
          //               {
          //                 "bool": {
          //                   "must": [
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "service"}}, {"term": {"entity.value": "app_6"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     },
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "host"}}, {"term": {"entity.value": "server_1"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     }
          //                   ]
          //                 }
          //               }
          //             ],
          //             "minimum_should_match": 1
          //           }
          //         }
          //       ]
          //     }
          //   },
          //   "aggs": {
          //     "top_entities": {
          //       "terms": {
          //         "field": "entity_id",
          //         "size": 5,
          //         "order": {"_count": "desc"}
          //       }
          //     }
          //   }
          // }

          // Now use the appropriate query in aggregatorRequestBody
          const aggregatorRequestBody = {
            size: 0,
            query: queryForAggregation,
            aggs: {
              top_entities: {
                terms: {
                  field: 'entity_id',
                  size: maxEntities,
                  order: {
                    _count: 'desc'
                  }
                }
              }
            }
          };

          // We'll call the same or custom search method:
          const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
          const aggResponse = !resultIndex ? await callWithRequest('forecast.searchResults', {
            body: aggregatorRequestBody
          }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
            resultIndex: resultIndex,
            body: aggregatorRequestBody
          });

          // Extract top entity_ids
          const topEntityBuckets = (0, _lodash.get)(aggResponse, 'aggregations.top_entities.buckets', []);
          restrictedEntityIds = topEntityBuckets.map(b => b.key);

          // If no entities matched, return empty
          if (!restrictedEntityIds.length) {
            return opensearchDashboardsResponse.ok({
              body: {
                ok: true,
                response: {
                  totalResults: 0,
                  results: []
                }
              }
            });
          }
        }

        // ─────────────────────────────────────────────────────────────
        // Add a terms filter to restrict final hits if we have top entities
        // ─────────────────────────────────────────────────────────────
        if (restrictedEntityIds.length > 0) {
          requestBody.query.bool.filter.push({
            terms: {
              entity_id: restrictedEntityIds
            }
          });
        }
        let requestParams = {
          resultIndex: resultIndex
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);

        // Add pagination with search_after
        let allResults = [];
        let lastSort = null;
        let hasMoreResults = true;
        let totalHits = 0;

        // Create a copy of your existing requestBody to use in the pagination loop
        const paginatedRequestBody = {
          ...requestBody,
          "track_total_hits": true // Add this to ensure accurate total count for large result sets
        };

        // Add sort if not already present
        if (!paginatedRequestBody.sort) {
          paginatedRequestBody.sort = [{
            [sortField]: sortDirection.toLowerCase()
          }, {
            "_id": "asc"
          } // Secondary sort for tiebreaker
          ];
        }

        // Execute paginated search
        while (hasMoreResults) {
          // Add search_after for subsequent pages
          if (lastSort) {
            paginatedRequestBody.search_after = lastSort;
          }

          // Your existing API call, but with our paginated request body
          const response = !resultIndex ? await callWithRequest('forecast.searchResults', {
            body: paginatedRequestBody
          }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
            ...requestParams,
            body: paginatedRequestBody
          });
          const hits = (0, _lodash.get)(response, 'hits.hits', []);

          // Track total hits from first page
          if (!lastSort) {
            totalHits = (0, _lodash.get)(response, 'hits.total.value', 0);
          }
          if (hits.length === 0 || hits.length < size) {
            hasMoreResults = false;
          }
          if (hits.length > 0) {
            // Save sort values from last hit for next iteration
            lastSort = hits[hits.length - 1].sort;

            // Collect results
            allResults.push(...hits);
          }
        }
        const groupedResults = new Map();
        allResults.forEach(result => {
          const source = result._source;
          const key = `${source.forecaster_id}|${source.entity_id || 'default'}|${source.data_end_time}`;
          if (!groupedResults.has(key)) {
            groupedResults.set(key, {
              featureData: null,
              forecasts: []
            });
          }
          if (source.feature_data) {
            groupedResults.get(key).featureData = result;
          } else {
            groupedResults.get(key).forecasts.push(result);
          }
        });
        const forecastResult = [];

        // Process each group
        groupedResults.forEach(({
          featureData,
          forecasts
        }) => {
          if (!featureData) return; // Skip if no feature data found

          // Check if any forecast has horizon_index
          const hasHorizonIndex = forecasts.some(forecast => forecast._source.horizon_index != null);
          if (hasHorizonIndex) {
            // Sort forecasts by horizon_index and combine into arrays
            const sortedForecasts = (0, _lodash.orderBy)(forecasts, ['_source.horizon_index'], ['asc']);
            const forecastValues = [];
            const forecastLowerBounds = [];
            const forecastUpperBounds = [];
            const forecastStartTimes = [];
            const forecastEndTimes = [];
            sortedForecasts.forEach(forecast => {
              const source = forecast._source;
              forecastValues.push(source.forecast_value != null && source.forecast_value !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_value)) : 0);
              forecastLowerBounds.push(source.forecast_lower_bound != null && source.forecast_lower_bound !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_lower_bound)) : 0);
              forecastUpperBounds.push(source.forecast_upper_bound != null && source.forecast_upper_bound !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_upper_bound)) : 0);
              forecastStartTimes.push(source.forecast_data_start_time);
              forecastEndTimes.push(source.forecast_data_end_time);
            });
            forecastResult.push({
              startTime: featureData._source.data_start_time,
              endTime: featureData._source.data_end_time,
              plotTime: featureData._source.data_end_time,
              forecastValue: forecastValues,
              forecastLowerBound: forecastLowerBounds,
              forecastUpperBound: forecastUpperBounds,
              forecastStartTime: forecastStartTimes,
              forecastEndTime: forecastEndTimes,
              ...(featureData._source.entity != null ? {
                entity: featureData._source.entity,
                entityId: featureData._source.entity_id
              } : {}),
              features: this.getFeatureData(featureData)
            });
          } else {
            // Direct push for single forecasts without horizon_index
            forecastResult.push({
              startTime: featureData._source.data_start_time,
              endTime: featureData._source.data_end_time,
              plotTime: featureData._source.data_end_time,
              forecastValue: [],
              forecastLowerBound: [],
              forecastUpperBound: [],
              forecastStartTime: [],
              forecastEndTime: [],
              ...(featureData._source.entity != null ? {
                entity: featureData._source.entity,
                entityId: featureData._source.entity_id
              } : {}),
              features: this.getFeatureData(featureData)
            });
          }
        });

        // Sort final results by plotTime
        const sortedForecastResult = (0, _lodash.orderBy)(forecastResult, ['plotTime'], ['asc']);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalResults: totalHits,
              results: sortedForecastResult
            }
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopForecastResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          forecasterId,
          isRunOnce
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isRunOnce = JSON.parse(isRunOnce);
        const requestPath = 'forecast.topForecastResults';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);

        // Define proper types for OpenSearch query structures

        // Extract the query parameters from the request body with defaults
        const {
          split_by = '',
          filter_by = '',
          build_in_query = '',
          forecast_from = 0,
          threshold = 0,
          relation_to_threshold = '',
          filter_query = {},
          subaggregations = []
        } = request.body || {};

        // Build query object with appropriate parameters
        const queryBody = {};

        // Add split_by if present
        if (split_by) {
          queryBody.split_by = split_by;
        }

        // Add filter_by and related parameters
        if (filter_by) {
          queryBody.filter_by = filter_by;
          if (filter_by === 'BUILD_IN_QUERY' && build_in_query) {
            queryBody.build_in_query = build_in_query;

            // Add threshold parameters if build_in_query is DISTANCE_TO_THRESHOLD_VALUE
            if (build_in_query === 'DISTANCE_TO_THRESHOLD_VALUE') {
              queryBody.threshold = threshold;
              queryBody.relation_to_threshold = relation_to_threshold;
            }
          } else if (filter_by === 'CUSTOM_QUERY') {
            // Add custom query parameters - check if the objects are not empty
            if (Object.keys(filter_query).length > 0) {
              queryBody.filter_query = filter_query;
            }
            if (subaggregations.length > 0) {
              queryBody.subaggregations = subaggregations;
            }
          }
        }

        // Add forecast_from timestamp if present
        if (forecast_from) {
          queryBody.forecast_from = forecast_from;
        }

        // Add run_once to body if isRunOnce is true
        const requestBody = {
          ...queryBody,
          ...(isRunOnce && {
            run_once: true
          })
        };
        const response = await callWithRequest(requestPath, {
          forecasterId: forecasterId,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - getTopForecastResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterName
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.matchForecaster', {
          forecasterName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - matchForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasterCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.forecasterCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - getForecasterCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name
        };
      });
      return featureResult;
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_FORECAST_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search forecast result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('forecast.searchResults', {
          body: requestBody
        }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to search forecast result', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasters: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = ForecastService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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