"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerADRoutes = registerADRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _adHelpers = require("./utils/adHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerADRoutes(apiRouter, adService) {
  // create detector
  apiRouter.post('/detectors', adService.putDetector);
  apiRouter.post('/detectors/{dataSourceId}', adService.putDetector);

  // put detector
  apiRouter.put('/detectors/{detectorId}', adService.putDetector);
  apiRouter.put('/detectors/{detectorId}/{dataSourceId}', adService.putDetector);

  // routes not used in the UI, therefore no data source id
  apiRouter.post('/detectors/_search', adService.searchDetector);

  // post search anomaly results
  apiRouter.post('/detectors/results/_search', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{dataSourceId}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}', adService.searchResults);
  apiRouter.post('/detectors/results/_search/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.searchResults);

  // list detectors
  apiRouter.get('/detectors/_list', adService.getDetectors);
  apiRouter.get('/detectors/_list/{dataSourceId}', adService.getDetectors);

  // preview detector
  apiRouter.post('/detectors/preview', adService.previewDetector);
  apiRouter.post('/detectors/preview/{dataSourceId}', adService.previewDetector);

  // suggest detector
  apiRouter.post('/detectors/_suggest/{suggestType}', adService.suggestDetector);
  apiRouter.post('/detectors/_suggest/{suggestType}/{dataSourceId}', adService.suggestDetector);

  // get detector anomaly results
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{resultIndex}/{onlyQueryCustomResultIndex}/{dataSourceId}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}', adService.getAnomalyResults);
  apiRouter.get('/detectors/{id}/results/{isHistorical}/{dataSourceId}', adService.getAnomalyResults);

  // delete detector
  apiRouter.delete('/detectors/{detectorId}', adService.deleteDetector);
  apiRouter.delete('/detectors/{detectorId}/{dataSourceId}', adService.deleteDetector);

  // start detector
  apiRouter.post('/detectors/{detectorId}/start', adService.startDetector);
  apiRouter.post('/detectors/{detectorId}/start/{dataSourceId}', adService.startDetector);

  // stop detector
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}', adService.stopDetector);
  apiRouter.post('/detectors/{detectorId}/stop/{isHistorical}/{dataSourceId}', adService.stopDetector);
  apiRouter.get('/detectors/{detectorId}/_profile', adService.getDetectorProfile);

  // get detector
  apiRouter.get('/detectors/{detectorId}', adService.getDetector);
  apiRouter.get('/detectors/{detectorId}/{dataSourceId}', adService.getDetector);

  // match detector
  apiRouter.get('/detectors/{detectorName}/_match', adService.matchDetector);
  apiRouter.get('/detectors/{detectorName}/_match/{dataSourceId}', adService.matchDetector);

  // get detector count
  apiRouter.get('/detectors/_count', adService.getDetectorCount);
  apiRouter.get('/detectors/_count/{dataSourceId}', adService.getDetectorCount);

  // post get top anomaly results
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}', adService.getTopAnomalyResults);
  apiRouter.post('/detectors/{detectorId}/_topAnomalies/{isHistorical}/{dataSourceId}', adService.getTopAnomalyResults);

  // validate detector
  apiRouter.post('/detectors/_validate/{validationType}', adService.validateDetector);
  apiRouter.post('/detectors/_validate/{validationType}/{dataSourceId}', adService.validateDetector);

  // start insights job
  apiRouter.post('/insights/_start', adService.startInsights);
  apiRouter.post('/insights/_start/{dataSourceId}', adService.startInsights);

  // stop insights job
  apiRouter.post('/insights/_stop', adService.stopInsights);
  apiRouter.post('/insights/_stop/{dataSourceId}', adService.stopInsights);

  // get insights status
  apiRouter.get('/insights/_status', adService.getInsightsStatus);
  apiRouter.get('/insights/_status/{dataSourceId}', adService.getInsightsStatus);

  // get insights results
  apiRouter.get('/insights/_results', adService.getInsightsResults);
  apiRouter.get('/insights/_results/{dataSourceId}', adService.getInsightsResults);
}
class AdService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "deleteDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.deleteDetector', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - deleteDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "previewDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.previewDetector', {
          body: requestBody
        });
        const transformedKeys = (0, _helpers.mapKeysDeep)(response, _helpers.toCamel);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            //@ts-ignore
            response: (0, _adHelpers.anomalyResultMapper)(transformedKeys.anomalyResult)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - previewDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "suggestDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          suggestType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        const response = await callWithRequest('ad.suggestDetector', {
          body: requestBody,
          suggestType: suggestType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - suggestDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;

        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _adHelpers.convertDetectorKeysToSnakeCase)(request.body));
        let params = {
          detectorId: detectorId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await callWithRequest('ad.updateDetector', params);
        } else {
          response = await callWithRequest('ad.createDetector', {
            body: params.body
          });
        }
        const resp = {
          ...response.anomaly_detector,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _adHelpers.convertDetectorKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Anomaly detector - PutDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _adHelpers.convertPreviewInputKeysToSnakeCase)(request.body));
        const response = await callWithRequest('ad.validateDetector', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - validateDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startInsights", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.startInsights', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response,
            message: 'Insights job started successfully'
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startInsights', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopInsights", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.stopInsights', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response,
            message: 'Insights job stopped successfully'
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopInsights', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getInsightsStatus", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.getInsightsStatus', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getInsightsStatus', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getInsightsResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.getInsightsResults', {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getInsightsResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const detectorResponse = await callWithRequest('ad.getDetector', {
          detectorId
        });

        // Populating static detector fields
        const staticFields = {
          id: detectorResponse._id,
          primaryTerm: detectorResponse._primary_term,
          seqNo: detectorResponse._seq_no,
          ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse.anomaly_detector)
        };

        // Get real-time and historical task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestTaskForDetectorQuery)(detectorId, false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTask = (0, _lodash.get)((0, _lodash.get)(realtimeTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const historicalTask = (0, _lodash.get)((0, _lodash.get)(historicalTasksResponse, 'hits.hits', []).map(taskResponse => {
          return {
            id: (0, _lodash.get)(taskResponse, '_id'),
            ...(0, _lodash.get)(taskResponse, '_source')
          };
        }), 0);
        const taskAndJobFields = (0, _adHelpers.convertTaskAndJobFieldsToCamelCase)(realtimeTask, historicalTask, detectorResponse.anomaly_detector_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          detectorId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          detectorId: detectorId
        };
        let requestPath = 'ad.startDetector';
        // If a start and end time are passed: we want to start a historical analysis
        if ((0, _lodash.isNumber)(startTime) && (0, _lodash.isNumber)(endTime)) {
          requestParams = {
            ...requestParams,
            body: {
              start_time: startTime,
              end_time: endTime
            }
          };
          requestPath = 'ad.startHistoricalDetector';
        }
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - startDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.stopHistoricalDetector' : 'ad.stopDetector';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - stopDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.detectorProfile', {
          detectorId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - detectorProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectors = (0, _lodash.get)(response, 'hits.hits', []).map(detector => ({
          ...(0, _adHelpers.convertDetectorKeysToCamelCase)(detector._source),
          id: detector._id,
          seqNo: detector._seq_no,
          primaryTerm: detector._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectors
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex,
          onlyQueryCustomResultIndex
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search anomaly result', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectors: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectors", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          from = 0,
          size = 20,
          search = '',
          indices = '',
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = 'name',
          dataSourceId = ''
        } = request.query;
        const mustQueries = [];
        if (search.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['name', 'description'],
              default_operator: 'AND',
              query: `*${search.trim().split('-').join('* *')}*`
            }
          });
        }
        if (indices.trim()) {
          mustQueries.push({
            query_string: {
              fields: ['indices'],
              default_operator: 'AND',
              query: `*${indices.trim().split('-').join('* *')}*`
            }
          });
        }
        //Allowed sorting columns
        const sortQueryMap = {
          name: {
            'name.keyword': sortDirection
          },
          indices: {
            'indices.keyword': sortDirection
          },
          lastUpdateTime: {
            last_update_time: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }
        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              must: mustQueries
            }
          }
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.searchDetector', {
          body: requestBody
        });
        const totalDetectors = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all detectors from search detector API
        const allDetectors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, detectorResponse) => ({
          ...acc,
          [detectorResponse._id]: {
            id: detectorResponse._id,
            primaryTerm: detectorResponse._primary_term,
            seqNo: detectorResponse._seq_no,
            ...(0, _adHelpers.convertStaticFieldsToCamelCase)(detectorResponse._source)
          }
        }), {});
        const allDetectorsMap = Object.values(allDetectors).reduce((acc, detector) => ({
          ...acc,
          [detector.id]: detector
        }), {});

        //Given each detector from previous result, get aggregation to power list
        const allDetectorIds = Object.keys(allDetectorsMap);
        let requestParams = {
          // If specifying result index, will query anomaly result from both default and custom result indices.
          // If no valid result index specified, just query anomaly result from default result index.
          // Here we specify custom AD result index prefix pattern to query all custom result indices.
          resultIndex: _constants.CUSTOM_AD_RESULT_INDEX_PREFIX + '*',
          onlyQueryCustomResultIndex: 'false'
        };
        const aggregationResult = await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: (0, _adHelpers.getResultAggregationQuery)(allDetectorIds, {
            from,
            size,
            sortField,
            sortDirection,
            search,
            indices
          })
        });
        const aggsDetectors = (0, _lodash.get)(aggregationResult, 'aggregations.unique_detectors.buckets', []).reduce((acc, agg) => {
          return {
            ...acc,
            [agg.key]: {
              ...allDetectorsMap[agg.key],
              totalAnomalies: agg.total_anomalies_in_24hr.doc_count,
              lastActiveAnomaly: agg.latest_anomaly_time.value
            }
          };
        }, {});

        // Aggregation will not return values where anomalies for detectors are not generated, loop through it and fill values with 0
        const unUsedDetectors = (0, _lodash.pullAll)(allDetectorIds, Object.keys(aggsDetectors)).reduce((acc, unusedDetector) => {
          return {
            ...acc,
            [unusedDetector]: {
              ...allDetectorsMap[unusedDetector],
              totalAnomalies: 0,
              lastActiveAnomaly: 0
            }
          };
        }, {});

        // If sorting criteria is from the aggregation manage pagination in memory.
        let finalDetectors = (0, _lodash.orderBy)({
          ...aggsDetectors,
          ...unUsedDetectors
        }, [sortField], [sortDirection]);
        if (!sortQueryMap[sortField]) {
          finalDetectors = Object.values(finalDetectors).slice(from, from + size).reduce((acc, detector) => ({
            ...acc,
            [detector.id]: detector
          }), {});
        }

        // Fetch the latest realtime and historical tasks for all detectors
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no detector jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let historicalTasksResponse = {};
        try {
          realtimeTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(true)
          });
          historicalTasksResponse = await callWithRequest('ad.searchTasks', {
            body: (0, _adHelpers.getLatestDetectorTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const historicalTasks = (0, _lodash.get)(historicalTasksResponse, 'aggregations.detectors.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              historicalTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and historical task info by looping through each detector & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        finalDetectors.forEach(detector => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[detector.id], 'realtimeTask._source');
          detector.curState = (0, _adHelpers.getTaskState)(realtimeTask);
          detector.enabledTime = (0, _lodash.get)(realtimeTask, 'execution_start_time');
          detector.taskId = (0, _lodash.get)(historicalTasks[detector.id], 'historicalTask._id');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalDetectors,
              detectorList: Object.values(finalDetectors)
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to search detectors', err);
        if ((0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalDetectors: 0,
                detectorList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isHistorical,
        resultIndex,
        onlyQueryCustomResultIndex
      } = request.params;
      const {
        dataSourceId = ''
      } = request.params;
      if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_AD_RESULT_INDEX_PREFIX)) {
        // Set resultIndex as '' means no custom result index specified, will only search anomaly result from default index.
        resultIndex = '';
      }
      isHistorical = JSON.parse(isHistorical);

      // Search by task id if historical, or by detector id if realtime
      const searchTerm = isHistorical ? {
        task_id: id
      } : {
        detector_id: id
      };
      try {
        const {
          from = 0,
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.AD_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          anomalyThreshold = -1,
          entityList = ''
        } = request.query;
        const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
        const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? [] : (0, _adHelpers.getFiltersFromEntityList)(entityListAsObj);

        //Allowed sorting columns
        const sortQueryMap = {
          anomalyGrade: {
            anomaly_grade: sortDirection
          },
          confidence: {
            confidence: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.AD_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.AD_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          from,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }, {
                range: {
                  anomaly_grade: {
                    gt: anomalyThreshold
                  }
                }
              }, ...entityFilters]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a historical result from a historical task.
        if (!isHistorical) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          const filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }
        let requestParams = {
          resultIndex: resultIndex,
          onlyQueryCustomResultIndex: onlyQueryCustomResultIndex
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('ad.searchResults', {
          body: requestBody
        }) : await callWithRequest('ad.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        const totalResults = (0, _lodash.get)(response, 'hits.total.value', 0);
        const detectorResult = [];
        const featureResult = {};
        (0, _lodash.get)(response, 'hits.hits', []).forEach(result => {
          detectorResult.push({
            startTime: result._source.data_start_time,
            endTime: result._source.data_end_time,
            plotTime: result._source.data_end_time,
            contributions: result._source.anomaly_grade > 0 ? result._source.relevant_attribution : {},
            confidence: result._source.confidence != null && result._source.confidence !== 'NaN' && result._source.confidence > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.confidence)) : 0,
            anomalyGrade: result._source.anomaly_grade != null && result._source.anomaly_grade !== 'NaN' && result._source.anomaly_grade > 0 ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(result._source.anomaly_grade)) : 0,
            ...(result._source.entity != null ? {
              entity: result._source.entity
            } : {}),
            // TODO: we should refactor other places to read feature data from
            // AnomalyResult, instead of having separate FeatureData which is hard
            // to know feature data belongs to which anomaly result
            features: this.getFeatureData(result)
          });
          result._source.feature_data.forEach(featureData => {
            if (!featureResult[featureData.feature_id]) {
              featureResult[featureData.feature_id] = [];
            }
            featureResult[featureData.feature_id].push({
              startTime: result._source.data_start_time,
              endTime: result._source.data_end_time,
              plotTime: result._source.data_end_time,
              data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
              name: featureData.feature_name,
              expectedValue: this.getExpectedValue(result, featureData)
            });
          });
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalAnomalies: totalResults,
              results: detectorResult,
              featureResults: featureResult
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopAnomalyResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          detectorId,
          isHistorical
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isHistorical = JSON.parse(isHistorical);
        const requestPath = isHistorical ? 'ad.topHistoricalAnomalyResults' : 'ad.topAnomalyResults';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          detectorId: detectorId,
          body: request.body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getTopAnomalyResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchDetector", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          detectorName
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.matchDetector', {
          detectorName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - matchDetector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getDetectorCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('ad.detectorCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Anomaly detector - getDetectorCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name,
          expectedValue: this.getExpectedValue(rawResult, featureData)
        };
      });
      return featureResult;
    });
    _defineProperty(this, "getExpectedValue", (rawResult, featureData) => {
      let expectedValue = featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForAnomaly)(Number.parseFloat(featureData.data)) : 0;
      if (rawResult._source.anomaly_grade > 0) {
        const expectedValueList = rawResult._source.expected_values;
        if ((expectedValueList === null || expectedValueList === void 0 ? void 0 : expectedValueList.length) > 0) {
          expectedValueList[0].value_list.forEach(expect => {
            if (expect.feature_id === featureData.feature_id) {
              expectedValue = expect.data;
            }
          });
        }
      }
      return expectedValue;
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = AdService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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