"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _constants = require("../../utils/constants");
var _helpers = require("./utils/helpers");
var _MDSEnabledClientService = require("./MDSEnabledClientService");
var _constants2 = require("./utils/constants");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * Copyright OpenSearch Contributors
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          */
class MonitorService extends _MDSEnabledClientService.MDSEnabledClientService {
  constructor(...args) {
    super(...args);
    _defineProperty(this, "createMonitor", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "createWorkflow", async (context, req, res) => {
      try {
        const params = {
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const createResponse = await client('alerting.createWorkflow', params);
        return res.ok({
          body: {
            ok: true,
            resp: createResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - createWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteMonitor', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "deleteWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const response = await client('alerting.deleteWorkflow', params);
        return res.ok({
          body: {
            ok: response.result === 'deleted' || response.result === undefined
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - deleteWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          headers: _constants2.DEFAULT_HEADERS
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitor', params);
        let monitor = _lodash.default.get(getResponse, 'monitor', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        const associated_workflows = _lodash.default.get(getResponse, 'associated_workflows', null);
        if (monitor) {
          const client = this.getClientBasedOnDataSource(context, req);
          const aggsParams = {
            index: _constants.INDEX.ALL_ALERTS,
            body: {
              size: 0,
              query: {
                bool: {
                  must: {
                    term: {
                      monitor_id: id
                    }
                  }
                }
              },
              aggs: {
                active_count: {
                  terms: {
                    field: 'state'
                  }
                },
                '24_hour_count': {
                  date_range: {
                    field: 'start_time',
                    ranges: [{
                      from: 'now-24h/h'
                    }]
                  }
                }
              }
            }
          };
          const searchResponse = await client('alerting.getMonitors', aggsParams);
          const dayCount = _lodash.default.get(searchResponse, 'aggregations.24_hour_count.buckets.0.doc_count', 0);
          const activeBuckets = _lodash.default.get(searchResponse, 'aggregations.active_count.buckets', []);
          const activeCount = activeBuckets.reduce((acc, curr) => curr.key === 'ACTIVE' ? curr.doc_count : acc, 0);
          if (associated_workflows) {
            monitor = {
              ...monitor,
              associated_workflows,
              associatedCompositeMonitorCnt: associated_workflows.length
            };
          }
          monitor = {
            ...monitor,
            item_type: monitor.workflow_type || monitor.monitor_type,
            id,
            version
          };
          return res.ok({
            body: {
              ok: true,
              resp: monitor,
              activeCount,
              dayCount,
              version,
              ifSeqNo,
              ifPrimaryTerm
            }
          });
        } else {
          return res.ok({
            body: {
              ok: false
            }
          });
        }
      } catch (err) {
        console.error('Alerting - MonitorService - getMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getWorkflow", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getWorkflow', params);
        let workflow = _lodash.default.get(getResponse, 'workflow', null);
        const version = _lodash.default.get(getResponse, '_version', null);
        const ifSeqNo = _lodash.default.get(getResponse, '_seq_no', null);
        const ifPrimaryTerm = _lodash.default.get(getResponse, '_primary_term', null);
        workflow.monitor_type = workflow.workflow_type;
        workflow = {
          ...workflow,
          item_type: workflow.workflow_type,
          id,
          version
        };
        return res.ok({
          body: {
            ok: true,
            resp: workflow,
            activeCount: 0,
            dayCount: 0,
            version,
            ifSeqNo,
            ifPrimaryTerm
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - getWorkflow:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "updateMonitor", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body,
          refresh: 'wait_for'
        };
        const {
          type
        } = req.body;

        // TODO DRAFT: Are we sure we need to include ifSeqNo and ifPrimaryTerm from the UI side when updating monitors?
        const {
          ifSeqNo,
          ifPrimaryTerm
        } = req.query;
        if (ifSeqNo && ifPrimaryTerm) {
          params.if_seq_no = ifSeqNo;
          params.if_primary_term = ifPrimaryTerm;
        }
        const client = this.getClientBasedOnDataSource(context, req);
        const updateResponse = await client(`alerting.${type === 'workflow' ? 'updateWorkflow' : 'updateMonitor'}`, params);
        const {
          _version,
          _id
        } = updateResponse;
        return res.ok({
          body: {
            ok: true,
            version: _version,
            id: _id
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - updateMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "getMonitors", async (context, req, res) => {
      try {
        const {
          from,
          size,
          search,
          sortDirection,
          sortField,
          state,
          monitorIds
        } = req.query;
        let must = {
          match_all: {}
        };
        if (search.trim()) {
          // This is an expensive wildcard query to match monitor names such as: "This is a long monitor name"
          // search query => "long monit"
          // This is acceptable because we will never allow more than 1,000 monitors
          must = {
            query_string: {
              default_field: 'monitor.name',
              default_operator: 'AND',
              query: `*${search.trim().split(' ').join('* *')}*`
            }
          };
        }
        const should = [];
        const mustList = [must];
        if (monitorIds !== undefined) {
          mustList.push({
            terms: {
              _id: Array.isArray(monitorIds) ? monitorIds : [monitorIds]
            }
          });
        } else if (monitorIds === 'empty') {
          mustList.push({
            terms: {
              _id: []
            }
          });
        }
        if (state !== 'all') {
          const enabled = state === 'enabled';
          should.push({
            term: {
              'monitor.enabled': enabled
            }
          });
          should.push({
            term: {
              'workflow.enabled': enabled
            }
          });
        }
        const monitorSorts = {
          name: 'monitor.name.keyword'
        };
        const monitorSortPageData = {
          size: 1000
        };
        if (monitorSorts[sortField]) {
          monitorSortPageData.sort = [{
            [monitorSorts[sortField]]: sortDirection
          }];
          monitorSortPageData.size = _lodash.default.defaultTo(size, 1000);
          monitorSortPageData.from = _lodash.default.defaultTo(from, 0);
        }
        const params = {
          body: {
            seq_no_primary_term: true,
            version: true,
            ...monitorSortPageData,
            query: {
              bool: {
                should,
                minimum_should_match: state !== 'all' ? 1 : 0,
                must: mustList
              }
            },
            aggregations: {
              associated_composite_monitors: {
                nested: {
                  path: 'workflow.inputs.composite_input.sequence.delegates'
                },
                aggs: {
                  monitor_ids: {
                    terms: {
                      field: 'workflow.inputs.composite_input.sequence.delegates.monitor_id'
                    }
                  }
                }
              }
            }
          }
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const getResponse = await client('alerting.getMonitors', params);
        const totalMonitors = _lodash.default.get(getResponse, 'hits.total.value', 0);
        const monitorKeyValueTuples = _lodash.default.get(getResponse, 'hits.hits', []).map(result => {
          const {
            _id: id,
            _version: version,
            _seq_no: ifSeqNo,
            _primary_term: ifPrimaryTerm,
            _source
          } = result;
          const monitor = _source.monitor ? _source.monitor : _source;
          monitor['item_type'] = monitor.workflow_type || monitor.monitor_type;
          const {
            name,
            enabled,
            item_type
          } = monitor;
          return [id, {
            id,
            version,
            ifSeqNo,
            ifPrimaryTerm,
            name,
            enabled,
            item_type,
            monitor
          }];
        }, {});
        const monitorMap = new Map(monitorKeyValueTuples);
        const associatedCompositeMonitorCountMap = {};
        _lodash.default.get(getResponse, 'aggregations.associated_composite_monitors.monitor_ids.buckets', []).forEach(({
          key,
          doc_count
        }) => {
          associatedCompositeMonitorCountMap[key] = doc_count;
        });
        const monitorIdsOutput = [...monitorMap.keys()];
        const aggsOrderData = {};
        const aggsSorts = {
          active: 'active',
          acknowledged: 'acknowledged',
          errors: 'errors',
          ignored: 'ignored',
          lastNotificationTime: 'last_notification_time'
        };
        if (aggsSorts[sortField]) {
          aggsOrderData.order = {
            [aggsSorts[sortField]]: sortDirection
          };
        }
        const aggsParams = {
          index: _constants.INDEX.ALL_ALERTS,
          body: {
            size: 0,
            query: {
              terms: {
                monitor_id: monitorIdsOutput
              }
            },
            aggregations: {
              uniq_monitor_ids: {
                terms: {
                  field: 'monitor_id',
                  ...aggsOrderData,
                  size: from + size
                },
                aggregations: {
                  active: {
                    filter: {
                      term: {
                        state: 'ACTIVE'
                      }
                    }
                  },
                  acknowledged: {
                    filter: {
                      term: {
                        state: 'ACKNOWLEDGED'
                      }
                    }
                  },
                  errors: {
                    filter: {
                      term: {
                        state: 'ERROR'
                      }
                    }
                  },
                  ignored: {
                    filter: {
                      bool: {
                        filter: {
                          term: {
                            state: 'COMPLETED'
                          }
                        },
                        must_not: {
                          exists: {
                            field: 'acknowledged_time'
                          }
                        }
                      }
                    }
                  },
                  last_notification_time: {
                    max: {
                      field: 'last_notification_time'
                    }
                  },
                  latest_alert: {
                    top_hits: {
                      size: 1,
                      sort: [{
                        start_time: {
                          order: 'desc'
                        }
                      }],
                      _source: {
                        includes: ['last_notification_time', 'trigger_name']
                      }
                    }
                  }
                }
              }
            }
          }
        };
        const esAggsResponse = await client('alerting.getMonitors', aggsParams);
        const buckets = _lodash.default.get(esAggsResponse, 'aggregations.uniq_monitor_ids.buckets', []).map(bucket => {
          const {
            key: id,
            last_notification_time: {
              value: lastNotificationTime
            },
            ignored: {
              doc_count: ignored
            },
            acknowledged: {
              doc_count: acknowledged
            },
            active: {
              doc_count: active
            },
            errors: {
              doc_count: errors
            },
            latest_alert: {
              hits: {
                hits: [{
                  _source: {
                    trigger_name: latestAlert
                  }
                }]
              }
            }
          } = bucket;
          const monitor = monitorMap.get(id);
          monitorMap.delete(id);
          return {
            ...monitor,
            id,
            lastNotificationTime,
            ignored,
            latestAlert,
            acknowledged,
            active,
            errors,
            currentTime: Date.now(),
            associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[id] || 0
          };
        });
        const unusedMonitors = [...monitorMap.values()].map(monitor => ({
          ...monitor,
          lastNotificationTime: null,
          ignored: 0,
          active: 0,
          acknowledged: 0,
          errors: 0,
          latestAlert: '--',
          currentTime: Date.now(),
          associatedCompositeMonitorCnt: associatedCompositeMonitorCountMap[monitor.id] || 0
        }));
        let results = _lodash.default.orderBy(buckets.concat(unusedMonitors), [sortField], [sortDirection]);
        // If we sorted on monitor name then we already applied from/size to the first query to limit what we're aggregating over
        // Therefore we do not need to apply from/size to this result set
        // If we sorted on aggregations, then this is our in memory pagination
        if (!monitorSorts[sortField]) {
          results = results.slice(from, from + size);
        }
        return res.ok({
          body: {
            ok: true,
            monitors: results,
            totalMonitors
          }
        });
      } catch (err) {
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          // Config index is not created unitl a monitor is created.
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: [],
                message: "No monitors created"
              }
            }
          });
        } else {
          // If the index is created, some error in retrieving the monitors.
          console.error('Alerting - MonitorService - getMonitors', err);
          return res.ok({
            body: {
              ok: false,
              resp: err.message
            }
          });
        }
      }
    });
    _defineProperty(this, "acknowledgeAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          monitorId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const acknowledgeResponse = await client('alerting.acknowledgeAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "acknowledgeChainedAlerts", async (context, req, res) => {
      try {
        const {
          id
        } = req.params;
        const params = {
          workflowId: id,
          body: req.body
        };
        const client = this.getClientBasedOnDataSource(context, this.dataSourceEnabled, req, this.esDriver);
        const acknowledgeResponse = await client('alerting.acknowledgeChainedAlerts', params);
        return res.ok({
          body: {
            ok: !acknowledgeResponse.failed.length,
            resp: acknowledgeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - acknowledgeChainedAlerts:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    _defineProperty(this, "executeMonitor", async (context, req, res) => {
      try {
        const {
          dryrun = 'true'
        } = req.query;
        const params = {
          body: req.body,
          dryrun
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const executeResponse = await client('alerting.executeMonitor', params);
        return res.ok({
          body: {
            ok: true,
            resp: executeResponse
          }
        });
      } catch (err) {
        console.error('Alerting - MonitorService - executeMonitor:', err);
        return res.ok({
          body: {
            ok: false,
            resp: err.message
          }
        });
      }
    });
    //TODO: This is temporarily a pass through call which needs to be deprecated
    _defineProperty(this, "searchMonitors", async (context, req, res) => {
      try {
        const {
          query,
          index,
          size
        } = req.body;
        const params = {
          index,
          size,
          body: query
        };
        const client = this.getClientBasedOnDataSource(context, req);
        const results = await client('alerting.getMonitors', params);
        return res.ok({
          body: {
            ok: true,
            resp: results
          }
        });
      } catch (err) {
        if ((0, _helpers.isIndexNotFoundError)(err)) {
          // Config index is not created unitl a monitor is created.
          return res.ok({
            body: {
              ok: false,
              resp: {
                totalMonitors: 0,
                monitors: [],
                message: "No monitors created"
              }
            }
          });
        } else {
          // If the index is created, some error in retrieving the monitors.
          console.error('Alerting - MonitorService - searchMonitor:', err);
          return res.ok({
            body: {
              ok: false,
              resp: err.message
            }
          });
        }
      }
    });
  }
}
exports.default = MonitorService;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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