"use strict";
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const tslib_1 = require("tslib");
const path_1 = tslib_1.__importDefault(require("path"));
const fs_1 = tslib_1.__importDefault(require("fs"));
const promises_1 = require("fs/promises");
const path_2 = require("./path");
const tmpTestFolder = './__test_artifacts__';
const longFolderName = '.long-folder-name';
const longFileName = '.long-file-name.txt';
const longSymlinkName = '.sym.link';
const shortFolderName = 'LONG-F~1';
const shortFileName = 'LONG-F~1.TXT';
const dummyWindowsPath = 'C:\\a\\b\\c';
const dummyWindowsPOSIXPath = 'C:/a/b/c';
const dummyPOSIXPath = '/a/b/c';
const onWindows = process.platform === 'win32' ? describe : xdescribe;
const onWindowsWithShortNames = (0, path_2.shortNamesSupportedSync)() ? describe : xdescribe;
// Save the real process.platform
const realPlatform = Object.getOwnPropertyDescriptor(process, 'platform');
describe('Cross Platform', () => {
    describe('path', () => {
        onWindows('on Windows', () => {
            onWindowsWithShortNames('when 8.3 is supported', () => {
                beforeAll(async () => {
                    // Cleanup
                    try {
                        // If leftover artifacts were found, get rid of them
                        await (0, promises_1.access)(tmpTestFolder);
                        await (0, promises_1.rm)(tmpTestFolder, { recursive: true });
                    }
                    catch (ex) {
                        // Do nothing; if `rm` failed, let the `mkdir` below throw the error
                    }
                    await (0, promises_1.mkdir)(tmpTestFolder);
                    await (0, promises_1.mkdir)(path_1.default.resolve(tmpTestFolder, longFolderName));
                    await (0, promises_1.writeFile)(path_1.default.resolve(tmpTestFolder, longFolderName, longFileName), '');
                    await (0, promises_1.symlink)(path_1.default.resolve(tmpTestFolder, longFolderName), path_1.default.resolve(tmpTestFolder, longSymlinkName), 'junction');
                });
                afterAll(async () => {
                    try {
                        await (0, promises_1.rm)(tmpTestFolder, { recursive: true });
                    }
                    catch (ex) {
                        // Do nothing
                    }
                });
                it('can synchronously extract full name of a folder', () => {
                    const name = path_1.default.basename((0, path_2.resolveToFullPathSync)(path_1.default.resolve(tmpTestFolder, shortFolderName)));
                    expect(name).toBe(longFolderName);
                });
                it('can synchronously extract full name of a file', () => {
                    const name = path_1.default.basename((0, path_2.resolveToFullNameSync)(path_1.default.resolve(tmpTestFolder, shortFolderName, shortFileName)));
                    expect(name).toBe(longFileName);
                });
                it('can synchronously extract short name of a folder', () => {
                    const name = path_1.default.basename((0, path_2.resolveToShortPathSync)(path_1.default.resolve(tmpTestFolder, longFolderName)));
                    expect(name).toBe(shortFolderName);
                });
                it('can synchronously extract short name of a file', () => {
                    const name = path_1.default.basename((0, path_2.resolveToShortNameSync)(path_1.default.resolve(tmpTestFolder, longFolderName, longFileName)));
                    expect(name).toBe(shortFileName);
                });
                it('can synchronously extract full name of a symbolic link', () => {
                    const name = path_1.default.basename((0, path_2.realPathSync)(path_1.default.resolve(tmpTestFolder, longSymlinkName)));
                    expect(name).toBe(longFolderName);
                });
                it('can synchronously extract short name of a symbolic link', () => {
                    const name = path_1.default.basename((0, path_2.realShortPathSync)(path_1.default.resolve(tmpTestFolder, longSymlinkName)));
                    expect(name).toBe(shortFolderName);
                });
            });
        });
        describe('on platforms other than Windows', () => {
            let mockPathNormalize;
            let mockPathResolve;
            let mockFSRealPathSync;
            beforeAll(() => {
                Object.defineProperty(process, 'platform', {
                    ...Object.getOwnPropertyDescriptor(process, 'property'),
                    value: 'linux',
                });
                mockPathNormalize = jest.spyOn(path_1.default, 'normalize').mockReturnValue(dummyPOSIXPath);
                mockPathResolve = jest.spyOn(path_1.default, 'resolve').mockReturnValue(dummyPOSIXPath);
                mockFSRealPathSync = jest
                    .spyOn(fs_1.default, 'realpathSync')
                    .mockReturnValue(dummyPOSIXPath);
            });
            afterAll(() => {
                // Restore the real property value after each test
                Object.defineProperty(process, 'platform', realPlatform);
                mockPathNormalize.mockRestore();
                mockPathResolve.mockRestore();
                mockFSRealPathSync.mockRestore();
            });
            it('all short and full name methods return just the normalized paths', () => {
                expect((0, path_2.shortNamesSupportedSync)()).toBe(false);
                expect((0, path_2.resolveToFullPathSync)(dummyPOSIXPath)).toBe(dummyPOSIXPath);
                expect((0, path_2.resolveToShortPathSync)(dummyPOSIXPath)).toBe(dummyPOSIXPath);
            });
        });
        describe('standardize', () => {
            describe('on Windows', () => {
                let mockPathNormalize;
                beforeAll(() => {
                    Object.defineProperty(process, 'platform', {
                        ...Object.getOwnPropertyDescriptor(process, 'property'),
                        value: 'win32',
                    });
                    mockPathNormalize = jest.spyOn(path_1.default, 'normalize').mockReturnValue(dummyWindowsPath);
                });
                afterAll(() => {
                    // Restore the real property value after each test
                    Object.defineProperty(process, 'platform', realPlatform);
                    mockPathNormalize.mockRestore();
                });
                it('produces a path in native format', () => {
                    expect((0, path_2.standardize)(dummyWindowsPath, false, false)).toMatchSnapshot();
                });
                it('produces a path in native format even for POSIX input', () => {
                    expect((0, path_2.standardize)(dummyWindowsPOSIXPath, false, false)).toMatchSnapshot();
                });
                it('produces a path in native format with escaped backslashes', () => {
                    expect((0, path_2.standardize)(dummyWindowsPath, false, true)).toMatchSnapshot();
                });
                it('produces a path in POSIX format', () => {
                    expect((0, path_2.standardize)(dummyWindowsPath)).toMatchSnapshot();
                });
            });
            describe('on POSIX-compatible platforms', () => {
                let mockPathNormalize;
                beforeAll(() => {
                    Object.defineProperty(process, 'platform', {
                        ...Object.getOwnPropertyDescriptor(process, 'property'),
                        value: 'linux',
                    });
                    mockPathNormalize = jest.spyOn(path_1.default, 'normalize').mockReturnValue(dummyPOSIXPath);
                });
                afterAll(() => {
                    // Restore the real property value after each test
                    Object.defineProperty(process, 'platform', realPlatform);
                    mockPathNormalize.mockRestore();
                });
                it('produces a path in POSIX format', () => {
                    expect((0, path_2.standardize)(dummyPOSIXPath)).toMatchSnapshot();
                });
                it('ignores additional parameters', () => {
                    expect((0, path_2.standardize)(dummyPOSIXPath, false, true)).toMatchSnapshot();
                });
            });
        });
    });
});
