"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const env_test_mocks_1 = require("./env.test.mocks");
const env_1 = require("./env");
const env_2 = require("./__mocks__/env");
const REPO_ROOT = '/test/opensearchDashboardsRoot';
const packageInfos = {
    branch: 'main',
    version: '8.0.0',
    build: {
        number: 42,
        sha: 'one',
    },
};
beforeEach(() => {
    env_test_mocks_1.mockPackage.raw = {};
});
test('correctly creates default environment in dev mode when isDevClusterMaster (deprecated) is true', () => {
    env_test_mocks_1.mockPackage.raw = {
        branch: 'some-branch',
        version: 'some-version',
    };
    const defaultEnv = env_1.Env.createDefault(REPO_ROOT, (0, env_2.getEnvOptions)({
        configs: ['/test/cwd/config/opensearch_dashboards.yml'],
        isDevClusterMaster: true,
        isDevClusterManager: false,
    }));
    expect(defaultEnv).toMatchSnapshot('env properties');
    expect(defaultEnv.isDevClusterManager).toBeTruthy();
});
test('correctly creates default environment in dev mode when isDevClusterManager is true', () => {
    env_test_mocks_1.mockPackage.raw = {
        branch: 'some-branch',
        version: 'some-version',
    };
    const defaultEnv = env_1.Env.createDefault(REPO_ROOT, (0, env_2.getEnvOptions)({
        configs: ['/test/cwd/config/opensearch_dashboards.yml'],
        isDevClusterMaster: false,
        isDevClusterManager: true,
    }));
    expect(defaultEnv).toMatchSnapshot('env properties');
    expect(defaultEnv.isDevClusterManager).toBeTruthy();
});
test('correctly creates default environment in dev mode when isDevClusterManager and isDevClusterMaster both are true', () => {
    env_test_mocks_1.mockPackage.raw = {
        branch: 'some-branch',
        version: 'some-version',
    };
    const defaultEnv = env_1.Env.createDefault(REPO_ROOT, (0, env_2.getEnvOptions)({
        configs: ['/test/cwd/config/opensearch_dashboards.yml'],
        isDevClusterMaster: true,
        isDevClusterManager: true,
    }));
    expect(defaultEnv).toMatchSnapshot('env properties');
    expect(defaultEnv.isDevClusterManager).toBeTruthy();
});
test('correctly creates default environment in prod distributable mode.', () => {
    env_test_mocks_1.mockPackage.raw = {
        branch: 'feature-v1',
        version: 'v1',
        build: {
            distributable: true,
            number: 100,
            sha: 'feature-v1-build-sha',
        },
    };
    const defaultEnv = env_1.Env.createDefault(REPO_ROOT, (0, env_2.getEnvOptions)({
        cliArgs: { dev: false },
        configs: ['/some/other/path/some-opensearch-dashboards.yml'],
    }));
    expect(defaultEnv).toMatchSnapshot('env properties');
});
test('correctly creates default environment in prod non-distributable mode.', () => {
    env_test_mocks_1.mockPackage.raw = {
        branch: 'feature-v1',
        version: 'v1',
        build: {
            distributable: false,
            number: 100,
            sha: 'feature-v1-build-sha',
        },
    };
    const defaultEnv = env_1.Env.createDefault(REPO_ROOT, (0, env_2.getEnvOptions)({
        cliArgs: { dev: false },
        configs: ['/some/other/path/some-opensearch-dashboards.yml'],
    }));
    expect(defaultEnv).toMatchSnapshot('env properties');
});
test('correctly creates default environment if `--env.name` is supplied.', () => {
    env_test_mocks_1.mockPackage.raw = {
        branch: 'feature-v1',
        version: 'v1',
        build: {
            distributable: false,
            number: 100,
            sha: 'feature-v1-build-sha',
        },
    };
    const defaultDevEnv = env_1.Env.createDefault(REPO_ROOT, (0, env_2.getEnvOptions)({
        cliArgs: { envName: 'development' },
        configs: ['/some/other/path/some-opensearch-dashboards.yml'],
    }));
    const defaultProdEnv = env_1.Env.createDefault(REPO_ROOT, (0, env_2.getEnvOptions)({
        cliArgs: { dev: false, envName: 'production' },
        configs: ['/some/other/path/some-opensearch-dashboards.yml'],
    }));
    expect(defaultDevEnv).toMatchSnapshot('dev env properties');
    expect(defaultProdEnv).toMatchSnapshot('prod env properties');
});
test('correctly creates environment with constructor.', () => {
    const env = new env_1.Env('/some/home/dir', {
        branch: 'feature-v1',
        version: 'v1',
        build: {
            distributable: true,
            number: 100,
            sha: 'feature-v1-build-sha',
        },
    }, (0, env_2.getEnvOptions)({
        cliArgs: { dev: false },
        configs: ['/some/other/path/some-opensearch-dashboards.yml'],
    }));
    expect(env).toMatchSnapshot('env properties');
});
test('pluginSearchPaths contains examples plugins path if --run-examples flag is true', () => {
    const env = new env_1.Env('/some/home/dir', packageInfos, (0, env_2.getEnvOptions)({
        cliArgs: { runExamples: true },
    }));
    expect(env.pluginSearchPaths).toContain('/some/home/dir/examples');
});
test('pluginSearchPaths does not contains examples plugins path if --run-examples flag is false', () => {
    const env = new env_1.Env('/some/home/dir', packageInfos, (0, env_2.getEnvOptions)({
        cliArgs: { runExamples: false },
    }));
    expect(env.pluginSearchPaths).not.toContain('/some/home/dir/examples');
});
