"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const tsd_1 = require("tsd");
const __1 = require("..");
test('returns value by default', () => {
    const type = __1.schema.object({
        name: __1.schema.string(),
    });
    const value = {
        name: 'test',
    };
    expect(type.validate(value)).toEqual({ name: 'test' });
});
test('returns empty object if undefined', () => {
    const type = __1.schema.object({});
    expect(type.validate(undefined)).toEqual({});
});
test('properly parse the value if input is a string', () => {
    const type = __1.schema.object({
        name: __1.schema.string(),
    });
    const value = `{"name": "test"}`;
    expect(type.validate(value)).toEqual({ name: 'test' });
});
test('fails if string input cannot be parsed', () => {
    const type = __1.schema.object({
        name: __1.schema.string(),
    });
    expect(() => type.validate(`invalidjson`)).toThrowErrorMatchingInlineSnapshot(`"could not parse object value from json input"`);
});
test('fails with correct type if parsed input is not an object', () => {
    const type = __1.schema.object({
        name: __1.schema.string(),
    });
    expect(() => type.validate('[1,2,3]')).toThrowErrorMatchingInlineSnapshot(`"expected a plain object value, but found [Array] instead."`);
});
test('fails if missing required value', () => {
    const type = __1.schema.object({
        name: __1.schema.string(),
    });
    const value = {};
    expect(() => type.validate(value)).toThrowErrorMatchingInlineSnapshot(`"[name]: expected value of type [string] but got [undefined]"`);
});
test('returns value if undefined string with default', () => {
    const type = __1.schema.object({
        name: __1.schema.string({ defaultValue: 'test' }),
    });
    const value = {};
    expect(type.validate(value)).toEqual({ name: 'test' });
});
test('fails if key does not exist in schema', () => {
    const type = __1.schema.object({
        foo: __1.schema.string(),
    });
    const value = {
        bar: 'baz',
        foo: 'bar',
    };
    expect(() => type.validate(value)).toThrowErrorMatchingInlineSnapshot(`"[bar]: definition for this key is missing"`);
});
test('defined object within object', () => {
    const type = __1.schema.object({
        foo: __1.schema.object({
            bar: __1.schema.string({ defaultValue: 'hello world' }),
        }),
    });
    expect(type.validate({ foo: {} })).toEqual({
        foo: {
            bar: 'hello world',
        },
    });
});
test('undefined object within object', () => {
    const type = __1.schema.object({
        foo: __1.schema.object({
            bar: __1.schema.string({ defaultValue: 'hello world' }),
        }),
    });
    expect(type.validate(undefined)).toEqual({
        foo: {
            bar: 'hello world',
        },
    });
    expect(type.validate({})).toEqual({
        foo: {
            bar: 'hello world',
        },
    });
    expect(type.validate({ foo: {} })).toEqual({
        foo: {
            bar: 'hello world',
        },
    });
});
test('object within object with key without defaultValue', () => {
    const type = __1.schema.object({
        foo: __1.schema.object({
            bar: __1.schema.string(),
        }),
    });
    const value = { foo: {} };
    expect(() => type.validate(undefined)).toThrowErrorMatchingInlineSnapshot(`"[foo.bar]: expected value of type [string] but got [undefined]"`);
    expect(() => type.validate(value)).toThrowErrorMatchingInlineSnapshot(`"[foo.bar]: expected value of type [string] but got [undefined]"`);
});
describe('#validate', () => {
    test('is called after all content is processed', () => {
        const mockValidate = jest.fn();
        const type = __1.schema.object({
            foo: __1.schema.object({
                bar: __1.schema.string({ defaultValue: 'baz' }),
            }),
        }, {
            validate: mockValidate,
        });
        type.validate({ foo: {} });
        expect(mockValidate).toHaveBeenCalledWith({
            foo: {
                bar: 'baz',
            },
        });
    });
});
test('called with wrong type', () => {
    const type = __1.schema.object({});
    expect(() => type.validate('foo')).toThrowErrorMatchingInlineSnapshot(`"could not parse object value from json input"`);
    expect(() => type.validate(123)).toThrowErrorMatchingInlineSnapshot(`"expected a plain object value, but found [number] instead."`);
});
test('handles oneOf', () => {
    const type = __1.schema.object({
        key: __1.schema.oneOf([__1.schema.string()]),
    });
    expect(type.validate({ key: 'foo' })).toEqual({ key: 'foo' });
    expect(() => type.validate({ key: 123 })).toThrowErrorMatchingInlineSnapshot(`
"[key]: types that failed validation:
- [key.0]: expected value of type [string] but got [number]"
`);
});
test('handles references', () => {
    const type = __1.schema.object({
        context: __1.schema.string({
            defaultValue: __1.schema.contextRef('context_value'),
        }),
        key: __1.schema.string(),
        value: __1.schema.string({ defaultValue: __1.schema.siblingRef('key') }),
    });
    expect(type.validate({ key: 'key#1' }, { context_value: 'context#1' })).toEqual({
        context: 'context#1',
        key: 'key#1',
        value: 'key#1',
    });
    expect(type.validate({ key: 'key#1', value: 'value#1' })).toEqual({
        key: 'key#1',
        value: 'value#1',
    });
});
test('handles conditionals', () => {
    const type = __1.schema.object({
        key: __1.schema.string(),
        value: __1.schema.conditional(__1.schema.siblingRef('key'), 'some-key', __1.schema.string({ defaultValue: 'some-value' }), __1.schema.string({ defaultValue: 'unknown-value' })),
    });
    expect(type.validate({ key: 'some-key' })).toEqual({
        key: 'some-key',
        value: 'some-value',
    });
    expect(type.validate({ key: 'another-key' })).toEqual({
        key: 'another-key',
        value: 'unknown-value',
    });
});
test('includes namespace in failure when wrong top-level type', () => {
    const type = __1.schema.object({
        foo: __1.schema.string(),
    });
    expect(() => type.validate([], {}, 'foo-namespace')).toThrowErrorMatchingInlineSnapshot(`"[foo-namespace]: expected a plain object value, but found [Array] instead."`);
});
test('includes namespace in failure when wrong value type', () => {
    const type = __1.schema.object({
        foo: __1.schema.string(),
    });
    const value = {
        foo: 123,
    };
    expect(() => type.validate(value, {}, 'foo-namespace')).toThrowErrorMatchingInlineSnapshot(`"[foo-namespace.foo]: expected value of type [string] but got [number]"`);
});
test('individual keys can validated', () => {
    const type = __1.schema.object({
        foo: __1.schema.boolean(),
    });
    const value = false;
    expect(() => type.validateKey('foo', value)).not.toThrowError();
    expect(() => type.validateKey('bar', '')).toThrowErrorMatchingInlineSnapshot(`"bar is not a valid part of this schema"`);
});
test('allow unknown keys when unknowns = `allow`', () => {
    const type = __1.schema.object({ foo: __1.schema.string({ defaultValue: 'test' }) }, { unknowns: 'allow' });
    expect(type.validate({
        bar: 'baz',
    })).toEqual({
        foo: 'test',
        bar: 'baz',
    });
});
test('unknowns = `allow` affects only own keys', () => {
    const type = __1.schema.object({ foo: __1.schema.object({ bar: __1.schema.string() }) }, { unknowns: 'allow' });
    expect(() => type.validate({
        foo: {
            bar: 'bar',
            baz: 'baz',
        },
    })).toThrowErrorMatchingInlineSnapshot(`"[foo.baz]: definition for this key is missing"`);
});
test('does not allow unknown keys when unknowns = `forbid`', () => {
    const type = __1.schema.object({ foo: __1.schema.string({ defaultValue: 'test' }) }, { unknowns: 'forbid' });
    expect(() => type.validate({
        bar: 'baz',
    })).toThrowErrorMatchingInlineSnapshot(`"[bar]: definition for this key is missing"`);
});
test('allow and remove unknown keys when unknowns = `ignore`', () => {
    const type = __1.schema.object({ foo: __1.schema.string({ defaultValue: 'test' }) }, { unknowns: 'ignore' });
    expect(type.validate({
        bar: 'baz',
    })).toEqual({
        foo: 'test',
    });
});
test('unknowns = `ignore` affects only own keys', () => {
    const type = __1.schema.object({ foo: __1.schema.object({ bar: __1.schema.string() }) }, { unknowns: 'ignore' });
    expect(() => type.validate({
        foo: {
            bar: 'bar',
            baz: 'baz',
        },
    })).toThrowErrorMatchingInlineSnapshot(`"[foo.baz]: definition for this key is missing"`);
});
test('handles optional properties', () => {
    const type = __1.schema.object({
        required: __1.schema.string(),
        optional: __1.schema.maybe(__1.schema.string()),
    });
    (0, tsd_1.expectType)({
        required: 'foo',
    });
    (0, tsd_1.expectType)({
        required: 'hello',
        optional: undefined,
    });
    (0, tsd_1.expectType)({
        required: 'hello',
        optional: 'bar',
    });
});
describe('#extends', () => {
    it('allows to extend an existing schema by adding new properties', () => {
        const origin = __1.schema.object({
            initial: __1.schema.string(),
        });
        const extended = origin.extends({
            added: __1.schema.number(),
        });
        expect(() => {
            extended.validate({ initial: 'foo' });
        }).toThrowErrorMatchingInlineSnapshot(`"[added]: expected value of type [number] but got [undefined]"`);
        expect(() => {
            extended.validate({ initial: 'foo', added: 42 });
        }).not.toThrowError();
        (0, tsd_1.expectType)({
            added: 12,
            initial: 'foo',
        });
    });
    it('allows to extend an existing schema by removing properties', () => {
        const origin = __1.schema.object({
            string: __1.schema.string(),
            number: __1.schema.number(),
        });
        const extended = origin.extends({ number: undefined });
        expect(() => {
            extended.validate({ string: 'foo', number: 12 });
        }).toThrowErrorMatchingInlineSnapshot(`"[number]: definition for this key is missing"`);
        expect(() => {
            extended.validate({ string: 'foo' });
        }).not.toThrowError();
        (0, tsd_1.expectType)({
            string: 'foo',
        });
    });
    it('allows to extend an existing schema by overriding an existing properties', () => {
        const origin = __1.schema.object({
            string: __1.schema.string(),
            mutated: __1.schema.number(),
        });
        const extended = origin.extends({
            mutated: __1.schema.string(),
        });
        expect(() => {
            extended.validate({ string: 'foo', mutated: 12 });
        }).toThrowErrorMatchingInlineSnapshot(`"[mutated]: expected value of type [string] but got [number]"`);
        expect(() => {
            extended.validate({ string: 'foo', mutated: 'bar' });
        }).not.toThrowError();
        (0, tsd_1.expectType)({
            string: 'foo',
            mutated: 'bar',
        });
    });
    it('properly infer the type from optional properties', () => {
        const origin = __1.schema.object({
            original: __1.schema.maybe(__1.schema.string()),
            mutated: __1.schema.maybe(__1.schema.number()),
            removed: __1.schema.maybe(__1.schema.string()),
        });
        const extended = origin.extends({
            removed: undefined,
            mutated: __1.schema.string(),
        });
        expect(() => {
            extended.validate({ original: 'foo' });
        }).toThrowErrorMatchingInlineSnapshot(`"[mutated]: expected value of type [string] but got [undefined]"`);
        expect(() => {
            extended.validate({ original: 'foo' });
        }).toThrowErrorMatchingInlineSnapshot(`"[mutated]: expected value of type [string] but got [undefined]"`);
        expect(() => {
            extended.validate({ original: 'foo', mutated: 'bar' });
        }).not.toThrowError();
        (0, tsd_1.expectType)({
            original: 'foo',
            mutated: 'bar',
        });
        (0, tsd_1.expectType)({
            mutated: 'bar',
        });
    });
    it(`allows to override the original schema's options`, () => {
        const origin = __1.schema.object({
            initial: __1.schema.string(),
        }, { defaultValue: { initial: 'foo' } });
        const extended = origin.extends({
            added: __1.schema.number(),
        }, { defaultValue: { initial: 'bar', added: 42 } });
        expect(extended.validate(undefined)).toEqual({ initial: 'bar', added: 42 });
    });
});
