"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const __1 = require("..");
test('returns string value when passed string', () => {
    const type = __1.schema.nullable(__1.schema.string());
    expect(type.validate('test')).toBe('test');
});
test('returns number value when passed number', () => {
    const type = __1.schema.nullable(__1.schema.number());
    expect(type.validate(42)).toBe(42);
});
test('returns boolean value when passed boolean', () => {
    const type = __1.schema.nullable(__1.schema.boolean());
    expect(type.validate(true)).toBe(true);
});
test('returns object value when passed object', () => {
    const type = __1.schema.nullable(__1.schema.object({
        foo: __1.schema.number(),
        bar: __1.schema.boolean(),
        baz: __1.schema.string(),
    }));
    const object = {
        foo: 666,
        bar: true,
        baz: 'foo bar baz',
    };
    expect(type.validate(object)).toEqual(object);
});
test('returns null if null for string', () => {
    const type = __1.schema.nullable(__1.schema.string());
    expect(type.validate(null)).toBe(null);
});
test('returns null if null for number', () => {
    const type = __1.schema.nullable(__1.schema.number());
    expect(type.validate(null)).toBe(null);
});
test('returns null if null for boolean', () => {
    const type = __1.schema.nullable(__1.schema.boolean());
    expect(type.validate(null)).toBe(null);
});
test('returns null if undefined for string', () => {
    const type = __1.schema.nullable(__1.schema.string());
    expect(type.validate(undefined)).toBe(null);
});
test('returns null if undefined for number', () => {
    const type = __1.schema.nullable(__1.schema.number());
    expect(type.validate(undefined)).toBe(null);
});
test('returns null if undefined for boolean', () => {
    const type = __1.schema.nullable(__1.schema.boolean());
    expect(type.validate(undefined)).toBe(null);
});
test('returns null even if contained type has a default value', () => {
    const type = __1.schema.nullable(__1.schema.string({
        defaultValue: 'abc',
    }));
    expect(type.validate(undefined)).toBe(null);
});
test('validates contained type', () => {
    const type = __1.schema.nullable(__1.schema.string({ maxLength: 1 }));
    expect(() => type.validate('foo')).toThrowErrorMatchingInlineSnapshot(`
"types that failed validation:
- [0]: value has length [3] but it must have a maximum length of [1].
- [1]: expected value to equal [null]"
`);
});
test('validates basic type', () => {
    const type = __1.schema.nullable(__1.schema.string());
    expect(() => type.validate(666)).toThrowErrorMatchingInlineSnapshot(`
"types that failed validation:
- [0]: expected value of type [string] but got [number]
- [1]: expected value to equal [null]"
`);
});
test('validates type in object', () => {
    const type = __1.schema.object({
        foo: __1.schema.nullable(__1.schema.string({ maxLength: 1 })),
        bar: __1.schema.nullable(__1.schema.boolean()),
    });
    expect(type.validate({ foo: 'a' })).toEqual({ foo: 'a', bar: null });
    expect(type.validate({ foo: null })).toEqual({ foo: null, bar: null });
    expect(type.validate({})).toEqual({ foo: null, bar: null });
    expect(type.validate({ bar: null })).toEqual({ foo: null, bar: null });
});
test('validates type errors in object', () => {
    const type = __1.schema.object({
        foo: __1.schema.nullable(__1.schema.string({ maxLength: 1 })),
        bar: __1.schema.nullable(__1.schema.boolean()),
    });
    expect(() => type.validate({ foo: 'ab' })).toThrowErrorMatchingInlineSnapshot(`
"[foo]: types that failed validation:
- [foo.0]: value has length [2] but it must have a maximum length of [1].
- [foo.1]: expected value to equal [null]"
`);
});
test('includes namespace in failure', () => {
    const type = __1.schema.nullable(__1.schema.string({ maxLength: 1 }));
    expect(() => type.validate('foo', {}, 'foo-namespace')).toThrowErrorMatchingInlineSnapshot(`
"[foo-namespace]: types that failed validation:
- [foo-namespace.0]: value has length [3] but it must have a maximum length of [1].
- [foo-namespace.1]: expected value to equal [null]"
`);
});
