"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const __1 = require("..");
test('returns value if specified', () => {
    const type = __1.schema.maybe(__1.schema.string());
    expect(type.validate('test')).toEqual('test');
});
test('returns undefined if undefined', () => {
    const type = __1.schema.maybe(__1.schema.string());
    expect(type.validate(undefined)).toEqual(undefined);
});
test('returns undefined even if contained type has a default value', () => {
    const type = __1.schema.maybe(__1.schema.string({
        defaultValue: 'abc',
    }));
    expect(type.validate(undefined)).toEqual(undefined);
});
test('validates contained type', () => {
    const type = __1.schema.maybe(__1.schema.string({ maxLength: 1 }));
    expect(() => type.validate('foo')).toThrowErrorMatchingInlineSnapshot(`"value has length [3] but it must have a maximum length of [1]."`);
});
test('validates basic type', () => {
    const type = __1.schema.maybe(__1.schema.string());
    expect(() => type.validate(666)).toThrowErrorMatchingInlineSnapshot(`"expected value of type [string] but got [number]"`);
});
test('fails if null', () => {
    const type = __1.schema.maybe(__1.schema.string());
    expect(() => type.validate(null)).toThrowErrorMatchingInlineSnapshot(`"expected value of type [string] but got [null]"`);
});
test('includes namespace in failure', () => {
    const type = __1.schema.maybe(__1.schema.string());
    expect(() => type.validate(null, {}, 'foo-namespace')).toThrowErrorMatchingInlineSnapshot(`"[foo-namespace]: expected value of type [string] but got [null]"`);
});
describe('maybe + object', () => {
    test('returns undefined if undefined object', () => {
        const type = __1.schema.maybe(__1.schema.object({}));
        expect(type.validate(undefined)).toEqual(undefined);
    });
    test('returns undefined if undefined object with no defaults', () => {
        const type = __1.schema.maybe(__1.schema.object({
            type: __1.schema.string(),
            id: __1.schema.string(),
        }));
        expect(type.validate(undefined)).toEqual(undefined);
    });
    test('returns empty object if maybe keys', () => {
        const type = __1.schema.object({
            name: __1.schema.maybe(__1.schema.string()),
        });
        expect(type.validate({})).toEqual({});
    });
    test('returns empty object if maybe nested object', () => {
        const type = __1.schema.object({
            name: __1.schema.maybe(__1.schema.object({
                type: __1.schema.string(),
                id: __1.schema.string(),
            })),
        });
        expect(type.validate({})).toEqual({});
    });
});
