"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const _1 = require(".");
describe('parsing units', () => {
    test('number string (bytes)', () => {
        expect(_1.ByteSizeValue.parse('123').getValueInBytes()).toBe(123);
    });
    test('bytes', () => {
        expect(_1.ByteSizeValue.parse('123b').getValueInBytes()).toBe(123);
    });
    test('kilobytes', () => {
        expect(_1.ByteSizeValue.parse('1kb').getValueInBytes()).toBe(1024);
        expect(_1.ByteSizeValue.parse('15kb').getValueInBytes()).toBe(15360);
    });
    test('megabytes', () => {
        expect(_1.ByteSizeValue.parse('1mb').getValueInBytes()).toBe(1048576);
    });
    test('gigabytes', () => {
        expect(_1.ByteSizeValue.parse('1gb').getValueInBytes()).toBe(1073741824);
    });
    test('throws an error when unsupported unit specified', () => {
        expect(() => _1.ByteSizeValue.parse('1tb')).toThrowErrorMatchingInlineSnapshot(`"Failed to parse value as byte value. Value must be either number of bytes, or follow the format <count>[b|kb|mb|gb] (e.g., '1024kb', '200mb', '1gb'), where the number is a safe positive integer."`);
    });
});
describe('#constructor', () => {
    test('throws if number of bytes is negative', () => {
        expect(() => new _1.ByteSizeValue(-1024)).toThrowErrorMatchingInlineSnapshot(`"Value in bytes is expected to be a safe positive integer."`);
    });
    test('throws if number of bytes is not safe', () => {
        expect(() => new _1.ByteSizeValue(NaN)).toThrowErrorMatchingInlineSnapshot(`"Value in bytes is expected to be a safe positive integer."`);
        expect(() => new _1.ByteSizeValue(Infinity)).toThrowErrorMatchingInlineSnapshot(`"Value in bytes is expected to be a safe positive integer."`);
        expect(() => new _1.ByteSizeValue(Math.pow(2, 53))).toThrowErrorMatchingInlineSnapshot(`"Value in bytes is expected to be a safe positive integer."`);
    });
    test('accepts 0', () => {
        const value = new _1.ByteSizeValue(0);
        expect(value.getValueInBytes()).toBe(0);
    });
    test('accepts safe positive integer', () => {
        const value = new _1.ByteSizeValue(1024);
        expect(value.getValueInBytes()).toBe(1024);
    });
});
describe('#isGreaterThan', () => {
    test('handles true', () => {
        const a = _1.ByteSizeValue.parse('2kb');
        const b = _1.ByteSizeValue.parse('1kb');
        expect(a.isGreaterThan(b)).toBe(true);
    });
    test('handles false', () => {
        const a = _1.ByteSizeValue.parse('2kb');
        const b = _1.ByteSizeValue.parse('1kb');
        expect(b.isGreaterThan(a)).toBe(false);
    });
});
describe('#isLessThan', () => {
    test('handles true', () => {
        const a = _1.ByteSizeValue.parse('2kb');
        const b = _1.ByteSizeValue.parse('1kb');
        expect(b.isLessThan(a)).toBe(true);
    });
    test('handles false', () => {
        const a = _1.ByteSizeValue.parse('2kb');
        const b = _1.ByteSizeValue.parse('1kb');
        expect(a.isLessThan(b)).toBe(false);
    });
});
describe('#isEqualTo', () => {
    test('handles true', () => {
        const a = _1.ByteSizeValue.parse('1kb');
        const b = _1.ByteSizeValue.parse('1kb');
        expect(b.isEqualTo(a)).toBe(true);
    });
    test('handles false', () => {
        const a = _1.ByteSizeValue.parse('2kb');
        const b = _1.ByteSizeValue.parse('1kb');
        expect(a.isEqualTo(b)).toBe(false);
    });
});
describe('#toString', () => {
    test('renders to nearest lower unit by default', () => {
        expect(_1.ByteSizeValue.parse('1b').toString()).toBe('1b');
        expect(_1.ByteSizeValue.parse('10b').toString()).toBe('10b');
        expect(_1.ByteSizeValue.parse('1023b').toString()).toBe('1023b');
        expect(_1.ByteSizeValue.parse('1024b').toString()).toBe('1kb');
        expect(_1.ByteSizeValue.parse('1025b').toString()).toBe('1kb');
        expect(_1.ByteSizeValue.parse('1024kb').toString()).toBe('1mb');
        expect(_1.ByteSizeValue.parse('1024mb').toString()).toBe('1gb');
        expect(_1.ByteSizeValue.parse('1024gb').toString()).toBe('1024gb');
    });
    test('renders to specified unit', () => {
        expect(_1.ByteSizeValue.parse('1024b').toString('b')).toBe('1024b');
        expect(_1.ByteSizeValue.parse('1kb').toString('b')).toBe('1024b');
        expect(_1.ByteSizeValue.parse('1mb').toString('kb')).toBe('1024kb');
        expect(_1.ByteSizeValue.parse('1mb').toString('b')).toBe('1048576b');
        expect(_1.ByteSizeValue.parse('512mb').toString('gb')).toBe('0.5gb');
    });
});
