"use strict";
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Any modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
const config_test_mocks_1 = require("./config.test.mocks");
const config_1 = require("./config");
describe('ApmConfiguration', () => {
    beforeEach(() => {
        config_test_mocks_1.packageMock.raw = {
            version: '8.0.0',
            build: {
                sha: 'sha',
            },
        };
    });
    afterEach(() => {
        (0, config_test_mocks_1.resetAllMocks)();
    });
    it('sets the correct service name and version', () => {
        config_test_mocks_1.packageMock.raw = {
            version: '9.2.1',
        };
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, {}, false);
        expect(config.getConfig('myservice').serviceName).toBe('myservice');
        expect(config.getConfig('myservice').serviceVersion).toBe('9.2.1');
    });
    it('sets the git revision from `git rev-parse` command in non distribution mode', () => {
        config_test_mocks_1.gitRevExecMock.mockReturnValue('some-git-rev');
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, {}, false);
        expect(config.getConfig('serviceName').globalLabels.git_rev).toBe('some-git-rev');
    });
    it('sets the git revision from `pkg.build.sha` in distribution mode', () => {
        config_test_mocks_1.gitRevExecMock.mockReturnValue('dev-sha');
        config_test_mocks_1.packageMock.raw = {
            version: '9.2.1',
            build: {
                sha: 'distribution-sha',
            },
        };
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, {}, true);
        expect(config.getConfig('serviceName').globalLabels.git_rev).toBe('distribution-sha');
    });
    it('reads the opensearchDashboards uuid from the uuid file', () => {
        config_test_mocks_1.readUuidFileMock.mockReturnValue('instance-uuid');
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, {}, false);
        expect(config.getConfig('serviceName').globalLabels.opensearch_dashboards_uuid).toBe('instance-uuid');
    });
    it('uses the uuid from the opensearchDashboards config if present', () => {
        config_test_mocks_1.readUuidFileMock.mockReturnValue('uuid-from-file');
        const opensearchDashboardsConfig = {
            server: {
                uuid: 'uuid-from-config',
            },
        };
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, opensearchDashboardsConfig, false);
        expect(config.getConfig('serviceName').globalLabels.opensearch_dashboards_uuid).toBe('uuid-from-config');
    });
    it('uses the correct default config depending on the `isDistributable` parameter', () => {
        let config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, {}, false);
        expect(config.getConfig('serviceName')).toEqual(expect.objectContaining({
            serverUrl: expect.any(String),
            secretToken: expect.any(String),
        }));
        config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, {}, true);
        expect(Object.keys(config.getConfig('serviceName'))).not.toContain('serverUrl');
    });
    it('loads the configuration from the opensearchDashboards config file', () => {
        const opensearchDashboardsConfig = {
            opensearch: {
                apm: {
                    active: true,
                    serverUrl: 'https://url',
                    secretToken: 'secret',
                },
            },
        };
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, opensearchDashboardsConfig, true);
        expect(config.getConfig('serviceName')).toEqual(expect.objectContaining({
            active: true,
            serverUrl: 'https://url',
            secretToken: 'secret',
        }));
    });
    it('loads the configuration from the dev config is present', () => {
        config_test_mocks_1.devConfigMock.raw = {
            active: true,
            serverUrl: 'https://dev-url.co',
        };
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, {}, true);
        expect(config.getConfig('serviceName')).toEqual(expect.objectContaining({
            active: true,
            serverUrl: 'https://dev-url.co',
        }));
    });
    it('respect the precedence of the dev config', () => {
        const opensearchDashboardsConfig = {
            opensearch: {
                apm: {
                    active: true,
                    serverUrl: 'https://url',
                    secretToken: 'secret',
                },
            },
        };
        config_test_mocks_1.devConfigMock.raw = {
            active: true,
            serverUrl: 'https://dev-url.co',
        };
        const config = new config_1.ApmConfiguration(config_test_mocks_1.mockedRootDir, opensearchDashboardsConfig, true);
        expect(config.getConfig('serviceName')).toEqual(expect.objectContaining({
            active: true,
            serverUrl: 'https://dev-url.co',
            secretToken: 'secret',
        }));
    });
});
