/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.kafka.consumer;

import com.amazonaws.services.schemaregistry.serializers.json.JsonDataWithSchema;
import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.annotations.VisibleForTesting;
import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.avro.generic.GenericRecord;
import org.apache.commons.lang3.Range;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.RebalanceInProgressException;
import org.apache.kafka.common.errors.RecordDeserializationException;
import org.apache.kafka.common.header.Header;
import org.apache.kafka.common.header.Headers;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.model.buffer.Buffer;
import org.opensearch.dataprepper.model.buffer.SizeOverflowException;
import org.opensearch.dataprepper.model.codec.ByteDecoder;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.event.EventMetadata;
import org.opensearch.dataprepper.model.event.JacksonEvent;
import org.opensearch.dataprepper.model.log.JacksonLog;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.plugins.codec.CompressionOption;
import org.opensearch.dataprepper.plugins.kafka.configuration.KafkaConsumerConfig;
import org.opensearch.dataprepper.plugins.kafka.configuration.KafkaKeyMode;
import org.opensearch.dataprepper.plugins.kafka.configuration.TopicConsumerConfig;
import org.opensearch.dataprepper.plugins.kafka.consumer.CommitOffsetRange;
import org.opensearch.dataprepper.plugins.kafka.consumer.PauseConsumePredicate;
import org.opensearch.dataprepper.plugins.kafka.consumer.TopicPartitionCommitTracker;
import org.opensearch.dataprepper.plugins.kafka.util.KafkaTopicConsumerMetrics;
import org.opensearch.dataprepper.plugins.kafka.util.LogRateLimiter;
import org.opensearch.dataprepper.plugins.kafka.util.MessageFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.glue.model.AccessDeniedException;

public class KafkaCustomConsumer
implements Runnable,
ConsumerRebalanceListener {
    private static final Logger LOG = LoggerFactory.getLogger(KafkaCustomConsumer.class);
    private static final Long COMMIT_OFFSET_INTERVAL_MS = 300000L;
    private static final int RETRY_ON_EXCEPTION_SLEEP_MS = 1000;
    private static final int BUFFER_WRITE_TIMEOUT = 2000;
    static final String DEFAULT_KEY = "message";
    private volatile long lastCommitTime;
    private KafkaConsumer consumer = null;
    private AtomicBoolean shutdownInProgress;
    private final String topicName;
    private final TopicConsumerConfig topicConfig;
    private MessageFormat schema;
    private boolean paused;
    private final Buffer<Record<Event>> buffer;
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private final JsonFactory jsonFactory = new JsonFactory();
    private Map<TopicPartition, OffsetAndMetadata> offsetsToCommit;
    private Map<TopicPartition, Long> ownedPartitionsEpoch;
    private Set<TopicPartition> partitionsToReset;
    private final AcknowledgementSetManager acknowledgementSetManager;
    private final Map<Integer, TopicPartitionCommitTracker> partitionCommitTrackerMap;
    private List<Map<TopicPartition, CommitOffsetRange>> acknowledgedOffsets;
    private final boolean acknowledgementsEnabled;
    private final Duration acknowledgementsTimeout;
    private final KafkaTopicConsumerMetrics topicMetrics;
    private final PauseConsumePredicate pauseConsumePredicate;
    private long metricsUpdatedTime;
    private final AtomicInteger numberOfAcksPending;
    private long numRecordsCommitted = 0L;
    private final LogRateLimiter errLogRateLimiter;
    private final ByteDecoder byteDecoder;
    private final long maxRetriesOnException;
    private final Map<Integer, Long> partitionToLastReceivedTimestampMillis;
    private final CompressionOption compressionConfig;

    public KafkaCustomConsumer(KafkaConsumer consumer, AtomicBoolean shutdownInProgress, Buffer<Record<Event>> buffer, KafkaConsumerConfig consumerConfig, TopicConsumerConfig topicConfig, String schemaType, AcknowledgementSetManager acknowledgementSetManager, ByteDecoder byteDecoder, KafkaTopicConsumerMetrics topicMetrics, PauseConsumePredicate pauseConsumePredicate, CompressionOption compressionConfig) {
        this.topicName = topicConfig.getName();
        this.topicConfig = topicConfig;
        this.shutdownInProgress = shutdownInProgress;
        this.consumer = consumer;
        this.buffer = buffer;
        this.paused = false;
        this.byteDecoder = byteDecoder;
        this.topicMetrics = topicMetrics;
        this.maxRetriesOnException = topicConfig.getMaxPollInterval().toMillis() / 6000L;
        this.pauseConsumePredicate = pauseConsumePredicate;
        this.topicMetrics.register(consumer);
        this.offsetsToCommit = new HashMap<TopicPartition, OffsetAndMetadata>();
        this.partitionToLastReceivedTimestampMillis = new HashMap<Integer, Long>();
        this.ownedPartitionsEpoch = new HashMap<TopicPartition, Long>();
        this.metricsUpdatedTime = Instant.now().getEpochSecond();
        this.acknowledgedOffsets = new ArrayList<Map<TopicPartition, CommitOffsetRange>>();
        this.acknowledgementsTimeout = Duration.ofSeconds(Integer.MAX_VALUE);
        this.acknowledgementsEnabled = consumerConfig.getAcknowledgementsEnabled();
        this.acknowledgementSetManager = acknowledgementSetManager;
        this.partitionCommitTrackerMap = new HashMap<Integer, TopicPartitionCommitTracker>();
        this.partitionsToReset = Collections.synchronizedSet(new HashSet());
        this.schema = MessageFormat.getByMessageFormatByName(schemaType);
        this.lastCommitTime = System.currentTimeMillis();
        this.numberOfAcksPending = new AtomicInteger(0);
        this.errLogRateLimiter = new LogRateLimiter(2, System.currentTimeMillis());
        this.compressionConfig = compressionConfig == null ? CompressionOption.NONE : compressionConfig;
    }

    public KafkaCustomConsumer(KafkaConsumer consumer, AtomicBoolean shutdownInProgress, Buffer<Record<Event>> buffer, KafkaConsumerConfig consumerConfig, TopicConsumerConfig topicConfig, String schemaType, AcknowledgementSetManager acknowledgementSetManager, ByteDecoder byteDecoder, KafkaTopicConsumerMetrics topicMetrics, PauseConsumePredicate pauseConsumePredicate) {
        this(consumer, shutdownInProgress, buffer, consumerConfig, topicConfig, schemaType, acknowledgementSetManager, byteDecoder, topicMetrics, pauseConsumePredicate, CompressionOption.NONE);
    }

    KafkaTopicConsumerMetrics getTopicMetrics() {
        return this.topicMetrics;
    }

    <T> long getRecordTimeStamp(ConsumerRecord<String, T> consumerRecord, long nowMs) {
        long timestamp = consumerRecord.timestamp();
        int partition = consumerRecord.partition();
        if (timestamp > nowMs) {
            this.topicMetrics.getNumberOfInvalidTimeStamps().increment();
            if (this.partitionToLastReceivedTimestampMillis.containsKey(partition)) {
                return this.partitionToLastReceivedTimestampMillis.get(partition);
            }
            return nowMs;
        }
        this.partitionToLastReceivedTimestampMillis.put(partition, timestamp);
        return timestamp;
    }

    private long getCurrentTimeNanos() {
        Instant now = Instant.now();
        return now.getEpochSecond() * 1000000000L + (long)now.getNano();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateOffsetsToCommit(TopicPartition partition, OffsetAndMetadata offsetAndMetadata) {
        if (Objects.isNull(offsetAndMetadata)) {
            return;
        }
        Map<TopicPartition, OffsetAndMetadata> map = this.offsetsToCommit;
        synchronized (map) {
            this.offsetsToCommit.put(partition, offsetAndMetadata);
        }
    }

    private AcknowledgementSet createAcknowledgementSet(Map<TopicPartition, CommitOffsetRange> offsets) {
        AcknowledgementSet acknowledgementSet = this.acknowledgementSetManager.create(result -> {
            this.numberOfAcksPending.decrementAndGet();
            if (result.booleanValue()) {
                this.topicMetrics.getNumberOfPositiveAcknowledgements().increment();
                KafkaCustomConsumer kafkaCustomConsumer = this;
                synchronized (kafkaCustomConsumer) {
                    this.acknowledgedOffsets.add(offsets);
                }
            }
            this.topicMetrics.getNumberOfNegativeAcknowledgements().increment();
            KafkaCustomConsumer kafkaCustomConsumer = this;
            synchronized (kafkaCustomConsumer) {
                offsets.forEach((partition, offsetRange) -> this.partitionsToReset.add((TopicPartition)partition));
            }
        }, this.acknowledgementsTimeout);
        return acknowledgementSet;
    }

    <T> ConsumerRecords<String, T> doPoll() throws Exception {
        this.topicMetrics.recordTimeBetweenPolls();
        ConsumerRecords records = this.consumer.poll(Duration.ofMillis(this.topicConfig.getThreadWaitingTime().toMillis() / 2L));
        return records;
    }

    <T> void consumeRecords() throws Exception {
        try {
            ConsumerRecords<String, T> records = this.doPoll();
            LOG.debug("Consumed records with count {}", (Object)records.count());
            if (Objects.nonNull(records) && !records.isEmpty() && records.count() > 0) {
                HashMap<TopicPartition, CommitOffsetRange> offsets = new HashMap<TopicPartition, CommitOffsetRange>();
                AcknowledgementSet acknowledgementSet = null;
                if (this.acknowledgementsEnabled) {
                    acknowledgementSet = this.createAcknowledgementSet(offsets);
                }
                this.iterateRecordPartitions(records, acknowledgementSet, offsets);
                if (!this.acknowledgementsEnabled) {
                    offsets.forEach((partition, offsetRange) -> {
                        this.updateOffsetsToCommit((TopicPartition)partition, new OffsetAndMetadata((Long)offsetRange.getOffsets().getMaximum() + 1L));
                        this.numRecordsCommitted += (Long)offsetRange.getOffsets().getMaximum() - (Long)offsetRange.getOffsets().getMinimum() + 1L;
                    });
                } else {
                    acknowledgementSet.complete();
                    this.numberOfAcksPending.incrementAndGet();
                }
            }
        }
        catch (AuthenticationException e) {
            LOG.warn("Authentication error while doing poll(). Will retry after 10 seconds", (Throwable)e);
            this.topicMetrics.getNumberOfPollAuthErrors().increment();
            Thread.sleep(10000L);
        }
        catch (RecordDeserializationException e) {
            LOG.warn("Deserialization error - topic {} partition {} offset {}. Error message: {}", new Object[]{e.topicPartition().topic(), e.topicPartition().partition(), e.offset(), e.getMessage()});
            if (ExceptionUtils.getRootCause((Throwable)e) instanceof AccessDeniedException) {
                LOG.warn("AccessDenied for AWSGlue schema registry, retrying after 30 seconds");
                Thread.sleep(30000L);
            } else {
                LOG.warn("Seeking past the error record");
                this.consumer.seek(e.topicPartition(), e.offset() + 1L);
                if (this.acknowledgementsEnabled && this.partitionCommitTrackerMap.containsKey(e.topicPartition().partition())) {
                    this.addAcknowledgedOffsets(e.topicPartition(), (Range<Long>)Range.of((Comparable)Long.valueOf(e.offset()), (Comparable)Long.valueOf(e.offset())));
                }
            }
            this.topicMetrics.getNumberOfDeserializationErrors().increment();
        }
    }

    private void addAcknowledgedOffsets(TopicPartition topicPartition, Range<Long> offsetRange) {
        int partitionId = topicPartition.partition();
        TopicPartitionCommitTracker commitTracker = this.partitionCommitTrackerMap.get(partitionId);
        if (Objects.isNull(commitTracker) && this.errLogRateLimiter.isAllowed(System.currentTimeMillis())) {
            LOG.error("Commit tracker not found for TopicPartition: {}", (Object)topicPartition);
            return;
        }
        OffsetAndMetadata offsetAndMetadata = commitTracker.addCompletedOffsets(offsetRange);
        this.updateOffsetsToCommit(topicPartition, offsetAndMetadata);
    }

    private void resetOffsets() {
        long currentTimeMillis = System.currentTimeMillis();
        if (currentTimeMillis - this.lastCommitTime < this.topicConfig.getCommitInterval().toMillis()) {
            return;
        }
        if (this.partitionsToReset.size() > 0) {
            this.partitionsToReset.forEach(partition -> {
                try {
                    OffsetAndMetadata offsetAndMetadata = this.consumer.committed(partition);
                    if (Objects.isNull(offsetAndMetadata)) {
                        LOG.info("Seeking partition {} to the beginning", partition);
                        this.consumer.seekToBeginning(List.of(partition));
                    } else {
                        LOG.info("Seeking partition {} to {}", partition, (Object)offsetAndMetadata.offset());
                        this.consumer.seek(partition, offsetAndMetadata);
                    }
                    this.partitionCommitTrackerMap.remove(partition.partition());
                    long epoch = this.getCurrentTimeNanos();
                    this.ownedPartitionsEpoch.put((TopicPartition)partition, epoch);
                }
                catch (Exception e) {
                    LOG.error("Failed to seek to last committed offset upon negative acknowledgement {}", partition, (Object)e);
                }
            });
            this.partitionsToReset.clear();
        }
    }

    void processAcknowledgedOffsets() {
        this.acknowledgedOffsets.forEach(offsets -> offsets.forEach((partition, offsetRange) -> {
            if (this.getPartitionEpoch((TopicPartition)partition) == offsetRange.getEpoch()) {
                try {
                    this.addAcknowledgedOffsets((TopicPartition)partition, offsetRange.getOffsets());
                }
                catch (Exception e) {
                    LOG.error("Failed committed offsets upon positive acknowledgement {}", partition, (Object)e);
                }
            }
        }));
        this.acknowledgedOffsets.clear();
    }

    private void updateCommitCountMetric(TopicPartition topicPartition, OffsetAndMetadata offsetAndMetadata) {
        if (this.acknowledgementsEnabled) {
            TopicPartitionCommitTracker commitTracker = this.partitionCommitTrackerMap.get(topicPartition.partition());
            if (Objects.isNull(commitTracker) && this.errLogRateLimiter.isAllowed(System.currentTimeMillis())) {
                LOG.error("Commit tracker not found for TopicPartition: {}", (Object)topicPartition);
                return;
            }
            this.topicMetrics.getNumberOfRecordsCommitted().increment((double)commitTracker.getCommittedRecordCount());
        } else {
            this.topicMetrics.getNumberOfRecordsCommitted().increment((double)this.numRecordsCommitted);
            this.numRecordsCommitted = 0L;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void commitOffsets(boolean forceCommit) {
        if (this.topicConfig.getAutoCommit().booleanValue()) {
            return;
        }
        this.processAcknowledgedOffsets();
        long currentTimeMillis = System.currentTimeMillis();
        if (!forceCommit && currentTimeMillis - this.lastCommitTime < this.topicConfig.getCommitInterval().toMillis()) {
            return;
        }
        Map<TopicPartition, OffsetAndMetadata> map = this.offsetsToCommit;
        synchronized (map) {
            if (this.offsetsToCommit.isEmpty()) {
                return;
            }
            this.offsetsToCommit.forEach((partition, offset) -> this.updateCommitCountMetric((TopicPartition)partition, (OffsetAndMetadata)offset));
            try {
                this.consumer.commitSync(this.offsetsToCommit);
                this.lastCommitTime = currentTimeMillis;
            }
            catch (RebalanceInProgressException ex) {
                LOG.error("Failed to commit offsets in topic {} due to rebalance in progress", (Object)this.topicName, (Object)ex);
                return;
            }
            catch (Exception e) {
                LOG.error("Failed to commit offsets in topic {}", (Object)this.topicName, (Object)e);
            }
            this.offsetsToCommit.clear();
        }
    }

    @VisibleForTesting
    Map<TopicPartition, OffsetAndMetadata> getOffsetsToCommit() {
        return this.offsetsToCommit;
    }

    @VisibleForTesting
    Long getNumRecordsCommitted() {
        return this.numRecordsCommitted;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void run() {
        KafkaCustomConsumer kafkaCustomConsumer;
        this.consumer.subscribe(Arrays.asList(this.topicName), (ConsumerRebalanceListener)this);
        Set partitions = this.consumer.assignment();
        long currentEpoch = this.getCurrentTimeNanos();
        partitions.forEach(partition -> {
            OffsetAndMetadata offsetAndMetadata = this.consumer.committed(partition);
            LOG.info("Starting consumer with topic partition ({}) offset {}", partition, (Object)offsetAndMetadata);
            this.ownedPartitionsEpoch.put((TopicPartition)partition, currentEpoch);
        });
        boolean retryingAfterException = false;
        while (!this.shutdownInProgress.get()) {
            LOG.debug("Still running Kafka consumer in start of loop");
            try {
                if (retryingAfterException) {
                    LOG.debug("Pause consuming from Kafka topic due a previous exception.");
                    Thread.sleep(10000L);
                } else {
                    if (this.pauseConsumePredicate.pauseConsuming()) {
                        LOG.debug("Pause and skip consuming from Kafka topic due to an external condition: {}", (Object)this.pauseConsumePredicate);
                        this.paused = true;
                        this.consumer.pause((Collection)this.consumer.assignment());
                        Thread.sleep(1000L);
                        continue;
                    }
                    if (this.paused) {
                        LOG.debug("Resume consuming from Kafka topic.");
                        this.paused = false;
                        this.consumer.resume((Collection)this.consumer.assignment());
                    }
                }
                LOG.debug("Still running Kafka consumer preparing to commit offsets and consume records");
                kafkaCustomConsumer = this;
                synchronized (kafkaCustomConsumer) {
                    this.commitOffsets(false);
                    this.resetOffsets();
                }
                this.consumeRecords();
                LOG.debug("Exited consume records");
                this.topicMetrics.update(this.consumer);
                LOG.debug("Updated consumer metrics");
                retryingAfterException = false;
            }
            catch (Exception exp) {
                LOG.error("Error while reading the records from the topic {}. Retry after 10 seconds", (Object)this.topicName, (Object)exp);
                retryingAfterException = true;
            }
        }
        LOG.info("Shutting down, number of acks pending = {}", (Object)this.numberOfAcksPending.get());
        kafkaCustomConsumer = this;
        synchronized (kafkaCustomConsumer) {
            this.commitOffsets(true);
        }
    }

    private <T> Record<Event> getRecord(ConsumerRecord<String, T> consumerRecord, int partition) {
        Headers headers;
        Instant now = Instant.now();
        Map<String, Object> data = new HashMap<String, Object>();
        Object value = consumerRecord.value();
        String key = (String)consumerRecord.key();
        KafkaKeyMode kafkaKeyMode = this.topicConfig.getKafkaKeyMode();
        boolean plainTextMode = false;
        try {
            if (value instanceof JsonDataWithSchema) {
                JsonDataWithSchema j = (JsonDataWithSchema)consumerRecord.value();
                value = objectMapper.readValue(j.getPayload(), Map.class);
            } else if (this.schema == MessageFormat.AVRO || value instanceof GenericRecord) {
                JsonParser jsonParser = this.jsonFactory.createParser(consumerRecord.value().toString());
                value = objectMapper.readValue(jsonParser, Map.class);
            } else if (this.schema == MessageFormat.PLAINTEXT) {
                value = (String)consumerRecord.value();
                plainTextMode = true;
            } else if (this.schema == MessageFormat.JSON) {
                value = objectMapper.convertValue(value, Map.class);
            }
        }
        catch (Exception e) {
            LOG.error("Failed to parse JSON or AVRO record", (Throwable)e);
            this.topicMetrics.getNumberOfRecordsFailedToParse().increment();
        }
        if (!plainTextMode) {
            if (!(value instanceof Map)) {
                data.put(key, value);
            } else {
                Map valueMap = (Map)value;
                if (kafkaKeyMode == KafkaKeyMode.INCLUDE_AS_FIELD) {
                    valueMap.put("kafka_key", key);
                }
                data = valueMap;
            }
        } else {
            if (Objects.isNull(key)) {
                key = DEFAULT_KEY;
            }
            data.put(key, value);
        }
        JacksonEvent event = JacksonLog.builder().withData(data).build();
        EventMetadata eventMetadata = event.getMetadata();
        if (kafkaKeyMode == KafkaKeyMode.INCLUDE_AS_METADATA) {
            eventMetadata.setAttribute("kafka_key", (Object)key);
        }
        if ((headers = consumerRecord.headers()) != null) {
            HashMap<String, byte[]> headerData = new HashMap<String, byte[]>();
            for (Header header : headers) {
                headerData.put(header.key(), header.value());
            }
            eventMetadata.setAttribute("kafka_headers", headerData);
        }
        long receivedTimeStamp = this.getRecordTimeStamp(consumerRecord, now.toEpochMilli());
        eventMetadata.setAttribute("kafka_timestamp", (Object)receivedTimeStamp);
        eventMetadata.setAttribute("kafka_timestamp_type", (Object)consumerRecord.timestampType().toString());
        eventMetadata.setAttribute("kafka_topic", (Object)this.topicName);
        eventMetadata.setAttribute("kafka_partition", (Object)String.valueOf(partition));
        eventMetadata.setExternalOriginationTime(Instant.ofEpochMilli(receivedTimeStamp));
        event.getEventHandle().setExternalOriginationTime(Instant.ofEpochMilli(receivedTimeStamp));
        return new Record((Object)event);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processRecords(AcknowledgementSet acknowledgementSet, List<Record<Event>> eventRecords) {
        if (acknowledgementSet != null) {
            eventRecords.forEach(record -> acknowledgementSet.add((Event)record.getData()));
        }
        long numRetries = 0L;
        while (true) {
            LOG.debug("In while loop for processing records, paused = {}", (Object)this.paused);
            try {
                this.buffer.writeAll(eventRecords, 2000);
            }
            catch (Exception e) {
                if (!this.paused && numRetries++ > this.maxRetriesOnException) {
                    this.paused = true;
                    LOG.debug("Preparing to call pause");
                    this.consumer.pause((Collection)this.consumer.assignment());
                    LOG.debug("Pause was called");
                }
                if (e instanceof SizeOverflowException) {
                    this.topicMetrics.getNumberOfBufferSizeOverflows().increment();
                } else {
                    LOG.debug("Error while adding record to buffer, retrying ", (Throwable)e);
                }
                try {
                    LOG.debug("Sleeping due to exception");
                    Thread.sleep(1000L);
                    if (!this.paused) continue;
                    LOG.debug("Calling doPoll()");
                    ConsumerRecords records = this.doPoll();
                    if (records.count() <= 0) continue;
                    LOG.warn("Unexpected records received while the consumer is paused. Resetting the partitions to retry from last read pointer");
                    KafkaCustomConsumer kafkaCustomConsumer = this;
                    synchronized (kafkaCustomConsumer) {
                        this.partitionsToReset.addAll(this.consumer.assignment());
                    }
                }
                catch (Exception exception) {
                    continue;
                }
            }
            break;
        }
        if (this.paused) {
            LOG.debug("Resuming consumption");
            this.consumer.resume((Collection)this.consumer.assignment());
            this.paused = false;
        }
    }

    private <T> void iterateRecordPartitions(ConsumerRecords<String, T> records, AcknowledgementSet acknowledgementSet, Map<TopicPartition, CommitOffsetRange> offsets) throws Exception {
        for (TopicPartition topicPartition : records.partitions()) {
            long partitionEpoch = this.getPartitionEpoch(topicPartition);
            if (this.acknowledgementsEnabled && partitionEpoch == 0L) {
                if (!this.errLogRateLimiter.isAllowed(System.currentTimeMillis())) continue;
                LOG.error("Lost ownership of partition {}", (Object)topicPartition);
                continue;
            }
            List partitionRecords = records.records(topicPartition);
            ArrayList<Record<Event>> eventRecords = new ArrayList<Record<Event>>();
            for (ConsumerRecord consumerRecord : partitionRecords) {
                if (this.schema == MessageFormat.BYTES) {
                    ByteArrayInputStream byteInputStream = new ByteArrayInputStream((byte[])consumerRecord.value());
                    InputStream decompressedInputStream = this.compressionConfig.getDecompressionEngine().createInputStream((InputStream)byteInputStream);
                    if (this.byteDecoder != null) {
                        long receivedTimeStamp = this.getRecordTimeStamp(consumerRecord, Instant.now().toEpochMilli());
                        this.byteDecoder.parse(decompressedInputStream, Instant.ofEpochMilli(receivedTimeStamp), eventRecords::add);
                        continue;
                    }
                    JsonNode jsonNode = (JsonNode)objectMapper.readValue(decompressedInputStream, JsonNode.class);
                    JacksonEvent event = JacksonLog.builder().withData((Object)jsonNode).build();
                    Record record = new Record((Object)event);
                    eventRecords.add((Record<Event>)record);
                    continue;
                }
                Record<Event> record = this.getRecord(consumerRecord, topicPartition.partition());
                if (record == null) continue;
                eventRecords.add(record);
            }
            this.processRecords(acknowledgementSet, eventRecords);
            long lastOffset = ((ConsumerRecord)partitionRecords.get(partitionRecords.size() - 1)).offset();
            long firstOffset = ((ConsumerRecord)partitionRecords.get(0)).offset();
            Range offsetRange = Range.between((Comparable)Long.valueOf(firstOffset), (Comparable)Long.valueOf(lastOffset));
            offsets.put(topicPartition, new CommitOffsetRange((Range<Long>)offsetRange, partitionEpoch));
            if (!this.acknowledgementsEnabled || this.partitionCommitTrackerMap.containsKey(topicPartition.partition())) continue;
            this.partitionCommitTrackerMap.put(topicPartition.partition(), new TopicPartitionCommitTracker(topicPartition, firstOffset));
        }
    }

    public void closeConsumer() {
        this.consumer.close();
    }

    public void shutdownConsumer() {
        this.consumer.wakeup();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
        KafkaCustomConsumer kafkaCustomConsumer = this;
        synchronized (kafkaCustomConsumer) {
            long epoch = this.getCurrentTimeNanos();
            for (TopicPartition topicPartition : partitions) {
                if (this.ownedPartitionsEpoch.containsKey(topicPartition)) {
                    LOG.info("Partition {} already owned", (Object)topicPartition);
                    continue;
                }
                LOG.info("Assigned partition {}", (Object)topicPartition);
                this.ownedPartitionsEpoch.put(topicPartition, epoch);
            }
            if (this.paused) {
                this.consumer.pause((Collection)this.consumer.assignment());
            }
        }
        this.dumpTopicPartitionOffsets(partitions);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
        KafkaCustomConsumer kafkaCustomConsumer = this;
        synchronized (kafkaCustomConsumer) {
            this.commitOffsets(true);
            for (TopicPartition topicPartition : partitions) {
                if (!this.ownedPartitionsEpoch.containsKey(topicPartition)) {
                    LOG.info("Partition {} not owned", (Object)topicPartition);
                    continue;
                }
                LOG.info("Revoked partition {}", (Object)topicPartition);
                this.ownedPartitionsEpoch.remove(topicPartition);
                this.partitionCommitTrackerMap.remove(topicPartition.partition());
            }
            if (this.paused) {
                this.consumer.pause((Collection)this.consumer.assignment());
            }
        }
    }

    private long getPartitionEpoch(TopicPartition topicPartition) {
        return this.ownedPartitionsEpoch.getOrDefault(topicPartition, 0L);
    }

    final void dumpTopicPartitionOffsets(Collection<TopicPartition> partitions) {
        try {
            Map committedOffsets = this.consumer.committed(new HashSet<TopicPartition>(partitions));
            Map beginningOffsets = this.consumer.beginningOffsets(partitions);
            Map endOffsets = this.consumer.endOffsets(partitions);
            for (TopicPartition topicPartition : partitions) {
                OffsetAndMetadata offsetAndMetadata = (OffsetAndMetadata)committedOffsets.get(topicPartition);
                LOG.info("Partition {} offsets: beginningOffset: {}, endOffset: {}, committedOffset: {}", new Object[]{topicPartition, this.getTopicPartitionOffset(beginningOffsets, topicPartition), this.getTopicPartitionOffset(endOffsets, topicPartition), Objects.isNull(offsetAndMetadata) ? "-" : Long.valueOf(offsetAndMetadata.offset())});
            }
        }
        catch (Exception e) {
            LOG.error("Failed to get offsets in onPartitionsAssigned callback", (Throwable)e);
        }
    }

    final String getTopicPartitionOffset(Map<TopicPartition, Long> offsetMap, TopicPartition topicPartition) {
        Long offset = offsetMap.get(topicPartition);
        return Objects.isNull(offset) ? "-" : offset.toString();
    }
}

