/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.s3.common.source;

import com.google.common.base.Preconditions;
import com.google.common.io.ByteStreams;
import dev.failsafe.Failsafe;
import dev.failsafe.FailsafeException;
import dev.failsafe.Policy;
import dev.failsafe.RetryPolicy;
import dev.failsafe.RetryPolicyBuilder;
import dev.failsafe.function.CheckedSupplier;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.net.SocketException;
import java.net.SocketTimeoutException;
import java.nio.ByteBuffer;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.atomic.LongAdder;
import org.apache.http.ConnectionClosedException;
import org.apache.parquet.io.SeekableInputStream;
import org.opensearch.dataprepper.plugins.s3.common.ownership.BucketOwnerProvider;
import org.opensearch.dataprepper.plugins.s3.common.source.S3ObjectPluginMetrics;
import org.opensearch.dataprepper.plugins.s3.common.source.S3ObjectReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.http.Abortable;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.HeadObjectResponse;
import software.amazon.awssdk.services.s3.model.S3Exception;

class S3InputStream
extends SeekableInputStream {
    static final List<Class<? extends Throwable>> RETRYABLE_EXCEPTIONS = List.of(ConnectionClosedException.class, EOFException.class, SocketException.class, SocketTimeoutException.class);
    private static final int COPY_BUFFER_SIZE = 8192;
    private static final Logger LOG = LoggerFactory.getLogger(S3InputStream.class);
    private static final int SKIP_SIZE = 0x100000;
    private final S3Client s3Client;
    private final S3ObjectReference s3ObjectReference;
    private final HeadObjectResponse metadata;
    private final S3ObjectPluginMetrics s3ObjectPluginMetrics;
    private final LongAdder bytesCounter;
    private final GetObjectRequest.Builder getObjectRequestBuilder;
    private InputStream stream;
    private final byte[] temp = new byte[8192];
    private long pos = 0L;
    private long next = 0L;
    private long mark = 0L;
    private long markLimit = 0L;
    private boolean closed = false;
    private RetryPolicy<byte[]> retryPolicyReturningByteArray;
    private RetryPolicy<Integer> retryPolicyReturningInteger;

    public S3InputStream(S3Client s3Client, S3ObjectReference s3ObjectReference, BucketOwnerProvider bucketOwnerProvider, HeadObjectResponse metadata, S3ObjectPluginMetrics s3ObjectPluginMetrics, Duration retryDelay, int retries) {
        this.s3Client = s3Client;
        this.s3ObjectReference = s3ObjectReference;
        this.metadata = metadata;
        this.s3ObjectPluginMetrics = s3ObjectPluginMetrics;
        this.bytesCounter = new LongAdder();
        this.getObjectRequestBuilder = GetObjectRequest.builder().bucket(this.s3ObjectReference.getBucketName()).key(this.s3ObjectReference.getKey());
        bucketOwnerProvider.getBucketOwner(this.s3ObjectReference.getBucketName()).ifPresent(arg_0 -> ((GetObjectRequest.Builder)this.getObjectRequestBuilder).expectedBucketOwner(arg_0));
        this.retryPolicyReturningByteArray = ((RetryPolicyBuilder)RetryPolicy.builder().handle(RETRYABLE_EXCEPTIONS)).withDelay(retryDelay).withMaxRetries(retries).build();
        this.retryPolicyReturningInteger = ((RetryPolicyBuilder)RetryPolicy.builder().handle(RETRYABLE_EXCEPTIONS)).withDelay(retryDelay).withMaxRetries(retries).build();
    }

    public int available() throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        return this.stream.available();
    }

    public void close() throws IOException {
        super.close();
        this.closed = true;
        this.closeStream();
        this.s3ObjectPluginMetrics.getS3ObjectSizeProcessedSummary().record(this.bytesCounter.doubleValue());
    }

    public synchronized void mark(int readlimit) {
        this.mark = this.next;
        this.markLimit = this.mark + (long)readlimit;
    }

    public synchronized boolean markSupported() {
        return true;
    }

    public int read() throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        int byteRead = this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> this.stream.read()));
        if (byteRead != -1) {
            ++this.pos;
            ++this.next;
            this.bytesCounter.increment();
        }
        return byteRead;
    }

    public int read(byte[] b) throws IOException {
        return this.read(b, 0, b.length);
    }

    public int read(byte[] b, int off, int len) throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        int bytesRead = this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> this.stream.read(b, off, len)));
        if (bytesRead > 0) {
            this.pos += (long)bytesRead;
            this.next += (long)bytesRead;
            this.bytesCounter.add(bytesRead);
        }
        return bytesRead;
    }

    public byte[] readAllBytes() throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        byte[] bytesRead = this.executeWithRetriesAndReturnByteArray((CheckedSupplier<byte[]>)((CheckedSupplier)() -> this.stream.readAllBytes()));
        this.pos += (long)bytesRead.length;
        this.next += (long)bytesRead.length;
        this.bytesCounter.add(bytesRead.length);
        return bytesRead;
    }

    public int readNBytes(byte[] b, int off, int len) throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        int bytesRead = this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> this.stream.readNBytes(b, off, len)));
        if (bytesRead > 0) {
            this.pos += (long)bytesRead;
            this.next += (long)bytesRead;
            this.bytesCounter.add(bytesRead);
        }
        return bytesRead;
    }

    public byte[] readNBytes(int len) throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        byte[] bytesRead = this.executeWithRetriesAndReturnByteArray((CheckedSupplier<byte[]>)((CheckedSupplier)() -> this.stream.readNBytes(len)));
        this.pos += (long)bytesRead.length;
        this.next += (long)bytesRead.length;
        this.bytesCounter.add(bytesRead.length);
        return bytesRead;
    }

    public synchronized void reset() throws IOException {
        if (this.next > this.markLimit) {
            throw new IOException("Cannot reset stream because mark limit exceeded");
        }
        this.next = this.mark;
    }

    public long skip(long n) {
        if (this.next >= this.metadata.contentLength()) {
            return 0L;
        }
        long toSkip = Math.min(n, this.metadata.contentLength() - this.next);
        this.next += toSkip;
        return toSkip;
    }

    public long getPos() {
        return this.next;
    }

    public void seek(long newPos) {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        Preconditions.checkArgument((newPos >= 0L ? 1 : 0) != 0, (String)"position is negative: %s", (long)newPos);
        this.next = newPos;
    }

    public void readFully(byte[] bytes) throws IOException {
        this.readFully(bytes, 0, bytes.length);
    }

    public void readFully(byte[] bytes, int start, int len) throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        int bytesRead = this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> S3InputStream.readFully(this.stream, bytes, start, len)));
        if (bytesRead > 0) {
            this.pos += (long)bytesRead;
            this.next += (long)bytesRead;
            this.bytesCounter.add(bytesRead);
        }
    }

    public int read(ByteBuffer buf) throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        int bytesRead = 0;
        bytesRead = buf.hasArray() ? this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> S3InputStream.readHeapBuffer(this.stream, buf))) : this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> S3InputStream.readDirectBuffer(this.stream, buf, this.temp)));
        if (bytesRead > 0) {
            this.pos += (long)bytesRead;
            this.next += (long)bytesRead;
            this.bytesCounter.add(bytesRead);
        }
        return bytesRead;
    }

    public void readFully(ByteBuffer buf) throws IOException {
        Preconditions.checkState((!this.closed ? 1 : 0) != 0, (Object)"Cannot read: already closed");
        this.positionStream();
        int bytesRead = 0;
        bytesRead = buf.hasArray() ? this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> S3InputStream.readFullyHeapBuffer(this.stream, buf))) : this.executeWithRetriesAndReturnInt((CheckedSupplier<Integer>)((CheckedSupplier)() -> S3InputStream.readFullyDirectBuffer(this.stream, buf, this.temp)));
        if (bytesRead > 0) {
            this.pos += (long)bytesRead;
            this.next += (long)bytesRead;
            this.bytesCounter.add(bytesRead);
        }
    }

    private void positionStream() throws IOException {
        long skip;
        if (this.stream != null && this.next == this.pos) {
            return;
        }
        if (this.stream != null && this.next > this.pos && (skip = this.next - this.pos) <= (long)Math.max(this.stream.available(), 0x100000)) {
            LOG.debug("Read-through seek for {} to offset {}", (Object)this.s3ObjectReference, (Object)this.next);
            try {
                ByteStreams.skipFully((InputStream)this.stream, (long)skip);
                this.pos = this.next;
                return;
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        LOG.debug("Seek with new stream for {} to offset {}", (Object)this.s3ObjectReference, (Object)this.next);
        this.pos = this.next;
        this.openStream();
    }

    private void openStream() throws IOException {
        this.closeStream();
        if (this.pos >= this.metadata.contentLength()) {
            this.stream = InputStream.nullInputStream();
            return;
        }
        GetObjectRequest request = (GetObjectRequest)this.getObjectRequestBuilder.range(String.format("bytes=%s-", this.pos)).build();
        try {
            this.stream = (InputStream)this.s3Client.getObject(request, ResponseTransformer.toInputStream());
        }
        catch (Exception ex) {
            LOG.error("Error reading from S3 object: s3ObjectReference={}", (Object)this.s3ObjectReference);
            if (ex instanceof S3Exception) {
                this.recordS3Exception((S3Exception)ex);
            }
            throw new IOException(ex.getMessage());
        }
    }

    private void closeStream() throws IOException {
        if (this.stream != null) {
            block3: {
                this.abortStream();
                try {
                    this.stream.close();
                }
                catch (IOException e) {
                    if (e.getClass().getSimpleName().equals("ConnectionClosedException")) break block3;
                    throw e;
                }
            }
            this.stream = null;
        }
    }

    private void abortStream() {
        try {
            if (this.stream instanceof Abortable) {
                ((Abortable)this.stream).abort();
            }
        }
        catch (Exception e) {
            LOG.warn("An error occurred while aborting the stream", (Throwable)e);
        }
    }

    static int readHeapBuffer(InputStream f, ByteBuffer buf) throws IOException {
        int bytesRead = f.read(buf.array(), buf.arrayOffset() + buf.position(), buf.remaining());
        if (bytesRead < 0) {
            return bytesRead;
        }
        buf.position(buf.position() + bytesRead);
        return bytesRead;
    }

    static int readFully(InputStream f, byte[] bytes, int start, int len) throws IOException {
        int totalBytesRead = 0;
        int offset = start;
        int remaining = len;
        while (remaining > 0) {
            int bytesRead = f.read(bytes, offset, remaining);
            if (bytesRead < 0) {
                throw new EOFException("Reached the end of stream with " + remaining + " bytes left to read");
            }
            remaining -= bytesRead;
            offset += bytesRead;
            totalBytesRead += bytesRead;
        }
        return totalBytesRead;
    }

    static int readFullyHeapBuffer(InputStream f, ByteBuffer buf) throws IOException {
        int bytesRead = S3InputStream.readFully(f, buf.array(), buf.arrayOffset() + buf.position(), buf.remaining());
        buf.position(buf.limit());
        return bytesRead;
    }

    static int readDirectBuffer(InputStream f, ByteBuffer buf, byte[] temp) throws IOException {
        int bytesRead;
        int nextReadLength = Math.min(buf.remaining(), temp.length);
        int totalBytesRead = 0;
        while ((bytesRead = f.read(temp, 0, nextReadLength)) == temp.length) {
            buf.put(temp);
            totalBytesRead += bytesRead;
            nextReadLength = Math.min(buf.remaining(), temp.length);
        }
        if (bytesRead < 0) {
            return totalBytesRead == 0 ? -1 : totalBytesRead;
        }
        buf.put(temp, 0, bytesRead);
        return totalBytesRead += bytesRead;
    }

    static int readFullyDirectBuffer(InputStream f, ByteBuffer buf, byte[] temp) throws IOException {
        int totalBytesRead = 0;
        int nextReadLength = Math.min(buf.remaining(), temp.length);
        int bytesRead = 0;
        while (nextReadLength > 0 && (bytesRead = f.read(temp, 0, nextReadLength)) >= 0) {
            buf.put(temp, 0, bytesRead);
            nextReadLength = Math.min(buf.remaining(), temp.length);
            totalBytesRead += bytesRead;
        }
        if (bytesRead < 0 && buf.remaining() > 0) {
            throw new EOFException("Reached the end of stream with " + buf.remaining() + " bytes left to read");
        }
        return totalBytesRead;
    }

    private void recordS3Exception(S3Exception ex) {
        if (ex.statusCode() == 404) {
            this.s3ObjectPluginMetrics.getS3ObjectsFailedNotFoundCounter().increment();
        } else if (ex.statusCode() == 403) {
            this.s3ObjectPluginMetrics.getS3ObjectsFailedAccessDeniedCounter().increment();
        } else if (ex.isThrottlingException()) {
            this.s3ObjectPluginMetrics.getS3ObjectsThrottledCounter().increment();
        }
    }

    private int executeWithRetriesAndReturnInt(CheckedSupplier<Integer> supplier) throws IOException {
        return this.executeWithRetries(this.retryPolicyReturningInteger, supplier);
    }

    private byte[] executeWithRetriesAndReturnByteArray(CheckedSupplier<byte[]> supplier) throws IOException {
        return this.executeWithRetries(this.retryPolicyReturningByteArray, supplier);
    }

    private <T> T executeWithRetries(RetryPolicy<T> retryPolicy, CheckedSupplier<T> supplier) throws IOException {
        try {
            return (T)Failsafe.with(retryPolicy, (Policy[])new RetryPolicy[0]).get(() -> {
                try {
                    return supplier.get();
                }
                catch (EOFException | SocketException | SocketTimeoutException | ConnectionClosedException e) {
                    LOG.warn("Resetting stream due to underlying socket exception", e);
                    this.openStream();
                    throw e;
                }
            });
        }
        catch (FailsafeException e) {
            LOG.error("Failed to read with Retries", (Throwable)e);
            throw new IOException(e.getCause());
        }
    }
}

