/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.sink.opensearch;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.ImmutableList;
import com.linecorp.armeria.client.retry.Backoff;
import io.micrometer.core.instrument.Counter;
import jakarta.json.stream.JsonParsingException;
import java.io.IOException;
import java.net.SocketTimeoutException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.opensearch.client.opensearch._types.ErrorCause;
import org.opensearch.client.opensearch._types.OpenSearchException;
import org.opensearch.client.opensearch.core.BulkRequest;
import org.opensearch.client.opensearch.core.BulkResponse;
import org.opensearch.client.opensearch.core.bulk.BulkResponseItem;
import org.opensearch.client.opensearch.core.bulk.OperationType;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.plugins.sink.opensearch.BulkOperationWrapper;
import org.opensearch.dataprepper.plugins.sink.opensearch.RequestFunction;
import org.opensearch.dataprepper.plugins.sink.opensearch.bulk.AccumulatingBulkRequest;
import org.opensearch.dataprepper.plugins.sink.opensearch.dlq.FailedBulkOperation;
import org.opensearch.dataprepper.plugins.sink.opensearch.index.ExistingDocumentQueryManager;
import org.opensearch.rest.RestStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class BulkRetryStrategy {
    public static final String DOCUMENTS_SUCCESS = "documentsSuccess";
    public static final String DOCUMENTS_SUCCESS_FIRST_ATTEMPT = "documentsSuccessFirstAttempt";
    public static final String DOCUMENT_ERRORS = "documentErrors";
    public static final String DOCUMENT_STATUSES = "documentStatuses";
    public static final String BULK_REQUEST_FAILED = "bulkRequestFailed";
    public static final String BULK_REQUEST_NUMBER_OF_RETRIES = "bulkRequestNumberOfRetries";
    public static final String BULK_BAD_REQUEST_ERRORS = "bulkBadRequestErrors";
    public static final String BULK_REQUEST_NOT_ALLOWED_ERRORS = "bulkRequestNotAllowedErrors";
    public static final String BULK_REQUEST_INVALID_INPUT_ERRORS = "bulkRequestInvalidInputErrors";
    public static final String BULK_REQUEST_NOT_FOUND_ERRORS = "bulkRequestNotFoundErrors";
    public static final String BULK_REQUEST_TIMEOUT_ERRORS = "bulkRequestTimeoutErrors";
    public static final String BULK_REQUEST_SERVER_ERRORS = "bulkRequestServerErrors";
    public static final String DOCUMENTS_VERSION_CONFLICT_ERRORS = "documentsVersionConflictErrors";
    public static final String DOCUMENTS_DUPLICATES = "documentsDuplicates";
    static final long INITIAL_DELAY_MS = 50L;
    static final long MAXIMUM_DELAY_MS = Duration.ofMinutes(10L).toMillis();
    static final String VERSION_CONFLICT_EXCEPTION_TYPE = "version_conflict_engine_exception";
    private static final int DELETE_404_MAX_RETRIES = 3;
    private static final Set<Integer> NON_RETRY_STATUS = new HashSet<Integer>(Arrays.asList(RestStatus.BAD_REQUEST.getStatus(), RestStatus.NOT_FOUND.getStatus(), RestStatus.CONFLICT.getStatus()));
    private static final Set<Integer> BAD_REQUEST_ERRORS = new HashSet<Integer>(Arrays.asList(RestStatus.BAD_REQUEST.getStatus(), RestStatus.EXPECTATION_FAILED.getStatus(), RestStatus.UNPROCESSABLE_ENTITY.getStatus(), RestStatus.FAILED_DEPENDENCY.getStatus(), RestStatus.NOT_ACCEPTABLE.getStatus()));
    private static final Set<Integer> NOT_ALLOWED_ERRORS = new HashSet<Integer>(Arrays.asList(RestStatus.UNAUTHORIZED.getStatus(), RestStatus.FORBIDDEN.getStatus(), RestStatus.PAYMENT_REQUIRED.getStatus(), RestStatus.METHOD_NOT_ALLOWED.getStatus(), RestStatus.PROXY_AUTHENTICATION.getStatus(), RestStatus.LOCKED.getStatus(), RestStatus.TOO_MANY_REQUESTS.getStatus()));
    private static final Set<Integer> INVALID_INPUT_ERRORS = new HashSet<Integer>(Arrays.asList(RestStatus.REQUEST_ENTITY_TOO_LARGE.getStatus(), RestStatus.REQUEST_URI_TOO_LONG.getStatus(), RestStatus.REQUESTED_RANGE_NOT_SATISFIED.getStatus(), RestStatus.LENGTH_REQUIRED.getStatus(), RestStatus.PRECONDITION_FAILED.getStatus(), RestStatus.UNSUPPORTED_MEDIA_TYPE.getStatus(), RestStatus.CONFLICT.getStatus()));
    private static final Set<Integer> NOT_FOUND_ERRORS = new HashSet<Integer>(Arrays.asList(RestStatus.NOT_FOUND.getStatus(), RestStatus.GONE.getStatus()));
    private static final Set<Integer> TIMEOUT_ERROR = new HashSet<Integer>(Arrays.asList(RestStatus.REQUEST_TIMEOUT.getStatus()));
    private static final Set<Integer> POTENTIAL_DUPLICATES_ERRORS = Set.of(Integer.valueOf(RestStatus.INTERNAL_SERVER_ERROR.getStatus()), Integer.valueOf(RestStatus.GATEWAY_TIMEOUT.getStatus()));
    private static final Set<Class<? extends Exception>> SOCKET_TIMEOUT_EXCEPTIONS = Set.of(SocketTimeoutException.class, JsonParsingException.class);
    private final RequestFunction<AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest>, BulkResponse> requestFunction;
    private final BiConsumer<List<FailedBulkOperation>, Throwable> logFailure;
    private final Consumer<List<BulkOperationWrapper>> successfulOperationsHandler;
    private final PluginMetrics pluginMetrics;
    private final Supplier<AccumulatingBulkRequest> bulkRequestSupplier;
    private final int maxRetries;
    private final ObjectMapper objectMapper;
    private final Counter sentDocumentsCounter;
    private final Counter sentDocumentsOnFirstAttemptCounter;
    private final Counter documentErrorsCounter;
    private final Counter bulkRequestFailedCounter;
    private final Counter bulkRequestNumberOfRetries;
    private final Counter bulkRequestBadErrors;
    private final Counter bulkRequestNotAllowedErrors;
    private final Counter bulkRequestInvalidInputErrors;
    private final Counter bulkRequestNotFoundErrors;
    private final Counter bulkRequestTimeoutErrors;
    private final Counter bulkRequestServerErrors;
    private final Counter documentsVersionConflictErrors;
    private final Counter documentsDuplicates;
    private final ExistingDocumentQueryManager existingDocumentQueryManager;
    private static final Logger LOG = LoggerFactory.getLogger(BulkRetryStrategy.class);

    public BulkRetryStrategy(RequestFunction<AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest>, BulkResponse> requestFunction, BiConsumer<List<FailedBulkOperation>, Throwable> logFailure, Consumer<List<BulkOperationWrapper>> successfulOperationsHandler, PluginMetrics pluginMetrics, int maxRetries, Supplier<AccumulatingBulkRequest> bulkRequestSupplier, String pipelineName, String pluginName, ExistingDocumentQueryManager existingDocumentQueryManager) {
        this.existingDocumentQueryManager = existingDocumentQueryManager;
        this.requestFunction = requestFunction;
        this.logFailure = logFailure;
        this.successfulOperationsHandler = successfulOperationsHandler;
        this.pluginMetrics = pluginMetrics;
        this.bulkRequestSupplier = bulkRequestSupplier;
        this.maxRetries = maxRetries;
        this.objectMapper = new ObjectMapper();
        this.sentDocumentsCounter = pluginMetrics.counter(DOCUMENTS_SUCCESS);
        this.sentDocumentsOnFirstAttemptCounter = pluginMetrics.counter(DOCUMENTS_SUCCESS_FIRST_ATTEMPT);
        this.documentErrorsCounter = pluginMetrics.counter(DOCUMENT_ERRORS);
        this.bulkRequestFailedCounter = pluginMetrics.counter(BULK_REQUEST_FAILED);
        this.bulkRequestNumberOfRetries = pluginMetrics.counter(BULK_REQUEST_NUMBER_OF_RETRIES);
        this.bulkRequestBadErrors = pluginMetrics.counter(BULK_BAD_REQUEST_ERRORS);
        this.bulkRequestNotAllowedErrors = pluginMetrics.counter(BULK_REQUEST_NOT_ALLOWED_ERRORS);
        this.bulkRequestInvalidInputErrors = pluginMetrics.counter(BULK_REQUEST_INVALID_INPUT_ERRORS);
        this.bulkRequestNotFoundErrors = pluginMetrics.counter(BULK_REQUEST_NOT_FOUND_ERRORS);
        this.bulkRequestTimeoutErrors = pluginMetrics.counter(BULK_REQUEST_TIMEOUT_ERRORS);
        this.bulkRequestServerErrors = pluginMetrics.counter(BULK_REQUEST_SERVER_ERRORS);
        this.documentsVersionConflictErrors = pluginMetrics.counter(DOCUMENTS_VERSION_CONFLICT_ERRORS);
        this.documentsDuplicates = pluginMetrics.counter(DOCUMENTS_DUPLICATES);
    }

    private void incrementErrorCounters(Exception e) {
        if (e instanceof OpenSearchException) {
            int status = ((OpenSearchException)((Object)e)).status();
            if (NOT_ALLOWED_ERRORS.contains(status)) {
                this.bulkRequestNotAllowedErrors.increment();
            } else if (INVALID_INPUT_ERRORS.contains(status)) {
                this.bulkRequestInvalidInputErrors.increment();
            } else if (NOT_FOUND_ERRORS.contains(status)) {
                this.bulkRequestNotFoundErrors.increment();
            } else if (status == RestStatus.REQUEST_TIMEOUT.getStatus()) {
                this.bulkRequestTimeoutErrors.increment();
            } else if (status >= RestStatus.INTERNAL_SERVER_ERROR.getStatus()) {
                this.bulkRequestServerErrors.increment();
            } else {
                this.bulkRequestBadErrors.increment();
            }
        }
    }

    public void execute(AccumulatingBulkRequest bulkRequest) throws InterruptedException {
        BulkOperationRequestResponse operationResponse;
        Backoff backoff = Backoff.exponential((long)50L, (long)MAXIMUM_DELAY_MS).withMaxAttempts(this.maxRetries);
        BulkResponse response = null;
        Exception exception = null;
        AccumulatingBulkRequest request = bulkRequest;
        int attempt = 1;
        do {
            if ((operationResponse = this.handleRetry(request, response, attempt, exception)) == null) continue;
            long delayMillis = backoff.nextDelayMillis(attempt++);
            String exceptionMessage = "";
            request = operationResponse.getBulkRequest();
            response = operationResponse.getResponse();
            exceptionMessage = operationResponse.getExceptionMessage();
            exception = operationResponse.getException();
            if (delayMillis < 0L) {
                RuntimeException e = new RuntimeException(String.format("Number of retries reached the limit of max retries (configured value %d. Last exception message: %s)", this.maxRetries, exceptionMessage));
                this.handleFailures(request, null, e);
                break;
            }
            try {
                Thread.sleep(delayMillis);
            }
            catch (InterruptedException e) {
                LOG.error("Thread is interrupted while attempting to bulk write to OpenSearch with retry.", (Throwable)e);
            }
        } while (operationResponse != null);
    }

    public boolean canRetry(BulkResponse response) {
        for (BulkResponseItem bulkItemResponse : response.items()) {
            if (!BulkRetryStrategy.isItemInError(bulkItemResponse) || !this.canRetryItem(bulkItemResponse)) continue;
            return true;
        }
        return false;
    }

    public static boolean canRetry(Exception e) {
        return e instanceof IOException || e instanceof OpenSearchException && !NON_RETRY_STATUS.contains(((OpenSearchException)((Object)e)).status());
    }

    private boolean canRetryItem(BulkResponseItem bulkItemResponse) {
        return this.canRetryItem(bulkItemResponse, 1);
    }

    private boolean canRetryItem(BulkResponseItem bulkItemResponse, int attemptNumber) {
        if (this.isDeleteOperationWithNotFoundError(bulkItemResponse)) {
            return this.canRetryDeleteNotFoundOperation(bulkItemResponse, attemptNumber);
        }
        return this.isGenerallyRetryableOperation(bulkItemResponse);
    }

    private boolean isDeleteOperationWithNotFoundError(BulkResponseItem bulkItemResponse) {
        return bulkItemResponse.status() == RestStatus.NOT_FOUND.getStatus() && bulkItemResponse.operationType() == OperationType.Delete;
    }

    private boolean canRetryDeleteNotFoundOperation(BulkResponseItem bulkItemResponse, int attemptNumber) {
        if (attemptNumber > 3) {
            LOG.info("DELETE operation for index '{}' reached maximum retry limit ({}) for 404 errors, sending to DLQ", (Object)bulkItemResponse.index(), (Object)3);
            return false;
        }
        return true;
    }

    private boolean isGenerallyRetryableOperation(BulkResponseItem bulkItemResponse) {
        return !NON_RETRY_STATUS.contains(bulkItemResponse.status());
    }

    private BulkOperationRequestResponse handleRetriesAndFailures(AccumulatingBulkRequest bulkRequestForRetry, int attemptNumber, BulkResponse bulkResponse, Exception exceptionFromRequest) {
        boolean doRetry;
        boolean bl = doRetry = Objects.isNull(exceptionFromRequest) ? this.canRetry(bulkResponse) : BulkRetryStrategy.canRetry(exceptionFromRequest);
        if (!Objects.isNull(bulkResponse) && attemptNumber == 1) {
            for (BulkResponseItem bulkItemResponse : bulkResponse.items()) {
                if (BulkRetryStrategy.isItemInError(bulkItemResponse)) continue;
                this.sentDocumentsOnFirstAttemptCounter.increment();
            }
        }
        if (doRetry) {
            if (attemptNumber % 5 == 1) {
                LOG.warn("Bulk Operation Failed. Number of retries {}. Retrying... ", (Object)(attemptNumber - 1), (Object)exceptionFromRequest);
                if (exceptionFromRequest == null) {
                    for (BulkResponseItem bulkItemResponse : bulkResponse.items()) {
                        if (!BulkRetryStrategy.isItemInError(bulkItemResponse)) continue;
                        ErrorCause error = bulkItemResponse.error();
                        LOG.warn("index = {} operation = {}, error = {}", new Object[]{bulkItemResponse.index(), bulkItemResponse.operationType(), error != null ? error.reason() : ""});
                    }
                }
            }
            this.bulkRequestNumberOfRetries.increment();
            return new BulkOperationRequestResponse(bulkRequestForRetry, bulkResponse, exceptionFromRequest);
        }
        this.handleFailures(bulkRequestForRetry, bulkResponse, exceptionFromRequest);
        return null;
    }

    private void handleFailures(AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest> bulkRequest, BulkResponse bulkResponse, Throwable failure) {
        if (failure == null) {
            for (BulkResponseItem bulkItemResponse : bulkResponse.items()) {
                ErrorCause error;
                if (!BulkRetryStrategy.isItemInError(bulkItemResponse) || (error = bulkItemResponse.error()) != null && VERSION_CONFLICT_EXCEPTION_TYPE.equals(error.type())) continue;
                LOG.warn("index = {}, operation = {}, status = {}, error = {}", new Object[]{bulkItemResponse.index(), bulkItemResponse.operationType(), bulkItemResponse.status(), error != null ? error.reason() : ""});
            }
            this.handleFailures(bulkRequest, bulkResponse.items());
        } else {
            LOG.warn("Bulk Operation Failed.", failure);
            this.handleFailures(bulkRequest, failure);
        }
        this.bulkRequestFailedCounter.increment();
    }

    private BulkOperationRequestResponse handleRetry(AccumulatingBulkRequest request, BulkResponse response, int attemptNumber, Exception previousException) throws InterruptedException {
        boolean firstAttempt;
        BulkResponse bulkResponse;
        AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest> bulkRequestForRetry = this.createBulkRequestForRetry(request, response, previousException, attemptNumber);
        if (bulkRequestForRetry.getOperationsCount() == 0) {
            return null;
        }
        try {
            bulkResponse = this.requestFunction.apply(bulkRequestForRetry);
        }
        catch (Exception e) {
            this.incrementErrorCounters(e);
            return this.handleRetriesAndFailures(bulkRequestForRetry, attemptNumber, null, e);
        }
        if (bulkResponse.errors()) {
            return this.handleRetriesAndFailures(bulkRequestForRetry, attemptNumber, bulkResponse, null);
        }
        int numberOfDocs = bulkRequestForRetry.getOperationsCount();
        boolean bl = firstAttempt = attemptNumber == 1;
        if (firstAttempt) {
            this.sentDocumentsOnFirstAttemptCounter.increment((double)numberOfDocs);
        }
        this.sentDocumentsCounter.increment((double)bulkRequestForRetry.getOperationsCount());
        ArrayList<BulkOperationWrapper> successfulOperations = new ArrayList<BulkOperationWrapper>(bulkRequestForRetry.getOperations());
        this.successfulOperationsHandler.accept(successfulOperations);
        int totalDuplicateDocuments = bulkResponse.items().stream().filter(this::isDuplicateDocument).mapToInt(i -> 1).sum();
        this.documentsDuplicates.increment((double)totalDuplicateDocuments);
        return null;
    }

    private AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest> createBulkRequestForRetry(AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest> request, BulkResponse response, Exception previousException, int attemptNumber) {
        if (this.shouldSendAllForQuerying(previousException)) {
            for (BulkOperationWrapper bulkOperationWrapper : request.getOperations()) {
                this.existingDocumentQueryManager.addBulkOperation(bulkOperationWrapper);
            }
            return this.bulkRequestSupplier.get();
        }
        if (response == null) {
            return request;
        }
        AccumulatingBulkRequest requestToReissue = this.bulkRequestSupplier.get();
        ImmutableList.Builder nonRetryableFailures = ImmutableList.builder();
        int index = 0;
        ArrayList<BulkOperationWrapper> successfulOperations = new ArrayList<BulkOperationWrapper>(response.items().size());
        for (BulkResponseItem bulkItemResponse : response.items()) {
            BulkOperationWrapper bulkOperation = request.getOperationAt(index);
            if (BulkRetryStrategy.isItemInError(bulkItemResponse)) {
                if (this.existingDocumentQueryManager != null && POTENTIAL_DUPLICATES_ERRORS.contains(bulkItemResponse.status())) {
                    this.existingDocumentQueryManager.addBulkOperation(bulkOperation);
                    ++index;
                    continue;
                }
                if (this.canRetryItem(bulkItemResponse, attemptNumber)) {
                    requestToReissue.addOperation(bulkOperation);
                } else if (bulkItemResponse.error() != null && VERSION_CONFLICT_EXCEPTION_TYPE.equals(bulkItemResponse.error().type())) {
                    this.documentsVersionConflictErrors.increment();
                    LOG.debug("Index: {}, Received version conflict from OpenSearch: {}", (Object)bulkItemResponse.index(), (Object)bulkItemResponse.error().reason());
                    bulkOperation.releaseEventHandle(true);
                } else {
                    nonRetryableFailures.add((Object)FailedBulkOperation.builder().withBulkOperation(bulkOperation).withBulkResponseItem(bulkItemResponse).build());
                    this.documentErrorsCounter.increment();
                    this.getDocumentStatusCounter(bulkItemResponse.status()).increment();
                }
            } else {
                this.sentDocumentsCounter.increment();
                if (this.isDuplicateDocument(bulkItemResponse)) {
                    this.documentsDuplicates.increment();
                }
                successfulOperations.add(bulkOperation);
            }
            ++index;
        }
        this.successfulOperationsHandler.accept(successfulOperations);
        ImmutableList failedBulkOperations = nonRetryableFailures.build();
        if (!failedBulkOperations.isEmpty()) {
            this.logFailure.accept((List<FailedBulkOperation>)failedBulkOperations, null);
        }
        return requestToReissue;
    }

    private void handleFailures(AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest> accumulatingBulkRequest, List<BulkResponseItem> itemResponses) {
        assert (accumulatingBulkRequest.getOperationsCount() == itemResponses.size());
        ImmutableList.Builder failures = ImmutableList.builder();
        ArrayList<BulkOperationWrapper> successfulOperations = new ArrayList<BulkOperationWrapper>(itemResponses.size());
        for (int i = 0; i < itemResponses.size(); ++i) {
            BulkResponseItem bulkItemResponse = itemResponses.get(i);
            BulkOperationWrapper bulkOperation = accumulatingBulkRequest.getOperationAt(i);
            if (BulkRetryStrategy.isItemInError(bulkItemResponse)) {
                if (bulkItemResponse.error() != null && VERSION_CONFLICT_EXCEPTION_TYPE.equals(bulkItemResponse.error().type())) {
                    this.documentsVersionConflictErrors.increment();
                    LOG.debug("Index: {}, Received version conflict from OpenSearch: {}", (Object)bulkOperation.getIndex(), (Object)bulkItemResponse.error().reason());
                    bulkOperation.releaseEventHandle(true);
                } else {
                    failures.add((Object)FailedBulkOperation.builder().withBulkOperation(bulkOperation).withBulkResponseItem(bulkItemResponse).build());
                }
                this.documentErrorsCounter.increment();
                this.getDocumentStatusCounter(bulkItemResponse.status()).increment();
                continue;
            }
            this.sentDocumentsCounter.increment();
            if (this.isDuplicateDocument(bulkItemResponse)) {
                this.documentsDuplicates.increment();
            }
            successfulOperations.add(bulkOperation);
        }
        this.successfulOperationsHandler.accept(successfulOperations);
        this.logFailure.accept((List<FailedBulkOperation>)failures.build(), null);
    }

    private void handleFailures(AccumulatingBulkRequest<BulkOperationWrapper, BulkRequest> accumulatingBulkRequest, Throwable failure) {
        this.documentErrorsCounter.increment((double)accumulatingBulkRequest.getOperationsCount());
        ImmutableList.Builder failures = ImmutableList.builder();
        List<BulkOperationWrapper> bulkOperations = accumulatingBulkRequest.getOperations();
        for (int i = 0; i < bulkOperations.size(); ++i) {
            BulkOperationWrapper bulkOperation = bulkOperations.get(i);
            failures.add((Object)FailedBulkOperation.builder().withBulkOperation(bulkOperation).withFailure(failure).build());
        }
        this.logFailure.accept((List<FailedBulkOperation>)failures.build(), failure);
    }

    private static boolean isItemInError(BulkResponseItem item) {
        return item.status() >= 300 || item.error() != null;
    }

    private boolean isDuplicateDocument(BulkResponseItem item) {
        return item.seqNo() != null && item.seqNo() > 0L;
    }

    private Counter getDocumentStatusCounter(int status) {
        return this.pluginMetrics.counterWithTags(DOCUMENT_STATUSES, new String[]{"status", Integer.toString(status)});
    }

    private boolean shouldSendAllForQuerying(Exception exception) {
        if (exception != null && this.existingDocumentQueryManager != null) {
            LOG.warn("Received exception that may result in querying for duplicate documents", (Throwable)exception);
            if (SOCKET_TIMEOUT_EXCEPTIONS.contains(exception.getClass())) {
                return true;
            }
            return exception instanceof OpenSearchException && POTENTIAL_DUPLICATES_ERRORS.contains(((OpenSearchException)((Object)exception)).status());
        }
        return false;
    }

    static class BulkOperationRequestResponse {
        final AccumulatingBulkRequest bulkRequest;
        final BulkResponse response;
        final Exception exception;

        public BulkOperationRequestResponse(AccumulatingBulkRequest bulkRequest, BulkResponse response, Exception exception) {
            this.bulkRequest = bulkRequest;
            this.response = response;
            this.exception = exception;
        }

        AccumulatingBulkRequest getBulkRequest() {
            return this.bulkRequest;
        }

        BulkResponse getResponse() {
            return this.response;
        }

        Exception getException() {
            return this.exception;
        }

        String getExceptionMessage() {
            return this.exception != null ? this.exception.getMessage() : "-";
        }
    }
}

