/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.source_crawler.coordination.scheduler;

import com.google.common.annotations.VisibleForTesting;
import io.micrometer.core.instrument.Counter;
import java.time.Duration;
import java.time.Instant;
import java.util.Optional;
import org.opensearch.dataprepper.logging.DataPrepperMarkers;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.model.buffer.Buffer;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourcePartition;
import org.opensearch.dataprepper.plugins.source.source_crawler.base.Crawler;
import org.opensearch.dataprepper.plugins.source.source_crawler.base.CrawlerSourceConfig;
import org.opensearch.dataprepper.plugins.source.source_crawler.base.SaasWorkerProgressState;
import org.opensearch.dataprepper.plugins.source.source_crawler.coordination.partition.SaasSourcePartition;
import org.opensearch.dataprepper.plugins.source.source_crawler.coordination.state.DimensionalTimeSliceWorkerProgressState;
import org.opensearch.dataprepper.plugins.source.source_crawler.exception.SaaSCrawlerException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class WorkerScheduler
implements Runnable {
    public static final String ACKNOWLEDGEMENT_SET_SUCCESS_METRIC_NAME = "acknowledgementSetSuccesses";
    public static final String ACKNOWLEDGEMENT_SET_FAILURES_METRIC_NAME = "acknowledgementSetFailures";
    public static final String WORKER_PARTITIONS_FAILED = "workerPartitionsFailed";
    public static final String WORKER_PARTITIONS_COMPLETED = "workerPartitionsCompleted";
    private static final Duration ACKNOWLEDGEMENT_SET_TIMEOUT = Duration.ofHours(2L);
    private static final Logger log = LoggerFactory.getLogger(WorkerScheduler.class);
    private static final int RETRY_BACKOFF_ON_EXCEPTION_MILLIS = 5000;
    private static final Duration DEFAULT_SLEEP_DURATION_MILLIS = Duration.ofMillis(10000L);
    private final Counter parititionsCompletedCounter;
    private final Counter parititionsFailedCounter;
    private final EnhancedSourceCoordinator sourceCoordinator;
    private final CrawlerSourceConfig sourceConfig;
    private final Crawler crawler;
    private final Buffer<Record<Event>> buffer;
    private final PluginMetrics pluginMetrics;
    private final AcknowledgementSetManager acknowledgementSetManager;
    private final Counter acknowledgementSetSuccesses;
    private final Counter acknowledgementSetFailures;
    private final String sourcePluginName;

    public WorkerScheduler(String sourcePluginName, Buffer<Record<Event>> buffer, EnhancedSourceCoordinator sourceCoordinator, CrawlerSourceConfig sourceConfig, Crawler crawler, PluginMetrics pluginMetrics, AcknowledgementSetManager acknowledgementSetManager) {
        this.sourceCoordinator = sourceCoordinator;
        this.sourceConfig = sourceConfig;
        this.crawler = crawler;
        this.buffer = buffer;
        this.sourcePluginName = sourcePluginName;
        this.acknowledgementSetManager = acknowledgementSetManager;
        this.pluginMetrics = pluginMetrics;
        this.acknowledgementSetSuccesses = pluginMetrics.counter(ACKNOWLEDGEMENT_SET_SUCCESS_METRIC_NAME);
        this.acknowledgementSetFailures = pluginMetrics.counter(ACKNOWLEDGEMENT_SET_FAILURES_METRIC_NAME);
        this.parititionsCompletedCounter = pluginMetrics.counter(WORKER_PARTITIONS_COMPLETED);
        this.parititionsFailedCounter = pluginMetrics.counter(WORKER_PARTITIONS_FAILED);
    }

    @Override
    public void run() {
        log.info("Worker thread started");
        log.info("Processing Partitions");
        while (!Thread.currentThread().isInterrupted()) {
            Optional partition = Optional.empty();
            try {
                partition = this.sourceCoordinator.acquireAvailablePartition("SAAS-WORKER");
                if (partition.isPresent()) {
                    this.processPartition((EnhancedSourcePartition)partition.get(), this.buffer);
                    this.parititionsCompletedCounter.increment();
                    continue;
                }
                log.debug("No partition available. This thread will sleep for {}", (Object)DEFAULT_SLEEP_DURATION_MILLIS);
                try {
                    Thread.sleep(DEFAULT_SLEEP_DURATION_MILLIS.toMillis());
                }
                catch (InterruptedException e) {
                    log.info("InterruptedException occurred");
                    break;
                }
            }
            catch (Exception e) {
                SaaSCrawlerException saasException;
                this.parititionsFailedCounter.increment();
                boolean shouldLocalRetry = true;
                if (e instanceof SaaSCrawlerException && !(saasException = (SaaSCrawlerException)e).isRetryable()) {
                    shouldLocalRetry = this.delayWorkerPartitionRetry(partition, e);
                }
                if (!shouldLocalRetry) continue;
                this.backoffRetry(e);
            }
        }
        log.warn("SourceItemWorker Scheduler is interrupted, looks like shutdown has triggered");
    }

    private void backoffRetry(Exception e) {
        log.error("[Retryable Exception] Error processing partition", (Throwable)e);
        try {
            Thread.sleep(5000L);
        }
        catch (InterruptedException ex) {
            log.warn("Thread interrupted while waiting to retry due to {}", (Object)ex.getMessage());
        }
    }

    private boolean delayWorkerPartitionRetry(Optional<EnhancedSourcePartition> sourcePartition, Exception ex) {
        log.error("[Non-Retryable Exception] Error processing worker partition. Will delay retry with the configured duration", (Throwable)ex);
        try {
            SaasWorkerProgressState progressState;
            SaasSourcePartition workerPartition = (SaasSourcePartition)sourcePartition.get();
            boolean shouldLocalRetry = true;
            if (workerPartition != null && (progressState = workerPartition.getProgressState().get()) instanceof DimensionalTimeSliceWorkerProgressState) {
                DimensionalTimeSliceWorkerProgressState workerProgressState = (DimensionalTimeSliceWorkerProgressState)progressState;
                this.updateWorkerPartition(workerProgressState.getPartitionCreationTime(), workerPartition);
                shouldLocalRetry = false;
            }
            return shouldLocalRetry;
        }
        catch (Exception e) {
            log.error("Error updating workerPartition ", (Throwable)e);
            return false;
        }
    }

    private void updateWorkerPartition(Instant partitionCreationTime, SaasSourcePartition workerPartition) {
        log.info("Updating workerPartition {}", (Object)workerPartition.getPartitionKey());
        Duration age = Duration.between(partitionCreationTime, Instant.now());
        if (age.compareTo(this.sourceConfig.getDurationToGiveUpRetry()) <= 0) {
            log.info(DataPrepperMarkers.NOISY, "Partition {} is within or equal to the configured max duration, scheduling retry", (Object)workerPartition.getPartitionKey());
            this.sourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)workerPartition, this.sourceConfig.getDurationToDelayRetry());
        } else {
            log.info("Partition {} is older than the configured max duration, giving up", (Object)workerPartition.getPartitionKey());
            this.sourceCoordinator.giveUpPartition((EnhancedSourcePartition)workerPartition);
        }
    }

    private void processPartition(EnhancedSourcePartition partition, Buffer<Record<Event>> buffer) {
        if (partition.getProgressState().isPresent()) {
            AcknowledgementSet acknowledgementSet = null;
            if (this.sourceConfig.isAcknowledgments()) {
                acknowledgementSet = this.createAcknowledgementSet(partition);
                this.crawler.executePartition((SaasWorkerProgressState)partition.getProgressState().get(), buffer, acknowledgementSet);
            } else {
                this.crawler.executePartition((SaasWorkerProgressState)partition.getProgressState().get(), buffer, acknowledgementSet);
                this.sourceCoordinator.completePartition(partition);
            }
        } else {
            this.sourceCoordinator.completePartition(partition);
        }
    }

    @VisibleForTesting
    AcknowledgementSet createAcknowledgementSet(EnhancedSourcePartition partition) {
        return this.acknowledgementSetManager.create(result -> {
            if (result.booleanValue()) {
                this.acknowledgementSetSuccesses.increment();
                this.sourceCoordinator.completePartition(partition);
                log.debug("acknowledgements received for partitionKey: {}", (Object)partition.getPartitionKey());
            } else {
                this.acknowledgementSetFailures.increment();
                log.debug("acknowledgements received with false for partitionKey: {}", (Object)partition.getPartitionKey());
                this.sourceCoordinator.giveUpPartition(partition);
            }
        }, ACKNOWLEDGEMENT_SET_TIMEOUT);
    }
}

