/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.source_crawler.base;

import com.google.common.annotations.VisibleForTesting;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Timer;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import javax.inject.Named;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.model.buffer.Buffer;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourcePartition;
import org.opensearch.dataprepper.plugins.source.source_crawler.base.Crawler;
import org.opensearch.dataprepper.plugins.source.source_crawler.base.CrawlerClient;
import org.opensearch.dataprepper.plugins.source.source_crawler.base.LeaderOnlyTokenCrawlerClient;
import org.opensearch.dataprepper.plugins.source.source_crawler.coordination.partition.LeaderPartition;
import org.opensearch.dataprepper.plugins.source.source_crawler.coordination.partition.SaasSourcePartition;
import org.opensearch.dataprepper.plugins.source.source_crawler.coordination.state.PaginationCrawlerWorkerProgressState;
import org.opensearch.dataprepper.plugins.source.source_crawler.coordination.state.TokenPaginationCrawlerLeaderProgressState;
import org.opensearch.dataprepper.plugins.source.source_crawler.model.ItemInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Named
public class LeaderOnlyTokenCrawler
implements Crawler<PaginationCrawlerWorkerProgressState> {
    private static final Logger log = LoggerFactory.getLogger(LeaderOnlyTokenCrawler.class);
    private static final Duration NO_ACK_TIME_OUT_SECONDS = Duration.ofSeconds(900L);
    private static final Duration CHECKPOINT_INTERVAL = Duration.ofMinutes(1L);
    private static final Duration DEFAULT_LEASE_DURATION = Duration.ofMinutes(15L);
    private static final int BATCH_SIZE = 1000;
    private static final String METRIC_BATCHES_FAILED = "batchesFailed";
    private static final String METRIC_BUFFER_WRITE_TIME = "bufferWriteTime";
    private static final String WORKER_PARTITION_WAIT_TIME = "workerPartitionWaitTime";
    private static final String WORKER_PARTITION_PROCESS_LATENCY = "workerPartitionProcessLatency";
    public static final String ACKNOWLEDGEMENT_SET_SUCCESS_METRIC_NAME = "acknowledgementSetSuccesses";
    public static final String ACKNOWLEDGEMENT_SET_FAILURES_METRIC_NAME = "acknowledgementSetFailures";
    private final CrawlerClient client;
    private final Timer crawlingTimer;
    private final PluginMetrics pluginMetrics;
    private boolean acknowledgementsEnabled;
    private AcknowledgementSetManager acknowledgementSetManager;
    private Buffer<Record<Event>> buffer;
    private final Counter batchesFailedCounter;
    private final Counter acknowledgementSetSuccesses;
    private final Counter acknowledgementSetFailures;
    private final Timer bufferWriteTimer;
    private final Timer partitionWaitTimeTimer;
    private final Timer partitionProcessLatencyTimer;
    private String lastToken;
    private Duration noAckTimeout;

    public LeaderOnlyTokenCrawler(CrawlerClient client, PluginMetrics pluginMetrics) {
        this.client = client;
        this.pluginMetrics = pluginMetrics;
        this.crawlingTimer = pluginMetrics.timer("crawlingTime");
        this.batchesFailedCounter = pluginMetrics.counter(METRIC_BATCHES_FAILED);
        this.bufferWriteTimer = pluginMetrics.timer(METRIC_BUFFER_WRITE_TIME);
        this.partitionWaitTimeTimer = pluginMetrics.timer(WORKER_PARTITION_WAIT_TIME);
        this.partitionProcessLatencyTimer = pluginMetrics.timer(WORKER_PARTITION_PROCESS_LATENCY);
        this.acknowledgementSetSuccesses = pluginMetrics.counter(ACKNOWLEDGEMENT_SET_SUCCESS_METRIC_NAME);
        this.acknowledgementSetFailures = pluginMetrics.counter(ACKNOWLEDGEMENT_SET_FAILURES_METRIC_NAME);
        this.noAckTimeout = NO_ACK_TIME_OUT_SECONDS;
    }

    @Override
    public Instant crawl(LeaderPartition leaderPartition, EnhancedSourceCoordinator coordinator) {
        long startTime = System.currentTimeMillis();
        Instant lastCheckpointTime = Instant.now();
        TokenPaginationCrawlerLeaderProgressState leaderProgressState = (TokenPaginationCrawlerLeaderProgressState)leaderPartition.getProgressState().get();
        this.lastToken = leaderProgressState.getLastToken();
        log.info("Starting leader-only crawl with token: {}", (Object)this.lastToken);
        Iterator<ItemInfo> itemIterator = ((LeaderOnlyTokenCrawlerClient)this.client).listItems(this.lastToken);
        while (itemIterator.hasNext()) {
            List<ItemInfo> batch = this.collectBatch(itemIterator);
            if (batch.isEmpty()) continue;
            ItemInfo lastItem = batch.get(batch.size() - 1);
            this.lastToken = lastItem.getItemId();
            try {
                this.processBatch(batch, leaderPartition, coordinator);
            }
            catch (Exception e) {
                this.batchesFailedCounter.increment();
                log.error("Failed to process batch ending with token {}", (Object)this.lastToken, (Object)e);
                throw e;
            }
            if (this.acknowledgementsEnabled || Duration.between(lastCheckpointTime, Instant.now()).compareTo(CHECKPOINT_INTERVAL) < 0) continue;
            this.updateLeaderProgressState(leaderPartition, this.lastToken, coordinator);
            lastCheckpointTime = Instant.now();
        }
        if (!this.acknowledgementsEnabled) {
            this.updateLeaderProgressState(leaderPartition, this.lastToken, coordinator);
        }
        long crawlTimeMillis = System.currentTimeMillis() - startTime;
        log.debug("Crawling completed in {} ms", (Object)crawlTimeMillis);
        this.crawlingTimer.record(crawlTimeMillis, TimeUnit.MILLISECONDS);
        return Instant.now();
    }

    @Override
    public void executePartition(PaginationCrawlerWorkerProgressState state, Buffer buffer, AcknowledgementSet acknowledgementSet) {
        this.partitionWaitTimeTimer.record(Duration.between(state.getExportStartTime(), Instant.now()));
        this.partitionProcessLatencyTimer.record(() -> this.client.executePartition(state, (Buffer<Record<Event>>)buffer, acknowledgementSet));
    }

    private List<ItemInfo> collectBatch(Iterator<ItemInfo> iterator) {
        ArrayList<ItemInfo> batch = new ArrayList<ItemInfo>();
        for (int i = 0; i < 1000 && iterator.hasNext(); ++i) {
            ItemInfo item = iterator.next();
            if (item == null) continue;
            batch.add(item);
        }
        return batch;
    }

    private void processBatch(List<ItemInfo> batch, LeaderPartition leaderPartition, EnhancedSourceCoordinator coordinator) {
        if (this.acknowledgementsEnabled) {
            AtomicBoolean ackReceived = new AtomicBoolean(false);
            long createTimestamp = System.currentTimeMillis();
            AcknowledgementSet acknowledgementSet = this.acknowledgementSetManager.create(success -> {
                ackReceived.set(true);
                if (success.booleanValue()) {
                    this.acknowledgementSetSuccesses.increment();
                } else {
                    this.acknowledgementSetFailures.increment();
                    log.warn("Batch processing received negative acknowledgment for token: {}. Creating retry partition.", (Object)this.lastToken);
                    this.createRetryPartition(batch, coordinator);
                }
            }, this.noAckTimeout);
            this.bufferWriteTimer.record(() -> {
                try {
                    ((LeaderOnlyTokenCrawlerClient)this.client).writeBatchToBuffer(batch, this.buffer, acknowledgementSet);
                    acknowledgementSet.complete();
                    while (!ackReceived.get()) {
                        Thread.sleep(Duration.ofSeconds(15L).toMillis());
                        Duration ackWaitDuration = Duration.between(Instant.ofEpochMilli(createTimestamp), Instant.now());
                        if (ackWaitDuration.minus(this.noAckTimeout).isNegative()) continue;
                        log.warn("No acknowledgment received for batch with token: {}. Creating retry partition.", (Object)this.lastToken);
                        this.createRetryPartition(batch, coordinator);
                        break;
                    }
                    this.updateLeaderProgressState(leaderPartition, this.lastToken, coordinator);
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException("Interrupted while waiting for acknowledgment", e);
                }
                catch (Exception e) {
                    log.error("Failed to process batch ending with token {}", (Object)this.lastToken, (Object)e);
                    acknowledgementSet.complete();
                    this.createRetryPartition(batch, coordinator);
                    throw e;
                }
            });
        } else {
            this.bufferWriteTimer.record(() -> {
                try {
                    ((LeaderOnlyTokenCrawlerClient)this.client).writeBatchToBuffer(batch, this.buffer, null);
                    this.updateLeaderProgressState(leaderPartition, this.lastToken, coordinator);
                }
                catch (Exception e) {
                    log.error("Failed to write batch to buffer", (Throwable)e);
                    throw e;
                }
            });
        }
    }

    private void createRetryPartition(List<ItemInfo> itemInfoList, EnhancedSourceCoordinator coordinator) {
        if (itemInfoList.isEmpty()) {
            return;
        }
        ItemInfo itemInfo = itemInfoList.get(0);
        String partitionKey = itemInfo.getPartitionKey();
        List<String> itemIds = itemInfoList.stream().map(ItemInfo::getId).collect(Collectors.toList());
        PaginationCrawlerWorkerProgressState state = new PaginationCrawlerWorkerProgressState();
        state.setKeyAttributes(itemInfo.getKeyAttributes());
        state.setItemIds(itemIds);
        state.setExportStartTime(Instant.now());
        state.setLoadedItems(itemInfoList.size());
        SaasSourcePartition sourcePartition = new SaasSourcePartition(state, partitionKey);
        coordinator.createPartition((EnhancedSourcePartition)sourcePartition);
    }

    private void updateLeaderProgressState(LeaderPartition leaderPartition, String updatedToken, EnhancedSourceCoordinator coordinator) {
        TokenPaginationCrawlerLeaderProgressState leaderProgressState = (TokenPaginationCrawlerLeaderProgressState)leaderPartition.getProgressState().get();
        String oldToken = leaderProgressState.getLastToken();
        leaderProgressState.setLastToken(updatedToken);
        leaderPartition.setLeaderProgressState(leaderProgressState);
        coordinator.saveProgressStateForPartition((EnhancedSourcePartition)leaderPartition, DEFAULT_LEASE_DURATION);
        log.info("Updated leader progress state: old lastToken={}, new lastToken={}", (Object)oldToken, (Object)updatedToken);
    }

    @VisibleForTesting
    void setNoAckTimeout(Duration timeout) {
        this.noAckTimeout = timeout;
    }

    public void setAcknowledgementsEnabled(boolean acknowledgementsEnabled) {
        this.acknowledgementsEnabled = acknowledgementsEnabled;
    }

    public void setAcknowledgementSetManager(AcknowledgementSetManager acknowledgementSetManager) {
        this.acknowledgementSetManager = acknowledgementSetManager;
    }

    public void setBuffer(Buffer<Record<Event>> buffer) {
        this.buffer = buffer;
    }
}

