/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.s3;

import com.linecorp.armeria.client.retry.Backoff;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Timer;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.plugins.s3.common.source.S3ObjectReference;
import org.opensearch.dataprepper.plugins.source.s3.S3Service;
import org.opensearch.dataprepper.plugins.source.s3.S3SourceConfig;
import org.opensearch.dataprepper.plugins.source.s3.configuration.NotificationSourceOption;
import org.opensearch.dataprepper.plugins.source.s3.configuration.OnErrorOption;
import org.opensearch.dataprepper.plugins.source.s3.configuration.SqsOptions;
import org.opensearch.dataprepper.plugins.source.s3.exception.SqsRetriesExhaustedException;
import org.opensearch.dataprepper.plugins.source.s3.filter.EventBridgeObjectCreatedFilter;
import org.opensearch.dataprepper.plugins.source.s3.filter.S3EventFilter;
import org.opensearch.dataprepper.plugins.source.s3.filter.S3ObjectCreatedFilter;
import org.opensearch.dataprepper.plugins.source.s3.parser.ParsedMessage;
import org.opensearch.dataprepper.plugins.source.s3.parser.SqsMessageParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkException;
import software.amazon.awssdk.services.sqs.SqsClient;
import software.amazon.awssdk.services.sqs.model.ChangeMessageVisibilityRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequestEntry;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchResultEntry;
import software.amazon.awssdk.services.sqs.model.KmsAccessDeniedException;
import software.amazon.awssdk.services.sqs.model.KmsNotFoundException;
import software.amazon.awssdk.services.sqs.model.KmsThrottledException;
import software.amazon.awssdk.services.sqs.model.Message;
import software.amazon.awssdk.services.sqs.model.MessageSystemAttributeName;
import software.amazon.awssdk.services.sqs.model.QueueDoesNotExistException;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageRequest;
import software.amazon.awssdk.services.sqs.model.SqsException;
import software.amazon.awssdk.services.sts.model.StsException;

public class SqsWorker
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(SqsWorker.class);
    static final String SQS_MESSAGES_RECEIVED_METRIC_NAME = "sqsMessagesReceived";
    static final String SQS_RECEIVE_MESSAGES_FAILED_METRIC_NAME = "sqsReceiveMessageFailed";
    static final String SQS_MESSAGES_DELETED_METRIC_NAME = "sqsMessagesDeleted";
    static final String SQS_MESSAGES_FAILED_METRIC_NAME = "sqsMessagesFailed";
    static final String SQS_MESSAGES_DELETE_FAILED_METRIC_NAME = "sqsMessagesDeleteFailed";
    static final String S3_OBJECTS_EMPTY_METRIC_NAME = "s3ObjectsEmpty";
    static final String SQS_MESSAGE_DELAY_METRIC_NAME = "sqsMessageDelay";
    static final String SQS_VISIBILITY_TIMEOUT_CHANGED_COUNT_METRIC_NAME = "sqsVisibilityTimeoutChangedCount";
    static final String SQS_VISIBILITY_TIMEOUT_CHANGE_FAILED_COUNT_METRIC_NAME = "sqsVisibilityTimeoutChangeFailedCount";
    static final String ACKNOWLEDGEMENT_SET_CALLACK_METRIC_NAME = "acknowledgementSetCallbackCounter";
    static final String SQS_MESSAGE_ACCESS_DENIED_METRIC_NAME = "sqsMessagesAccessDenied";
    static final String SQS_MESSAGE_THROTTLED_METRIC_NAME = "sqsMessagesThrottled";
    static final String SQS_RESOURCE_NOT_FOUND_METRIC_NAME = "sqsResourceNotFound";
    private final S3SourceConfig s3SourceConfig;
    private final SqsClient sqsClient;
    private final S3Service s3Service;
    private final SqsOptions sqsOptions;
    private final S3EventFilter objectCreatedFilter;
    private final S3EventFilter evenBridgeObjectCreatedFilter;
    private final Counter sqsMessagesReceivedCounter;
    private final Counter sqsReceiveMessagesFailedCounter;
    private final Counter sqsMessagesDeletedCounter;
    private final Counter sqsMessagesFailedCounter;
    private final Counter s3ObjectsEmptyCounter;
    private final Counter sqsMessagesDeleteFailedCounter;
    private final Counter acknowledgementSetCallbackCounter;
    private final Counter sqsVisibilityTimeoutChangedCount;
    private final Counter sqsVisibilityTimeoutChangeFailedCount;
    private final Counter sqsMessageAccessDeniedCounter;
    private final Counter sqsMessageThrottledCounter;
    private final Counter sqsResourceNotFoundCounter;
    private final Timer sqsMessageDelayTimer;
    private final Backoff standardBackoff;
    private final SqsMessageParser sqsMessageParser;
    private int failedAttemptCount;
    private final boolean endToEndAcknowledgementsEnabled;
    private final AcknowledgementSetManager acknowledgementSetManager;
    private volatile boolean isStopped = false;
    private Map<ParsedMessage, Integer> parsedMessageVisibilityTimesMap;

    public SqsWorker(AcknowledgementSetManager acknowledgementSetManager, SqsClient sqsClient, S3Service s3Service, S3SourceConfig s3SourceConfig, PluginMetrics pluginMetrics, Backoff backoff) {
        this.sqsClient = sqsClient;
        this.s3Service = s3Service;
        this.s3SourceConfig = s3SourceConfig;
        this.acknowledgementSetManager = acknowledgementSetManager;
        this.standardBackoff = backoff;
        this.endToEndAcknowledgementsEnabled = s3SourceConfig.getAcknowledgements();
        this.sqsOptions = s3SourceConfig.getSqsOptions();
        this.objectCreatedFilter = new S3ObjectCreatedFilter();
        this.evenBridgeObjectCreatedFilter = new EventBridgeObjectCreatedFilter();
        this.sqsMessageParser = new SqsMessageParser(s3SourceConfig);
        this.failedAttemptCount = 0;
        this.parsedMessageVisibilityTimesMap = new HashMap<ParsedMessage, Integer>();
        this.sqsMessagesReceivedCounter = pluginMetrics.counter(SQS_MESSAGES_RECEIVED_METRIC_NAME);
        this.sqsMessagesDeletedCounter = pluginMetrics.counter(SQS_MESSAGES_DELETED_METRIC_NAME);
        this.sqsMessagesFailedCounter = pluginMetrics.counter(SQS_MESSAGES_FAILED_METRIC_NAME);
        this.s3ObjectsEmptyCounter = pluginMetrics.counter(S3_OBJECTS_EMPTY_METRIC_NAME);
        this.sqsMessagesDeleteFailedCounter = pluginMetrics.counter(SQS_MESSAGES_DELETE_FAILED_METRIC_NAME);
        this.sqsMessageDelayTimer = pluginMetrics.timer(SQS_MESSAGE_DELAY_METRIC_NAME);
        this.sqsReceiveMessagesFailedCounter = pluginMetrics.counter(SQS_RECEIVE_MESSAGES_FAILED_METRIC_NAME);
        this.acknowledgementSetCallbackCounter = pluginMetrics.counter(ACKNOWLEDGEMENT_SET_CALLACK_METRIC_NAME);
        this.sqsVisibilityTimeoutChangedCount = pluginMetrics.counter(SQS_VISIBILITY_TIMEOUT_CHANGED_COUNT_METRIC_NAME);
        this.sqsVisibilityTimeoutChangeFailedCount = pluginMetrics.counter(SQS_VISIBILITY_TIMEOUT_CHANGE_FAILED_COUNT_METRIC_NAME);
        this.sqsMessageAccessDeniedCounter = pluginMetrics.counter(SQS_MESSAGE_ACCESS_DENIED_METRIC_NAME);
        this.sqsMessageThrottledCounter = pluginMetrics.counter(SQS_MESSAGE_THROTTLED_METRIC_NAME);
        this.sqsResourceNotFoundCounter = pluginMetrics.counter(SQS_RESOURCE_NOT_FOUND_METRIC_NAME);
    }

    @Override
    public void run() {
        while (!this.isStopped) {
            int messagesProcessed = 0;
            try {
                messagesProcessed = this.processSqsMessages();
            }
            catch (Exception e) {
                LOG.error("Unable to process SQS messages. Processing error due to: {}", (Object)e.getMessage());
                this.applyBackoff();
            }
            if (messagesProcessed <= 0 || this.s3SourceConfig.getSqsOptions().getPollDelay().toMillis() <= 0L) continue;
            try {
                Thread.sleep(this.s3SourceConfig.getSqsOptions().getPollDelay().toMillis());
            }
            catch (InterruptedException e) {
                LOG.error("Thread is interrupted while polling SQS.", (Throwable)e);
            }
        }
    }

    int processSqsMessages() {
        List<Message> sqsMessages = this.getMessagesFromSqs();
        if (!sqsMessages.isEmpty()) {
            this.sqsMessagesReceivedCounter.increment((double)sqsMessages.size());
            Collection<ParsedMessage> s3MessageEventNotificationRecords = this.sqsMessageParser.parseSqsMessages(sqsMessages);
            List<DeleteMessageBatchRequestEntry> deleteMessageBatchRequestEntries = this.processS3EventNotificationRecords(s3MessageEventNotificationRecords);
            if (!deleteMessageBatchRequestEntries.isEmpty()) {
                this.deleteSqsMessages(deleteMessageBatchRequestEntries);
            }
        }
        return sqsMessages.size();
    }

    private List<Message> getMessagesFromSqs() {
        try {
            ReceiveMessageRequest receiveMessageRequest = this.createReceiveMessageRequest();
            List messages = this.sqsClient.receiveMessage(receiveMessageRequest).messages();
            this.failedAttemptCount = 0;
            if (messages.isEmpty()) {
                this.sqsMessageDelayTimer.record(Duration.ZERO);
            }
            return messages;
        }
        catch (SqsException | StsException e) {
            LOG.error("Error reading from SQS: {}. Retrying with exponential backoff.", (Object)e.getMessage());
            this.recordSqsException((AwsServiceException)e);
            this.sqsReceiveMessagesFailedCounter.increment();
            this.applyBackoff();
            return Collections.emptyList();
        }
    }

    private void applyBackoff() {
        long delayMillis;
        if ((delayMillis = this.standardBackoff.nextDelayMillis(++this.failedAttemptCount)) < 0L) {
            Thread.currentThread().interrupt();
            throw new SqsRetriesExhaustedException("SQS retries exhausted. Make sure that SQS configuration is valid, SQS queue exists, and IAM role has required permissions.");
        }
        Duration delayDuration = Duration.ofMillis(delayMillis);
        LOG.info("Pausing SQS processing for {}.{} seconds due to an error in processing.", (Object)delayDuration.getSeconds(), (Object)delayDuration.toMillisPart());
        try {
            Thread.sleep(delayMillis);
        }
        catch (InterruptedException e) {
            LOG.error("Thread is interrupted while polling SQS with retry.", (Throwable)e);
        }
    }

    private ReceiveMessageRequest createReceiveMessageRequest() {
        return (ReceiveMessageRequest)ReceiveMessageRequest.builder().queueUrl(this.sqsOptions.getSqsUrl()).maxNumberOfMessages(Integer.valueOf(this.sqsOptions.getMaximumMessages())).visibilityTimeout(Integer.valueOf((int)this.sqsOptions.getVisibilityTimeout().getSeconds())).waitTimeSeconds(Integer.valueOf((int)this.sqsOptions.getWaitTime().getSeconds())).attributeNamesWithStrings(new String[]{MessageSystemAttributeName.APPROXIMATE_RECEIVE_COUNT.toString()}).build();
    }

    private List<DeleteMessageBatchRequestEntry> processS3EventNotificationRecords(Collection<ParsedMessage> s3EventNotificationRecords) {
        List waitingForAcknowledgements;
        ArrayList<DeleteMessageBatchRequestEntry> deleteMessageBatchRequestEntryCollection = new ArrayList<DeleteMessageBatchRequestEntry>();
        ArrayList<ParsedMessage> parsedMessagesToRead = new ArrayList<ParsedMessage>();
        HashMap<ParsedMessage, AcknowledgementSet> messageAcknowledgementSetMap = new HashMap<ParsedMessage, AcknowledgementSet>();
        HashMap messageWaitingForAcknowledgementsMap = new HashMap();
        HashMap messagesWaitingForS3ObjectDeletion = new HashMap();
        for (ParsedMessage parsedMessage : s3EventNotificationRecords) {
            if (parsedMessage.isFailedParsing()) {
                this.sqsMessagesFailedCounter.increment();
                if (!this.s3SourceConfig.getOnErrorOption().equals((Object)OnErrorOption.DELETE_MESSAGES)) continue;
                deleteMessageBatchRequestEntryCollection.add(this.buildDeleteMessageBatchRequestEntry(parsedMessage.getMessage()));
                continue;
            }
            if (parsedMessage.getObjectSize() == 0L) {
                this.s3ObjectsEmptyCounter.increment();
                LOG.debug("Received empty S3 object: {} in the SQS message. The S3 object is skipped and the SQS message will be deleted.", (Object)parsedMessage.getObjectKey());
                deleteMessageBatchRequestEntryCollection.add(this.buildDeleteMessageBatchRequestEntry(parsedMessage.getMessage()));
                continue;
            }
            if (this.s3SourceConfig.getNotificationSource().equals((Object)NotificationSourceOption.S3) && !parsedMessage.isEmptyNotification() && this.isS3EventNameCreated(parsedMessage)) {
                parsedMessagesToRead.add(parsedMessage);
                continue;
            }
            if (this.s3SourceConfig.getNotificationSource().equals((Object)NotificationSourceOption.EVENTBRIDGE) && this.isEventBridgeEventTypeCreated(parsedMessage)) {
                parsedMessagesToRead.add(parsedMessage);
                continue;
            }
            LOG.debug("Received SQS message other than s3:ObjectCreated:*. Deleting message from SQS queue.");
            deleteMessageBatchRequestEntryCollection.add(this.buildDeleteMessageBatchRequestEntry(parsedMessage.getMessage()));
        }
        LOG.info("Received {} messages from SQS. Processing {} messages.", (Object)s3EventNotificationRecords.size(), (Object)parsedMessagesToRead.size());
        for (ParsedMessage parsedMessage : parsedMessagesToRead) {
            int approximateReceiveCount = this.getApproximateReceiveCount(parsedMessage.getMessage());
            if (this.s3SourceConfig.getSqsOptions().getMaxReceiveAttempts() != null && approximateReceiveCount > this.s3SourceConfig.getSqsOptions().getMaxReceiveAttempts()) {
                this.deleteSqsMessages(List.of(this.buildDeleteMessageBatchRequestEntry(parsedMessage.getMessage())));
                parsedMessage.setShouldSkipProcessing(true);
                continue;
            }
            if (approximateReceiveCount <= 1) {
                this.sqsMessageDelayTimer.record(Duration.between(Instant.ofEpochMilli(parsedMessage.getEventTime().toInstant().getMillis()), Instant.now()));
            }
            waitingForAcknowledgements = new ArrayList();
            ArrayList s3ObjectDeletionWaitingForAcknowledgments = new ArrayList();
            AcknowledgementSet acknowledgementSet = null;
            int visibilityTimeout = (int)this.sqsOptions.getVisibilityTimeout().getSeconds();
            int maxVisibilityTimeout = (int)this.sqsOptions.getVisibilityDuplicateProtectionTimeout().getSeconds();
            int progressCheckInterval = visibilityTimeout / 2 - 1;
            if (!this.endToEndAcknowledgementsEnabled) continue;
            int expiryTimeout = visibilityTimeout - 2;
            boolean visibilityDuplicateProtectionEnabled = this.sqsOptions.getVisibilityDuplicateProtection();
            if (visibilityDuplicateProtectionEnabled) {
                expiryTimeout = maxVisibilityTimeout;
            }
            acknowledgementSet = this.acknowledgementSetManager.create(result -> {
                boolean successfullyDeletedAllMessages;
                this.acknowledgementSetCallbackCounter.increment();
                if (visibilityDuplicateProtectionEnabled) {
                    this.parsedMessageVisibilityTimesMap.remove(parsedMessage);
                }
                if (result.booleanValue() && (successfullyDeletedAllMessages = this.deleteSqsMessages(waitingForAcknowledgements)) && this.s3SourceConfig.isDeleteS3ObjectsOnRead()) {
                    this.deleteS3Objects(s3ObjectDeletionWaitingForAcknowledgments);
                }
            }, Duration.ofSeconds(expiryTimeout));
            if (visibilityDuplicateProtectionEnabled) {
                acknowledgementSet.addProgressCheck(ratio -> {
                    int newVisibilityTimeoutSeconds = visibilityTimeout;
                    int newValue = this.parsedMessageVisibilityTimesMap.getOrDefault(parsedMessage, visibilityTimeout) + progressCheckInterval;
                    if (newValue >= maxVisibilityTimeout) {
                        return;
                    }
                    this.parsedMessageVisibilityTimesMap.put(parsedMessage, newValue);
                    this.increaseVisibilityTimeout(parsedMessage, newVisibilityTimeoutSeconds);
                }, Duration.ofSeconds(progressCheckInterval));
            }
            messageAcknowledgementSetMap.put(parsedMessage, acknowledgementSet);
            messageWaitingForAcknowledgementsMap.put(parsedMessage, waitingForAcknowledgements);
            messagesWaitingForS3ObjectDeletion.put(parsedMessage, s3ObjectDeletionWaitingForAcknowledgments);
        }
        if (this.endToEndAcknowledgementsEnabled) {
            LOG.debug("Created acknowledgement sets for {} messages.", (Object)parsedMessagesToRead.size());
        }
        for (ParsedMessage parsedMessage : parsedMessagesToRead) {
            if (parsedMessage.isShouldSkipProcessing()) continue;
            AcknowledgementSet acknowledgementSet = (AcknowledgementSet)messageAcknowledgementSetMap.get(parsedMessage);
            waitingForAcknowledgements = (List)messageWaitingForAcknowledgementsMap.get(parsedMessage);
            List s3ObjectDeletionsWaitingForAcknowledgments = (List)messagesWaitingForS3ObjectDeletion.get(parsedMessage);
            S3ObjectReference s3ObjectReference = this.populateS3Reference(parsedMessage.getBucketName(), parsedMessage.getObjectKey());
            Optional<DeleteMessageBatchRequestEntry> deleteMessageBatchRequestEntry = this.processS3Object(parsedMessage, s3ObjectReference, acknowledgementSet);
            if (this.endToEndAcknowledgementsEnabled) {
                deleteMessageBatchRequestEntry.ifPresent(waitingForAcknowledgements::add);
                if (deleteMessageBatchRequestEntry.isPresent() && this.s3SourceConfig.isDeleteS3ObjectsOnRead()) {
                    s3ObjectDeletionsWaitingForAcknowledgments.add(s3ObjectReference);
                }
                acknowledgementSet.complete();
                continue;
            }
            deleteMessageBatchRequestEntry.ifPresent(deleteMessageBatchRequestEntryCollection::add);
        }
        return deleteMessageBatchRequestEntryCollection;
    }

    private void increaseVisibilityTimeout(ParsedMessage parsedMessage, int newVisibilityTimeoutSeconds) {
        if (this.isStopped) {
            LOG.info("Some messages are pending completion of acknowledgments. Data Prepper will not increase the visibility timeout because it is shutting down. {}", (Object)parsedMessage);
            return;
        }
        ChangeMessageVisibilityRequest changeMessageVisibilityRequest = (ChangeMessageVisibilityRequest)ChangeMessageVisibilityRequest.builder().visibilityTimeout(Integer.valueOf(newVisibilityTimeoutSeconds)).queueUrl(this.sqsOptions.getSqsUrl()).receiptHandle(parsedMessage.getMessage().receiptHandle()).build();
        try {
            this.sqsClient.changeMessageVisibility(changeMessageVisibilityRequest);
            this.sqsVisibilityTimeoutChangedCount.increment();
            LOG.debug("Set visibility timeout for message {} to {}", (Object)parsedMessage.getMessage().messageId(), (Object)newVisibilityTimeoutSeconds);
        }
        catch (Exception e) {
            LOG.error("Failed to set visibility timeout for message {} to {}", new Object[]{parsedMessage.getMessage().messageId(), newVisibilityTimeoutSeconds, e});
            this.sqsVisibilityTimeoutChangeFailedCount.increment();
        }
    }

    private Optional<DeleteMessageBatchRequestEntry> processS3Object(ParsedMessage parsedMessage, S3ObjectReference s3ObjectReference, AcknowledgementSet acknowledgementSet) {
        try {
            this.s3Service.addS3Object(s3ObjectReference, this.s3SourceConfig.getDataSelection(), acknowledgementSet);
            return Optional.of(this.buildDeleteMessageBatchRequestEntry(parsedMessage.getMessage()));
        }
        catch (Exception e) {
            LOG.error("Error processing from S3: {}. Retrying with exponential backoff.", (Object)e.getMessage());
            this.applyBackoff();
            return Optional.empty();
        }
    }

    private boolean deleteSqsMessages(List<DeleteMessageBatchRequestEntry> deleteMessageBatchRequestEntryCollection) {
        if (this.isStopped) {
            return false;
        }
        if (deleteMessageBatchRequestEntryCollection.size() == 0) {
            return false;
        }
        DeleteMessageBatchRequest deleteMessageBatchRequest = this.buildDeleteMessageBatchRequest(deleteMessageBatchRequestEntryCollection);
        try {
            int deletedMessagesCount;
            DeleteMessageBatchResponse deleteMessageBatchResponse = this.sqsClient.deleteMessageBatch(deleteMessageBatchRequest);
            if (deleteMessageBatchResponse.hasSuccessful() && (deletedMessagesCount = deleteMessageBatchResponse.successful().size()) > 0) {
                String successfullyDeletedMessages = deleteMessageBatchResponse.successful().stream().map(DeleteMessageBatchResultEntry::id).collect(Collectors.joining(", "));
                LOG.info("Deleted {} messages from SQS. [{}]", (Object)deletedMessagesCount, (Object)successfullyDeletedMessages);
                this.sqsMessagesDeletedCounter.increment((double)deletedMessagesCount);
            }
            if (deleteMessageBatchResponse.hasFailed()) {
                int failedDeleteCount = deleteMessageBatchResponse.failed().size();
                this.sqsMessagesDeleteFailedCounter.increment((double)failedDeleteCount);
                if (LOG.isErrorEnabled() && failedDeleteCount > 0) {
                    String failedMessages = deleteMessageBatchResponse.failed().stream().map(failed -> failed.toString()).collect(Collectors.joining(", "));
                    LOG.error("Failed to delete {} messages from SQS with errors: [{}].", (Object)failedDeleteCount, (Object)failedMessages);
                    return false;
                }
            }
        }
        catch (SdkException e) {
            int failedMessageCount = deleteMessageBatchRequestEntryCollection.size();
            this.sqsMessagesDeleteFailedCounter.increment((double)failedMessageCount);
            LOG.error("Failed to delete {} messages from SQS due to {}.", (Object)failedMessageCount, (Object)e.getMessage());
            if (e instanceof StsException) {
                this.applyBackoff();
            }
            return false;
        }
        return true;
    }

    private void deleteS3Objects(List<S3ObjectReference> objectsToDelete) {
        for (S3ObjectReference s3ObjectReference : objectsToDelete) {
            try {
                this.s3Service.deleteS3Object(s3ObjectReference);
            }
            catch (Exception e) {
                LOG.error("Received an exception while attempting to delete object {} from bucket {}: {}", new Object[]{s3ObjectReference.getKey(), s3ObjectReference.getBucketName(), e.getMessage()});
            }
        }
    }

    private DeleteMessageBatchRequestEntry buildDeleteMessageBatchRequestEntry(Message message) {
        return (DeleteMessageBatchRequestEntry)DeleteMessageBatchRequestEntry.builder().id(message.messageId()).receiptHandle(message.receiptHandle()).build();
    }

    private DeleteMessageBatchRequest buildDeleteMessageBatchRequest(List<DeleteMessageBatchRequestEntry> deleteMessageBatchRequestEntryCollection) {
        return (DeleteMessageBatchRequest)DeleteMessageBatchRequest.builder().queueUrl(this.s3SourceConfig.getSqsOptions().getSqsUrl()).entries(deleteMessageBatchRequestEntryCollection).build();
    }

    private boolean isS3EventNameCreated(ParsedMessage parsedMessage) {
        return this.objectCreatedFilter.filter(parsedMessage).isPresent();
    }

    private boolean isEventBridgeEventTypeCreated(ParsedMessage parsedMessage) {
        return this.evenBridgeObjectCreatedFilter.filter(parsedMessage).isPresent();
    }

    private S3ObjectReference populateS3Reference(String bucketName, String objectKey) {
        return S3ObjectReference.bucketAndKey((String)bucketName, (String)objectKey).build();
    }

    private int getApproximateReceiveCount(Message message) {
        return message.attributes() != null && message.attributes().get(MessageSystemAttributeName.APPROXIMATE_RECEIVE_COUNT) != null ? Integer.parseInt((String)message.attributes().get(MessageSystemAttributeName.APPROXIMATE_RECEIVE_COUNT)) : 0;
    }

    void stop() {
        this.isStopped = true;
    }

    private void recordSqsException(AwsServiceException e) {
        if (e.statusCode() == 403 || e instanceof KmsAccessDeniedException) {
            this.sqsMessageAccessDeniedCounter.increment();
        } else if (e.statusCode() == 404 || e instanceof QueueDoesNotExistException || e instanceof KmsNotFoundException) {
            this.sqsResourceNotFoundCounter.increment();
        } else if (e.isThrottlingException() || e instanceof KmsThrottledException) {
            this.sqsMessageThrottledCounter.increment();
        }
    }
}

