/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.file;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;
import org.opensearch.dataprepper.logging.DataPrepperMarkers;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.annotations.DataPrepperPlugin;
import org.opensearch.dataprepper.model.annotations.DataPrepperPluginConstructor;
import org.opensearch.dataprepper.model.buffer.Buffer;
import org.opensearch.dataprepper.model.codec.DecompressionEngine;
import org.opensearch.dataprepper.model.codec.InputCodec;
import org.opensearch.dataprepper.model.configuration.PluginModel;
import org.opensearch.dataprepper.model.configuration.PluginSetting;
import org.opensearch.dataprepper.model.event.EventBuilder;
import org.opensearch.dataprepper.model.event.EventFactory;
import org.opensearch.dataprepper.model.plugin.PluginFactory;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.model.source.Source;
import org.opensearch.dataprepper.plugins.source.file.FileSourceConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@DataPrepperPlugin(name="file", pluginType=Source.class, pluginConfigurationType=FileSourceConfig.class)
public class FileSource
implements Source<Record<Object>> {
    static final String MESSAGE_KEY = "message";
    private static final Logger LOG = LoggerFactory.getLogger(FileSource.class);
    private static final TypeReference<Map<String, Object>> MAP_TYPE_REFERENCE = new TypeReference<Map<String, Object>>(){};
    private static final ObjectMapper OBJECT_MAPPER = new ObjectMapper();
    private static final long STOP_WAIT_MILLIS = 200L;
    private final FileSourceConfig fileSourceConfig;
    private final FileStrategy fileStrategy;
    private final EventFactory eventFactory;
    private final DecompressionEngine decompressionEngine;
    private Thread readThread;
    private boolean isStopRequested;
    private final int writeTimeout;

    @DataPrepperPluginConstructor
    public FileSource(FileSourceConfig fileSourceConfig, PluginMetrics pluginMetrics, PluginFactory pluginFactory, EventFactory eventFactory) {
        this.eventFactory = eventFactory;
        fileSourceConfig.validate();
        this.fileSourceConfig = fileSourceConfig;
        this.isStopRequested = false;
        this.writeTimeout = 5000;
        this.decompressionEngine = fileSourceConfig.getCompression().getDecompressionEngine();
        this.fileStrategy = fileSourceConfig.getCodec() != null ? new CodecFileStrategy(pluginFactory) : new ClassicFileStrategy();
    }

    public void start(Buffer<Record<Object>> buffer) {
        Preconditions.checkNotNull(buffer, (Object)"Buffer cannot be null for file source to start");
        LOG.info("Starting file source with {} path.", (Object)this.fileSourceConfig.getFilePathToRead());
        this.readThread = new Thread(() -> {
            this.fileStrategy.start(buffer);
            LOG.info("Completed reading file.");
        }, "file-source");
        this.readThread.setDaemon(false);
        this.readThread.start();
    }

    public void stop() {
        this.isStopRequested = true;
        try {
            this.readThread.join(200L);
        }
        catch (InterruptedException e) {
            this.readThread.interrupt();
        }
    }

    private class CodecFileStrategy
    implements FileStrategy {
        private final InputCodec codec;

        CodecFileStrategy(PluginFactory pluginFactory) {
            PluginModel codecConfiguration = FileSource.this.fileSourceConfig.getCodec();
            PluginSetting codecPluginSettings = new PluginSetting(codecConfiguration.getPluginName(), codecConfiguration.getPluginSettings());
            this.codec = (InputCodec)pluginFactory.loadPlugin(InputCodec.class, codecPluginSettings, new Object[0]);
        }

        @Override
        public void start(Buffer<Record<Object>> buffer) {
            Path filePath = Paths.get(FileSource.this.fileSourceConfig.getFilePathToRead(), new String[0]);
            try (InputStream is = FileSource.this.decompressionEngine.createInputStream(Files.newInputStream(filePath, new OpenOption[0]));){
                this.codec.parse(is, eventRecord -> {
                    try {
                        buffer.write(eventRecord, FileSource.this.writeTimeout);
                    }
                    catch (TimeoutException e) {
                        throw new RuntimeException(e);
                    }
                });
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }

    private class ClassicFileStrategy
    implements FileStrategy {
        private ClassicFileStrategy() {
        }

        @Override
        public void start(Buffer<Record<Object>> buffer) {
            Path filePath = Paths.get(FileSource.this.fileSourceConfig.getFilePathToRead(), new String[0]);
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(FileSource.this.decompressionEngine.createInputStream(Files.newInputStream(filePath, new OpenOption[0])), StandardCharsets.UTF_8));){
                String line;
                while ((line = reader.readLine()) != null && !FileSource.this.isStopRequested) {
                    this.writeLineAsEventOrString(line, buffer);
                }
            }
            catch (IOException | IllegalArgumentException | TimeoutException ex) {
                LOG.error("Error processing the input file path [{}]", (Object)FileSource.this.fileSourceConfig.getFilePathToRead(), (Object)ex);
                throw new RuntimeException(String.format("Error processing the input file %s", FileSource.this.fileSourceConfig.getFilePathToRead()), ex);
            }
        }

        private Record<Object> getEventRecordFromLine(String line) {
            Map<String, Object> structuredLine = new HashMap<String, String>();
            switch (FileSource.this.fileSourceConfig.getFormat()) {
                case JSON: {
                    structuredLine = this.parseJson(line);
                    break;
                }
                case PLAIN: {
                    structuredLine.put(FileSource.MESSAGE_KEY, line);
                }
            }
            return new Record((Object)((EventBuilder)FileSource.this.eventFactory.eventBuilder(EventBuilder.class)).withEventType(FileSource.this.fileSourceConfig.getRecordType()).withData(structuredLine).build());
        }

        private Map<String, Object> parseJson(String jsonString) {
            try {
                return (Map)OBJECT_MAPPER.readValue(jsonString, MAP_TYPE_REFERENCE);
            }
            catch (JsonProcessingException e) {
                LOG.error(DataPrepperMarkers.SENSITIVE, "Unable to parse json data [{}], assuming plain text", (Object)jsonString, (Object)e);
                HashMap<String, Object> plainMap = new HashMap<String, Object>();
                plainMap.put(FileSource.MESSAGE_KEY, jsonString);
                return plainMap;
            }
        }

        private void writeLineAsEventOrString(String line, Buffer<Record<Object>> buffer) throws TimeoutException, IllegalArgumentException {
            if (FileSource.this.fileSourceConfig.getRecordType().equals("event")) {
                buffer.write(this.getEventRecordFromLine(line), FileSource.this.writeTimeout);
            } else if (FileSource.this.fileSourceConfig.getRecordType().equals("string")) {
                buffer.write(new Record((Object)line), FileSource.this.writeTimeout);
            }
        }
    }

    private static interface FileStrategy {
        public void start(Buffer<Record<Object>> var1);
    }
}

