/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.sqs;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Timer;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.model.buffer.Buffer;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.plugins.source.sqs.QueueConfig;
import org.opensearch.dataprepper.plugins.source.sqs.SqsEventProcessor;
import org.opensearch.dataprepper.plugins.source.sqs.SqsSourceConfig;
import org.opensearch.dataprepper.plugins.source.sqs.common.OnErrorOption;
import org.opensearch.dataprepper.plugins.source.sqs.common.SqsWorkerCommon;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.sqs.SqsClient;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequestEntry;
import software.amazon.awssdk.services.sqs.model.Message;
import software.amazon.awssdk.services.sqs.model.MessageSystemAttributeName;
import software.amazon.awssdk.services.sqs.model.SqsException;

public class SqsWorker
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(SqsWorker.class);
    static final String SQS_MESSAGE_DELAY_METRIC_NAME = "sqsMessageDelay";
    private final Timer sqsMessageDelayTimer;
    static final String ACKNOWLEDGEMENT_SET_CALLACK_METRIC_NAME = "acknowledgementSetCallbackCounter";
    private final SqsWorkerCommon sqsWorkerCommon;
    private final SqsEventProcessor sqsEventProcessor;
    private final SqsClient sqsClient;
    private final QueueConfig queueConfig;
    private final boolean endToEndAcknowledgementsEnabled;
    private final Buffer<Record<Event>> buffer;
    private final int bufferTimeoutMillis;
    private final AcknowledgementSetManager acknowledgementSetManager;
    private final Counter acknowledgementSetCallbackCounter;
    private int failedAttemptCount;
    private volatile boolean isStopped = false;
    private final Map<Message, Integer> messageVisibilityTimesMap;

    public SqsWorker(Buffer<Record<Event>> buffer, AcknowledgementSetManager acknowledgementSetManager, SqsClient sqsClient, SqsWorkerCommon sqsWorkerCommon, SqsSourceConfig sqsSourceConfig, QueueConfig queueConfig, PluginMetrics pluginMetrics, SqsEventProcessor sqsEventProcessor) {
        this.sqsWorkerCommon = sqsWorkerCommon;
        this.queueConfig = queueConfig;
        this.acknowledgementSetManager = acknowledgementSetManager;
        this.sqsEventProcessor = sqsEventProcessor;
        this.sqsClient = sqsClient;
        this.buffer = buffer;
        this.bufferTimeoutMillis = (int)sqsSourceConfig.getBufferTimeout().toMillis();
        this.endToEndAcknowledgementsEnabled = sqsSourceConfig.getAcknowledgements();
        this.messageVisibilityTimesMap = new HashMap<Message, Integer>();
        this.failedAttemptCount = 0;
        this.acknowledgementSetCallbackCounter = pluginMetrics.counter(ACKNOWLEDGEMENT_SET_CALLACK_METRIC_NAME);
        this.sqsMessageDelayTimer = pluginMetrics.timer(SQS_MESSAGE_DELAY_METRIC_NAME);
    }

    @Override
    public void run() {
        while (!this.isStopped) {
            int messagesProcessed = 0;
            try {
                messagesProcessed = this.processSqsMessages();
            }
            catch (Exception e) {
                LOG.error("Unable to process SQS messages. Processing error due to: {}", (Object)e.getMessage());
                this.sqsWorkerCommon.applyBackoff();
            }
            if (messagesProcessed <= 0 || this.queueConfig.getPollDelay().toMillis() <= 0L) continue;
            try {
                Thread.sleep(this.queueConfig.getPollDelay().toMillis());
            }
            catch (InterruptedException e) {
                LOG.error("Thread is interrupted while polling SQS.", (Throwable)e);
                Thread.currentThread().interrupt();
            }
        }
    }

    int processSqsMessages() {
        try {
            List messages = this.sqsWorkerCommon.pollSqsMessages(this.queueConfig.getUrl(), this.sqsClient, this.queueConfig.getMaximumMessages(), this.queueConfig.getWaitTime(), this.queueConfig.getVisibilityTimeout());
            if (!messages.isEmpty()) {
                List<DeleteMessageBatchRequestEntry> deleteEntries = this.processSqsEvents(messages);
                if (!deleteEntries.isEmpty()) {
                    this.sqsWorkerCommon.deleteSqsMessages(this.queueConfig.getUrl(), this.sqsClient, deleteEntries);
                }
            } else {
                this.sqsMessageDelayTimer.record(Duration.ZERO);
            }
            return messages.size();
        }
        catch (SqsException e) {
            this.sqsWorkerCommon.applyBackoff();
            return 0;
        }
    }

    private List<DeleteMessageBatchRequestEntry> processSqsEvents(List<Message> messages) {
        AcknowledgementSet acknowledgementSet;
        ArrayList<DeleteMessageBatchRequestEntry> deleteMessageBatchRequestEntryCollection = new ArrayList<DeleteMessageBatchRequestEntry>();
        HashMap<Message, AcknowledgementSet> messageAcknowledgementSetMap = new HashMap<Message, AcknowledgementSet>();
        HashMap messageWaitingForAcknowledgementsMap = new HashMap();
        for (Message message : messages) {
            ArrayList waitingForAcknowledgements = new ArrayList();
            acknowledgementSet = null;
            int visibilityTimeout = this.queueConfig.getVisibilityTimeout() != null ? (int)this.queueConfig.getVisibilityTimeout().getSeconds() : 30;
            int maxVisibilityTimeout = (int)this.queueConfig.getVisibilityDuplicateProtectionTimeout().getSeconds();
            int progressCheckInterval = visibilityTimeout / 2 - 1;
            if (!this.endToEndAcknowledgementsEnabled) continue;
            int expiryTimeout = this.queueConfig.getVisibilityDuplicateProtection() ? maxVisibilityTimeout : visibilityTimeout - 2;
            acknowledgementSet = this.acknowledgementSetManager.create(result -> {
                this.acknowledgementSetCallbackCounter.increment();
                if (this.queueConfig.getVisibilityDuplicateProtection()) {
                    this.messageVisibilityTimesMap.remove(message);
                }
                if (result.booleanValue()) {
                    this.sqsWorkerCommon.deleteSqsMessages(this.queueConfig.getUrl(), this.sqsClient, waitingForAcknowledgements);
                }
            }, Duration.ofSeconds(expiryTimeout));
            if (this.queueConfig.getVisibilityDuplicateProtection()) {
                acknowledgementSet.addProgressCheck(ratio -> {
                    int newValue = this.messageVisibilityTimesMap.getOrDefault(message, visibilityTimeout) + progressCheckInterval;
                    if (newValue >= maxVisibilityTimeout) {
                        return;
                    }
                    this.messageVisibilityTimesMap.put(message, newValue);
                    this.sqsWorkerCommon.increaseVisibilityTimeout(this.queueConfig.getUrl(), this.sqsClient, message.receiptHandle(), visibilityTimeout, message.messageId());
                }, Duration.ofSeconds(progressCheckInterval));
            }
            messageAcknowledgementSetMap.put(message, acknowledgementSet);
            messageWaitingForAcknowledgementsMap.put(message, waitingForAcknowledgements);
        }
        for (Message message : messages) {
            int receiveCount = Integer.parseInt((String)message.attributes().get(MessageSystemAttributeName.APPROXIMATE_RECEIVE_COUNT));
            if (receiveCount <= 1) {
                Duration duration = Duration.between(Instant.ofEpochMilli(Long.parseLong((String)message.attributes().get(MessageSystemAttributeName.SENT_TIMESTAMP))), Instant.now());
                this.sqsMessageDelayTimer.record(duration.isNegative() ? Duration.ZERO : duration);
            }
            acknowledgementSet = (AcknowledgementSet)messageAcknowledgementSetMap.get(message);
            List waitingForAcknowledgements = (List)messageWaitingForAcknowledgementsMap.get(message);
            Optional<DeleteMessageBatchRequestEntry> deleteEntry = this.processSqsObject(message, acknowledgementSet);
            if (this.endToEndAcknowledgementsEnabled) {
                deleteEntry.ifPresent(waitingForAcknowledgements::add);
                if (acknowledgementSet == null) continue;
                acknowledgementSet.complete();
                continue;
            }
            deleteEntry.ifPresent(deleteMessageBatchRequestEntryCollection::add);
        }
        return deleteMessageBatchRequestEntryCollection;
    }

    private Optional<DeleteMessageBatchRequestEntry> processSqsObject(Message message, AcknowledgementSet acknowledgementSet) {
        try {
            this.sqsEventProcessor.addSqsObject(message, this.queueConfig.getUrl(), this.buffer, this.bufferTimeoutMillis, acknowledgementSet);
            return Optional.of(this.sqsWorkerCommon.buildDeleteMessageBatchRequestEntry(message.messageId(), message.receiptHandle()));
        }
        catch (Exception e) {
            this.sqsWorkerCommon.getSqsMessagesFailedCounter().increment();
            LOG.error("Error processing from SQS: {}. Retrying with exponential backoff.", (Object)e.getMessage());
            this.sqsWorkerCommon.applyBackoff();
            if (this.queueConfig.getOnErrorOption().equals((Object)OnErrorOption.DELETE_MESSAGES)) {
                return Optional.of(this.sqsWorkerCommon.buildDeleteMessageBatchRequestEntry(message.messageId(), message.receiptHandle()));
            }
            return Optional.empty();
        }
    }

    void stop() {
        this.isStopped = true;
        this.sqsWorkerCommon.stop();
    }
}

