/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.microsoft_office365;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import javax.inject.Named;
import org.opensearch.dataprepper.logging.DataPrepperMarkers;
import org.opensearch.dataprepper.plugins.source.microsoft_office365.auth.Office365AuthenticationInterface;
import org.opensearch.dataprepper.plugins.source.microsoft_office365.models.AuditLogsResponse;
import org.opensearch.dataprepper.plugins.source.microsoft_office365.utils.Constants;
import org.opensearch.dataprepper.plugins.source.source_crawler.exception.SaaSCrawlerException;
import org.opensearch.dataprepper.plugins.source.source_crawler.metrics.VendorAPIMetricsRecorder;
import org.opensearch.dataprepper.plugins.source.source_crawler.utils.retry.DefaultRetryStrategy;
import org.opensearch.dataprepper.plugins.source.source_crawler.utils.retry.DefaultStatusCodeHandler;
import org.opensearch.dataprepper.plugins.source.source_crawler.utils.retry.RetryHandler;
import org.opensearch.dataprepper.plugins.source.source_crawler.utils.retry.RetryStrategy;
import org.opensearch.dataprepper.plugins.source.source_crawler.utils.retry.StatusCodeHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.HttpServerErrorException;
import org.springframework.web.client.RestTemplate;

@Named
public class Office365RestClient {
    private static final Logger log = LoggerFactory.getLogger(Office365RestClient.class);
    private static final String MANAGEMENT_API_BASE_URL = "https://manage.office.com/api/v1.0/";
    private static final String SUBSCRIPTION_LIST_URL = "https://manage.office.com/api/v1.0/%s/activity/feed/subscriptions/list";
    private static final String SUBSCRIPTION_START_URL = "https://manage.office.com/api/v1.0/%s/activity/feed/subscriptions/start?contentType=%s";
    private static final String GET_AUDIT_LOGS_URL = "https://manage.office.com/api/v1.0/%s/activity/feed/subscriptions/content?contentType=%s&startTime=%s&endTime=%s";
    private final RestTemplate restTemplate = new RestTemplate();
    private final RetryHandler retryHandler;
    private final Office365AuthenticationInterface authConfig;
    private final VendorAPIMetricsRecorder metricsRecorder;

    public Office365RestClient(Office365AuthenticationInterface authConfig, VendorAPIMetricsRecorder metricsRecorder) {
        this.authConfig = authConfig;
        this.metricsRecorder = metricsRecorder;
        this.retryHandler = new RetryHandler((RetryStrategy)new DefaultRetryStrategy(), (StatusCodeHandler)new DefaultStatusCodeHandler());
    }

    private List<Map<String, Object>> listSubscriptions() {
        log.info("Listing Office 365 subscriptions");
        String listUrl = String.format(SUBSCRIPTION_LIST_URL, this.authConfig.getTenantId());
        return (List)this.metricsRecorder.recordListSubscriptionLatency(() -> {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            try {
                List result = (List)this.retryHandler.executeWithRetry(() -> {
                    headers.setBearerAuth(this.authConfig.getAccessToken());
                    this.metricsRecorder.recordListSubscriptionCall();
                    ResponseEntity response = this.restTemplate.exchange(listUrl, HttpMethod.GET, new HttpEntity((MultiValueMap)headers), (ParameterizedTypeReference)new ParameterizedTypeReference<List<Map<String, Object>>>(){}, new Object[0]);
                    log.debug("Current subscriptions: {}", response.getBody());
                    return (List)response.getBody();
                }, () -> ((Office365AuthenticationInterface)this.authConfig).renewCredentials(), () -> ((VendorAPIMetricsRecorder)this.metricsRecorder).recordListSubscriptionFailure());
                this.metricsRecorder.recordListSubscriptionSuccess();
                return result;
            }
            catch (Exception e) {
                this.metricsRecorder.recordError(e);
                log.error(DataPrepperMarkers.NOISY, "Failed to list subscriptions: {}", (Object)e.getMessage());
                throw new SaaSCrawlerException("Failed to list subscriptions: " + e.getMessage(), (Throwable)e, true);
            }
        });
    }

    private void startSubscriptionsForContentTypes(List<String> contentTypesToStart) {
        log.info("Starting {} subscription(s)", (Object)contentTypesToStart.size());
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.setContentLength(0L);
        for (String contentType : contentTypesToStart) {
            String url = String.format(SUBSCRIPTION_START_URL, this.authConfig.getTenantId(), contentType);
            try {
                this.retryHandler.executeWithRetry(() -> {
                    headers.setBearerAuth(this.authConfig.getAccessToken());
                    this.metricsRecorder.recordSubscriptionCall();
                    ResponseEntity response = this.restTemplate.exchange(url, HttpMethod.POST, new HttpEntity((MultiValueMap)headers), String.class, new Object[0]);
                    log.info("Successfully started subscription for {}: {}", (Object)contentType, response.getBody());
                    return response;
                }, () -> ((Office365AuthenticationInterface)this.authConfig).renewCredentials(), () -> ((VendorAPIMetricsRecorder)this.metricsRecorder).recordSubscriptionFailure());
            }
            catch (HttpClientErrorException | HttpServerErrorException e) {
                if (e.getResponseBodyAsString().contains("AF20024")) {
                    log.debug("Subscription for {} is already enabled", (Object)contentType);
                    continue;
                }
                this.metricsRecorder.recordError((Exception)e);
                throw new SaaSCrawlerException("Failed to start subscription for " + contentType + ": " + e.getMessage(), e, true);
            }
            catch (Exception e) {
                this.metricsRecorder.recordError(e);
                throw new SaaSCrawlerException("Failed to start subscription for " + contentType + ": " + e.getMessage(), (Throwable)e, true);
            }
        }
        log.info("Successfully started {} subscription(s)", (Object)contentTypesToStart.size());
    }

    public void startSubscriptions() {
        log.info("Starting Office 365 subscriptions for audit logs");
        this.metricsRecorder.recordSubscriptionLatency(() -> {
            try {
                ArrayList<String> contentTypesToStart = new ArrayList<String>();
                try {
                    List<Map<String, Object>> currentSubscriptions = this.listSubscriptions();
                    HashSet<String> enabledContentTypes = new HashSet<String>();
                    for (Map<String, Object> subscription : currentSubscriptions) {
                        String contentType = (String)subscription.get("contentType");
                        String status = (String)subscription.get("status");
                        if (!"enabled".equalsIgnoreCase(status)) continue;
                        enabledContentTypes.add(contentType);
                        log.info("Content type {} is already enabled", (Object)contentType);
                    }
                    for (String contentType : Constants.CONTENT_TYPES) {
                        if (enabledContentTypes.contains(contentType)) continue;
                        contentTypesToStart.add(contentType);
                        log.info("Content type {} needs to be started", (Object)contentType);
                    }
                    if (contentTypesToStart.isEmpty()) {
                        log.info("All content types are already enabled. No subscriptions need to be started.");
                        this.metricsRecorder.recordSubscriptionSuccess();
                        return null;
                    }
                }
                catch (Exception e) {
                    log.warn("Failed to list subscriptions, will attempt to start all content types as fallback: {}", (Object)e.getMessage());
                    contentTypesToStart.clear();
                    for (String contentType : Constants.CONTENT_TYPES) {
                        contentTypesToStart.add(contentType);
                    }
                }
                this.startSubscriptionsForContentTypes(contentTypesToStart);
                this.metricsRecorder.recordSubscriptionSuccess();
                return null;
            }
            catch (Exception e) {
                this.metricsRecorder.recordError(e);
                log.error(DataPrepperMarkers.NOISY, "Failed to initialize subscriptions", (Throwable)e);
                throw new SaaSCrawlerException("Failed to initialize subscriptions: " + e.getMessage(), (Throwable)e, true);
            }
        });
    }

    public AuditLogsResponse searchAuditLogs(String contentType, Instant startTime, Instant endTime, String pageUri) {
        String url = pageUri != null ? pageUri : String.format(GET_AUDIT_LOGS_URL, this.authConfig.getTenantId(), contentType, startTime.toString(), endTime.toString());
        log.debug("Searching audit logs with URL: {}", (Object)url);
        HttpHeaders headers = new HttpHeaders();
        return (AuditLogsResponse)this.metricsRecorder.recordSearchLatency(() -> {
            try {
                return (AuditLogsResponse)this.retryHandler.executeWithRetry(() -> {
                    String nextPageUri;
                    headers.setBearerAuth(this.authConfig.getAccessToken());
                    this.metricsRecorder.recordDataApiRequest();
                    ResponseEntity response = this.restTemplate.exchange(url, HttpMethod.GET, new HttpEntity((MultiValueMap)headers), (ParameterizedTypeReference)new ParameterizedTypeReference<List<Map<String, Object>>>(){}, new Object[0]);
                    this.metricsRecorder.recordSearchResponseSize(response);
                    this.metricsRecorder.recordSearchSuccess();
                    List nextPageHeaders = response.getHeaders().get((Object)"NextPageUri");
                    String string = nextPageUri = nextPageHeaders != null && !nextPageHeaders.isEmpty() ? (String)nextPageHeaders.get(0) : null;
                    if (nextPageUri != null) {
                        log.debug("Next page URI found: {}", (Object)nextPageUri);
                    }
                    return new AuditLogsResponse((List)response.getBody(), nextPageUri);
                }, () -> ((Office365AuthenticationInterface)this.authConfig).renewCredentials(), () -> ((VendorAPIMetricsRecorder)this.metricsRecorder).recordSearchFailure());
            }
            catch (Exception e) {
                this.metricsRecorder.recordError(e);
                log.error(DataPrepperMarkers.NOISY, "Error while fetching audit logs for content type {} from URL: {}", new Object[]{contentType, url, e});
                throw new SaaSCrawlerException("Failed to fetch audit logs", (Throwable)e, true);
            }
        });
    }

    public String getAuditLog(String contentUri) {
        if (!contentUri.startsWith(MANAGEMENT_API_BASE_URL)) {
            throw new SaaSCrawlerException("ContentUri must be from Office365 Management API: " + contentUri, false);
        }
        log.debug("Getting audit log from content URI: {}", (Object)contentUri);
        this.metricsRecorder.recordLogsRequested();
        HttpHeaders headers = new HttpHeaders();
        return (String)this.metricsRecorder.recordGetLatency(() -> {
            try {
                String response = (String)this.retryHandler.executeWithRetry(() -> {
                    headers.setBearerAuth(this.authConfig.getAccessToken());
                    this.metricsRecorder.recordDataApiRequest();
                    ResponseEntity responseEntity = this.restTemplate.exchange(contentUri, HttpMethod.GET, new HttpEntity((MultiValueMap)headers), String.class, new Object[0]);
                    return (String)responseEntity.getBody();
                }, () -> ((Office365AuthenticationInterface)this.authConfig).renewCredentials(), () -> ((VendorAPIMetricsRecorder)this.metricsRecorder).recordGetFailure());
                this.metricsRecorder.recordGetResponseSize(response);
                this.metricsRecorder.recordGetSuccess();
                return response;
            }
            catch (Exception e) {
                this.metricsRecorder.recordError(e);
                log.error(DataPrepperMarkers.NOISY, "Error while fetching audit log content from URI: {}", (Object)contentUri, (Object)e);
                throw new SaaSCrawlerException("Failed to fetch audit log", (Throwable)e, true);
            }
        });
    }
}

