/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.dynamodb.leader;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.opensearch.dataprepper.model.plugin.InvalidPluginConfigurationException;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourcePartition;
import org.opensearch.dataprepper.plugins.source.dynamodb.configuration.StreamStartPosition;
import org.opensearch.dataprepper.plugins.source.dynamodb.configuration.TableConfig;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.partition.ExportPartition;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.partition.GlobalState;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.partition.LeaderPartition;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.partition.StreamPartition;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.state.ExportProgressState;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.state.LeaderProgressState;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.state.StreamProgressState;
import org.opensearch.dataprepper.plugins.source.dynamodb.leader.ShardManager;
import org.opensearch.dataprepper.plugins.source.dynamodb.model.TableInfo;
import org.opensearch.dataprepper.plugins.source.dynamodb.model.TableMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.dynamodb.DynamoDbClient;
import software.amazon.awssdk.services.dynamodb.model.DescribeContinuousBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContinuousBackupsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.services.dynamodb.model.Shard;

public class LeaderScheduler
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(LeaderScheduler.class);
    private static final int DEFAULT_EXTEND_LEASE_MINUTES = 3;
    private static final Duration DEFAULT_LEASE_INTERVAL = Duration.ofMinutes(1L);
    private final List<TableConfig> tableConfigs;
    private final EnhancedSourceCoordinator coordinator;
    private final DynamoDbClient dynamoDbClient;
    private final ShardManager shardManager;
    private final Duration leaseInterval;
    private LeaderPartition leaderPartition;
    private List<String> streamArns;

    public LeaderScheduler(EnhancedSourceCoordinator coordinator, DynamoDbClient dynamoDbClient, ShardManager shardManager, List<TableConfig> tableConfigs) {
        this(coordinator, dynamoDbClient, shardManager, tableConfigs, DEFAULT_LEASE_INTERVAL);
    }

    LeaderScheduler(EnhancedSourceCoordinator coordinator, DynamoDbClient dynamoDbClient, ShardManager shardManager, List<TableConfig> tableConfigs, Duration leaseInterval) {
        this.tableConfigs = tableConfigs;
        this.coordinator = coordinator;
        this.dynamoDbClient = dynamoDbClient;
        this.shardManager = shardManager;
        this.leaseInterval = leaseInterval;
    }

    @Override
    public void run() {
        LOG.debug("Starting Leader Scheduler for initialization and shard discovery");
        while (!Thread.currentThread().isInterrupted()) {
            try {
                Optional sourcePartition;
                if (this.leaderPartition == null && (sourcePartition = this.coordinator.acquireAvailablePartition("LEADER")).isPresent()) {
                    LOG.info("Running as a LEADER node");
                    this.leaderPartition = (LeaderPartition)((Object)sourcePartition.get());
                }
                if (this.leaderPartition == null) continue;
                LeaderProgressState leaderProgressState = this.leaderPartition.getProgressState().get();
                if (!leaderProgressState.isInitialized()) {
                    LOG.debug("The service is not been initialized");
                    this.init();
                } else if (this.streamArns == null) {
                    this.streamArns = leaderProgressState.getStreamArns();
                }
                if (this.streamArns == null || this.streamArns.isEmpty()) continue;
                this.streamArns.forEach(streamArn -> this.shardManager.runDiscovery((String)streamArn));
                List sourcePartitions = this.coordinator.queryCompletedPartitions("STREAM", Instant.now().minus(Duration.ofDays(1L)));
                this.compareAndCreateChildrenPartitions(sourcePartitions);
            }
            catch (Exception e) {
                LOG.error("Exception occurred in primary scheduling loop", (Throwable)e);
            }
            finally {
                if (this.leaderPartition != null) {
                    try {
                        this.coordinator.saveProgressStateForPartition((EnhancedSourcePartition)this.leaderPartition, Duration.ofMinutes(3L));
                    }
                    catch (Exception e) {
                        LOG.error("Failed to update ownership for leader partition. Will attempt to reacquire this partition...");
                        this.leaderPartition = null;
                    }
                }
                try {
                    Thread.sleep(this.leaseInterval.toMillis());
                }
                catch (InterruptedException e) {
                    LOG.info("InterruptedException occurred");
                    break;
                }
            }
        }
        LOG.warn("Quitting Leader Scheduler");
        if (this.leaderPartition != null) {
            this.coordinator.giveUpPartition((EnhancedSourcePartition)this.leaderPartition);
        }
    }

    private void init() {
        LOG.info("Try to initialize DynamoDB service");
        List<TableInfo> tableInfos = this.tableConfigs.stream().map(this::getTableInfo).collect(Collectors.toList());
        this.streamArns = new ArrayList<String>();
        tableInfos.forEach(tableInfo -> {
            this.coordinator.createPartition((EnhancedSourcePartition)new GlobalState(tableInfo.getTableArn(), Optional.of(tableInfo.getMetadata().toMap())));
            Instant startTime = Instant.now();
            if (tableInfo.getMetadata().isExportRequired()) {
                this.createExportPartition(tableInfo.getTableArn(), startTime, tableInfo.getMetadata().getExportBucket(), tableInfo.getMetadata().getExportPrefix(), tableInfo.getMetadata().getExportKmsKeyId());
            }
            if (tableInfo.getMetadata().isStreamRequired()) {
                List<Shard> shards = this.shardManager.runDiscovery(tableInfo.getMetadata().getStreamArn());
                List childIds = shards.stream().map(shard -> shard.shardId()).collect(Collectors.toList());
                List<Shard> rootShards = shards.stream().filter(shard -> shard.parentShardId() == null || !childIds.contains(shard.parentShardId())).collect(Collectors.toList());
                LOG.info("Found {} root shards in total", (Object)rootShards.size());
                rootShards.forEach(shard -> this.createRootStreamPartition(tableInfo.getMetadata().getStreamArn(), (Shard)shard, startTime, tableInfo.getMetadata().isExportRequired()));
                this.streamArns.add(tableInfo.getMetadata().getStreamArn());
            }
        });
        LOG.debug("Update initialization state");
        LeaderProgressState leaderProgressState = this.leaderPartition.getProgressState().get();
        leaderProgressState.setStreamArns(this.streamArns);
        leaderProgressState.setInitialized(true);
    }

    private void compareAndCreateChildrenPartitions(List<EnhancedSourcePartition> sourcePartitions) {
        if (sourcePartitions == null || sourcePartitions.isEmpty()) {
            return;
        }
        long startTime = System.currentTimeMillis();
        List completedShardIds = sourcePartitions.stream().map(sourcePartition -> ((StreamPartition)((Object)sourcePartition)).getShardId()).collect(Collectors.toList());
        sourcePartitions.forEach(sourcePartition -> {
            StreamPartition streamPartition = (StreamPartition)((Object)sourcePartition);
            List<String> childShardIds = this.shardManager.findChildShardIds(streamPartition.getStreamArn(), streamPartition.getShardId());
            if (childShardIds != null && !childShardIds.isEmpty()) {
                childShardIds.forEach(shardId -> {
                    if (!completedShardIds.contains(shardId)) {
                        this.createChildStreamPartition(streamPartition, (String)shardId);
                    }
                });
            }
        });
        long endTime = System.currentTimeMillis();
        LOG.debug("Compare and create children partitions took {} milliseconds", (Object)(endTime - startTime));
    }

    private TableInfo getTableInfo(TableConfig tableConfig) {
        DescribeTableResponse describeTableResult;
        String tableName = tableConfig.getTableArn();
        try {
            DescribeTableRequest req = (DescribeTableRequest)DescribeTableRequest.builder().tableName(tableName).build();
            describeTableResult = this.dynamoDbClient.describeTable(req);
        }
        catch (Exception e2) {
            LOG.error("Unable to call DescribeTableRequest to get information for table {} due to {}", (Object)tableName, (Object)e2.getMessage());
            throw new RuntimeException("Unable to get table information for " + tableName + ". Please make sure the permission is properly set");
        }
        Map<String, String> keys = describeTableResult.table().keySchema().stream().collect(Collectors.toMap(e -> e.keyTypeAsString(), e -> e.attributeName()));
        if (tableConfig.getExportConfig() != null) {
            String status = this.getContinuousBackupsStatus(tableName);
            LOG.debug("The PITR status for table " + tableName + " is " + status);
            if (!"ENABLED".equals(status)) {
                String errorMessage = "Point-in-time recovery (PITR) needs to be enabled for exporting data from table " + tableConfig.getTableArn();
                LOG.error(errorMessage);
                throw new InvalidPluginConfigurationException(errorMessage);
            }
        }
        StreamStartPosition streamStartPosition = null;
        if (tableConfig.getStreamConfig() != null) {
            if (describeTableResult.table().streamSpecification() == null) {
                String errorMessage = "Stream is not enabled for table " + tableConfig.getTableArn();
                LOG.error(errorMessage);
                throw new InvalidPluginConfigurationException(errorMessage);
            }
            String viewType = describeTableResult.table().streamSpecification().streamViewTypeAsString();
            LOG.debug("The stream view type for table " + tableName + " is " + viewType);
            List<String> supportedType = List.of("NEW_IMAGE", "NEW_AND_OLD_IMAGES");
            if (!supportedType.contains(viewType)) {
                String errorMessage = "Stream " + tableConfig.getTableArn() + " is enabled with " + viewType + ". Supported types are " + String.valueOf(supportedType);
                LOG.error(errorMessage);
                throw new InvalidPluginConfigurationException(errorMessage);
            }
            streamStartPosition = tableConfig.getStreamConfig().getStartPosition();
        }
        TableMetadata metadata = TableMetadata.builder().partitionKeyAttributeName(keys.get("HASH")).sortKeyAttributeName(keys.get("RANGE")).streamArn(describeTableResult.table().latestStreamArn()).streamRequired(tableConfig.getStreamConfig() != null).exportRequired(tableConfig.getExportConfig() != null).streamStartPosition(streamStartPosition).exportBucket(tableConfig.getExportConfig() == null ? null : tableConfig.getExportConfig().getS3Bucket()).exportPrefix(tableConfig.getExportConfig() == null ? null : tableConfig.getExportConfig().getS3Prefix()).exportKmsKeyId(tableConfig.getExportConfig() == null ? null : tableConfig.getExportConfig().getS3SseKmsKeyId()).build();
        return new TableInfo(tableConfig.getTableArn(), metadata);
    }

    private String getContinuousBackupsStatus(String tableName) {
        try {
            DescribeContinuousBackupsRequest req = (DescribeContinuousBackupsRequest)DescribeContinuousBackupsRequest.builder().tableName(tableName).build();
            DescribeContinuousBackupsResponse resp = this.dynamoDbClient.describeContinuousBackups(req);
            return resp.continuousBackupsDescription().pointInTimeRecoveryDescription().pointInTimeRecoveryStatus().toString();
        }
        catch (Exception e) {
            LOG.error("Unable to call describeContinuousBackupsRequest for table {} due to {}", (Object)tableName, (Object)e.getMessage());
            throw new RuntimeException("Unable to check if point in time recovery is enabled or not for " + tableName + ". Please make sure the permission is properly set");
        }
    }

    private void createRootStreamPartition(String streamArn, Shard shard, Instant exportTime, boolean waitForExport) {
        StreamProgressState streamProgressState = new StreamProgressState();
        streamProgressState.setWaitForExport(waitForExport);
        streamProgressState.setStartTime(exportTime.toEpochMilli());
        streamProgressState.setEndingSequenceNumber(shard.sequenceNumberRange().endingSequenceNumber());
        this.coordinator.createPartition((EnhancedSourcePartition)new StreamPartition(streamArn, shard.shardId(), Optional.of(streamProgressState)));
    }

    private void createChildStreamPartition(StreamPartition streamPartition, String childShardId) {
        StreamProgressState parentStreamProgressState = streamPartition.getProgressState().get();
        StreamProgressState streamProgressState = new StreamProgressState();
        streamProgressState.setStartTime(parentStreamProgressState.getStartTime());
        streamProgressState.setEndingSequenceNumber(this.shardManager.getEndingSequenceNumber(childShardId));
        streamProgressState.setWaitForExport(parentStreamProgressState.shouldWaitForExport());
        StreamPartition partition = new StreamPartition(streamPartition.getStreamArn(), childShardId, Optional.of(streamProgressState));
        this.coordinator.createPartition((EnhancedSourcePartition)partition);
    }

    private void createExportPartition(String tableArn, Instant exportTime, String bucket, String prefix, String kmsKeyId) {
        ExportProgressState exportProgressState = new ExportProgressState();
        exportProgressState.setBucket(bucket);
        exportProgressState.setPrefix(prefix);
        exportProgressState.setExportTime(exportTime.toString());
        exportProgressState.setKmsKeyId(kmsKeyId);
        ExportPartition exportPartition = new ExportPartition(tableArn, exportTime, Optional.of(exportProgressState));
        this.coordinator.createPartition((EnhancedSourcePartition)exportPartition);
    }
}

