/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.dynamodb.stream;

import com.google.common.annotations.VisibleForTesting;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.opensearch.dataprepper.common.concurrent.BackgroundThreadFactory;
import org.opensearch.dataprepper.logging.DataPrepperMarkers;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourcePartition;
import org.opensearch.dataprepper.model.source.coordinator.exceptions.PartitionUpdateException;
import org.opensearch.dataprepper.plugins.source.dynamodb.DynamoDBSourceConfig;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.partition.StreamPartition;
import org.opensearch.dataprepper.plugins.source.dynamodb.coordination.state.StreamProgressState;
import org.opensearch.dataprepper.plugins.source.dynamodb.model.ShardCheckpointStatus;
import org.opensearch.dataprepper.plugins.source.dynamodb.stream.ShardNotTrackedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ShardAcknowledgementManager {
    private static final Logger LOG = LoggerFactory.getLogger(ShardAcknowledgementManager.class);
    private static final String NULL_SEQUENCE_NUMBER = "null";
    private static final long WAIT_FOR_ACKNOWLEDGMENTS_TIMEOUT = 10L;
    static final Duration CHECKPOINT_INTERVAL = Duration.ofMinutes(3L);
    private final DynamoDBSourceConfig dynamoDBSourceConfig;
    private final Map<StreamPartition, ConcurrentLinkedQueue<ShardCheckpointStatus>> checkpoints = new ConcurrentHashMap<StreamPartition, ConcurrentLinkedQueue<ShardCheckpointStatus>>();
    private final ConcurrentHashMap<StreamPartition, ConcurrentHashMap<String, ShardCheckpointStatus>> ackStatuses = new ConcurrentHashMap();
    private final AcknowledgementSetManager acknowledgementSetManager;
    private final EnhancedSourceCoordinator sourceCoordinator;
    private final ExecutorService executorService;
    private final List<StreamPartition> partitionsToRemove;
    private final List<StreamPartition> partitionsToGiveUp;
    private boolean shutdownTriggered;
    private Instant lastCheckpointTime;
    private Lock lock;

    ShardAcknowledgementManager(AcknowledgementSetManager acknowledgementSetManager, EnhancedSourceCoordinator sourceCoordinator, DynamoDBSourceConfig dynamoDBSourceConfig, Consumer<StreamPartition> stopWorkerConsumer) {
        this(acknowledgementSetManager, sourceCoordinator, dynamoDBSourceConfig, ShardAcknowledgementManager.createExecutorService(), stopWorkerConsumer);
    }

    private static ExecutorService createExecutorService() {
        return Executors.newSingleThreadExecutor((ThreadFactory)BackgroundThreadFactory.defaultExecutorThreadFactory((String)"dynamodb-shard-ack-monitor"));
    }

    @VisibleForTesting
    ShardAcknowledgementManager(AcknowledgementSetManager acknowledgementSetManager, EnhancedSourceCoordinator sourceCoordinator, DynamoDBSourceConfig dynamoDBSourceConfig, ExecutorService executorService, Consumer<StreamPartition> stopWorkerConsumer) {
        this.executorService = executorService;
        this.acknowledgementSetManager = acknowledgementSetManager;
        this.sourceCoordinator = sourceCoordinator;
        this.dynamoDBSourceConfig = dynamoDBSourceConfig;
        this.partitionsToRemove = Collections.synchronizedList(new ArrayList());
        this.partitionsToGiveUp = Collections.synchronizedList(new ArrayList());
        this.lastCheckpointTime = Instant.now();
        this.lock = new ReentrantLock();
        executorService.submit(() -> this.monitorAcknowledgments(stopWorkerConsumer));
    }

    void monitorAcknowledgments(Consumer<StreamPartition> stopWorkerConsumer) {
        boolean exit;
        while (!Thread.currentThread().isInterrupted() && !(exit = this.runMonitorAcknowledgmentLoop(stopWorkerConsumer))) {
            try {
                Thread.sleep(2000L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        LOG.info("Exiting acknowledgment manager");
    }

    boolean runMonitorAcknowledgmentLoop(Consumer<StreamPartition> stopWorkerConsumer) {
        this.removePartitions();
        if (this.shutdownTriggered) {
            LOG.info("Shutdown was triggered giving up partitions and exiting cleanly");
            for (StreamPartition streamPartition : this.checkpoints.keySet()) {
                this.sourceCoordinator.giveUpPartition((EnhancedSourcePartition)streamPartition);
            }
            return true;
        }
        for (StreamPartition streamPartition : this.checkpoints.keySet()) {
            try {
                StreamProgressState streamProgressState = streamPartition.getProgressState().orElseThrow();
                ConcurrentLinkedQueue<ShardCheckpointStatus> checkpointStatuses = this.checkpoints.get((Object)streamPartition);
                ShardCheckpointStatus latestCheckpointForShard = null;
                boolean gaveUpPartition = false;
                while (!checkpointStatuses.isEmpty()) {
                    this.updateOwnershipForAllShardPartitions();
                    if (checkpointStatuses.peek().isPositiveAcknowledgement()) {
                        latestCheckpointForShard = checkpointStatuses.poll();
                        if (latestCheckpointForShard == null) continue;
                        this.ackStatuses.get((Object)streamPartition).remove(latestCheckpointForShard.getSequenceNumber());
                        continue;
                    }
                    if (!checkpointStatuses.peek().isNegativeAcknowledgement() && !checkpointStatuses.peek().isExpired(this.dynamoDBSourceConfig.getShardAcknowledgmentTimeout())) break;
                    this.handleFailure(streamPartition, streamProgressState, latestCheckpointForShard);
                    gaveUpPartition = true;
                    if (checkpointStatuses.peek().isNegativeAcknowledgement()) {
                        LOG.warn("Received negative acknowledgment for partition {} with sequence number {}, giving up partition", (Object)streamPartition.getPartitionKey(), (Object)checkpointStatuses.peek().getSequenceNumber());
                        break;
                    }
                    LOG.warn("Acknowledgment timed out for partition {} with sequence number {}, giving up partition", (Object)streamPartition.getPartitionKey(), (Object)checkpointStatuses.peek().getSequenceNumber());
                    break;
                }
                if (!gaveUpPartition) {
                    this.updateOwnershipForAllShardPartitions();
                }
                if (gaveUpPartition || latestCheckpointForShard == null) continue;
                if (latestCheckpointForShard.isFinalAcknowledgmentForPartition()) {
                    this.handleCompletedShard(streamPartition);
                    continue;
                }
                if (this.partitionsToRemove.contains((Object)streamPartition)) continue;
                streamProgressState.setSequenceNumber(Objects.equals(latestCheckpointForShard.getSequenceNumber(), NULL_SEQUENCE_NUMBER) ? null : latestCheckpointForShard.getSequenceNumber());
                try {
                    this.sourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)streamPartition, this.dynamoDBSourceConfig.getShardAcknowledgmentTimeout());
                }
                catch (PartitionUpdateException e) {
                    LOG.warn("Failed to checkpoint shard {}, stop processing shard. This shard will be processed by another worker.", (Object)streamPartition.getPartitionKey());
                    this.partitionsToRemove.add(streamPartition);
                }
                LOG.debug("Checkpointed shard {} with latest sequence number acknowledged {}", (Object)streamPartition.getShardId(), (Object)latestCheckpointForShard.getSequenceNumber());
            }
            catch (Exception e) {
                LOG.error(DataPrepperMarkers.NOISY, "Received exception while monitoring acknowledgments for stream partition {}, stop processing shard", (Object)streamPartition.getPartitionKey(), (Object)e);
                if (this.partitionsToRemove.contains((Object)streamPartition)) continue;
                this.partitionsToRemove.add(streamPartition);
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public AcknowledgementSet createAcknowledgmentSet(StreamPartition streamPartition, String sequenceNumber, boolean isFinalSetForPartition) {
        String sequenceNumberNoNull = sequenceNumber == null ? NULL_SEQUENCE_NUMBER : sequenceNumber;
        ShardCheckpointStatus shardCheckpointStatus = new ShardCheckpointStatus(sequenceNumber, Instant.now().toEpochMilli(), isFinalSetForPartition);
        this.lock.lock();
        try {
            ConcurrentLinkedQueue<ShardCheckpointStatus> queue = this.checkpoints.get((Object)streamPartition);
            if (queue == null) {
                throw new ShardNotTrackedException("The shard {} is not being tracked anymore, stop reading from shard");
            }
            queue.add(shardCheckpointStatus);
        }
        finally {
            this.lock.unlock();
        }
        this.ackStatuses.computeIfAbsent(streamPartition, segment -> new ConcurrentHashMap());
        this.ackStatuses.get((Object)streamPartition).put(sequenceNumberNoNull, shardCheckpointStatus);
        return this.acknowledgementSetManager.create(result -> {
            if (this.ackStatuses.containsKey((Object)streamPartition) && this.ackStatuses.get((Object)streamPartition).containsKey(sequenceNumberNoNull)) {
                ShardCheckpointStatus ackCheckpointStatus = this.ackStatuses.get((Object)streamPartition).get(sequenceNumberNoNull);
                ackCheckpointStatus.setAcknowledgedTimestamp(Instant.now().toEpochMilli());
                if (result.booleanValue()) {
                    LOG.debug("Received acknowledgment of completion from sink for partition {} with sequence number {}", (Object)streamPartition.getPartitionKey(), (Object)sequenceNumberNoNull);
                    ackCheckpointStatus.setAcknowledged(ShardCheckpointStatus.AcknowledgmentStatus.POSITIVE_ACK);
                } else {
                    LOG.debug(DataPrepperMarkers.NOISY, "Negative acknowledgment received for partition {} with sequence number {}", (Object)streamPartition.getPartitionKey(), (Object)sequenceNumberNoNull);
                    ackCheckpointStatus.setAcknowledged(ShardCheckpointStatus.AcknowledgmentStatus.NEGATIVE_ACK);
                }
            }
        }, this.dynamoDBSourceConfig.getShardAcknowledgmentTimeout());
    }

    void updateOwnershipForAllShardPartitions() {
        if (Duration.between(this.lastCheckpointTime, Instant.now()).compareTo(CHECKPOINT_INTERVAL) > 0) {
            for (StreamPartition streamPartition : this.checkpoints.keySet()) {
                if (this.partitionsToRemove.contains((Object)streamPartition)) continue;
                try {
                    this.sourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)streamPartition, this.dynamoDBSourceConfig.getShardAcknowledgmentTimeout());
                }
                catch (PartitionUpdateException e) {
                    LOG.warn(DataPrepperMarkers.NOISY, "Failed to update progress state for shard {}, will stop tracking this shard as someone else owns it", (Object)streamPartition.getShardId());
                    this.partitionsToRemove.add(streamPartition);
                }
            }
            this.lastCheckpointTime = Instant.now();
        }
    }

    private void handleFailure(StreamPartition streamPartition, StreamProgressState streamProgressState, ShardCheckpointStatus latestCheckpointForShard) {
        if (latestCheckpointForShard != null) {
            streamProgressState.setSequenceNumber(latestCheckpointForShard.getSequenceNumber());
            this.sourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)streamPartition, this.dynamoDBSourceConfig.getShardAcknowledgmentTimeout());
        }
        this.markPartitionForRemoval(streamPartition);
    }

    private void handleCompletedShard(StreamPartition streamPartition) {
        this.sourceCoordinator.completePartition((EnhancedSourcePartition)streamPartition);
        this.partitionsToRemove.add(streamPartition);
        this.partitionsToGiveUp.remove((Object)streamPartition);
        LOG.info("Received all acknowledgments for partition {}, marking partition as completed", (Object)streamPartition.getPartitionKey());
    }

    public void shutdown() {
        this.shutdownTriggered = true;
        this.executorService.shutdown();
        try {
            if (!this.executorService.awaitTermination(10L, TimeUnit.MINUTES)) {
                this.executorService.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.executorService.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    private void removePartitions() {
        this.lock.lock();
        try {
            this.partitionsToRemove.forEach(streamPartition -> {
                this.checkpoints.remove(streamPartition);
                this.ackStatuses.remove(streamPartition);
            });
            this.partitionsToGiveUp.forEach(partition -> {
                try {
                    this.sourceCoordinator.giveUpPartition((EnhancedSourcePartition)partition);
                    LOG.info("Gave up partition for shard {}", (Object)partition.getShardId());
                }
                catch (PartitionUpdateException e) {
                    LOG.warn("Received exception giving up shard {}, this shard will be reprocessed once the ownership timeout expires.", (Object)partition.getShardId());
                }
            });
            this.partitionsToRemove.clear();
            this.partitionsToGiveUp.clear();
        }
        finally {
            this.lock.unlock();
        }
    }

    public void giveUpPartition(StreamPartition streamPartition) {
        if (this.isStillTrackingShard(streamPartition)) {
            LOG.debug("Adding partition {} to give up list", (Object)streamPartition.getPartitionKey());
            this.partitionsToGiveUp.add(streamPartition);
            this.partitionsToRemove.add(streamPartition);
        }
    }

    public boolean isExportDone(StreamPartition streamPartition) {
        Optional globalPartition = this.sourceCoordinator.getPartition(streamPartition.getStreamArn());
        return globalPartition.isPresent();
    }

    void startUpdatingOwnershipForShard(StreamPartition streamPartition) {
        this.lock.lock();
        try {
            this.checkpoints.computeIfAbsent(streamPartition, segment -> new ConcurrentLinkedQueue());
        }
        finally {
            this.lock.unlock();
        }
    }

    boolean isStillTrackingShard(StreamPartition streamPartition) {
        return this.isStillTrackingShardInternal(streamPartition);
    }

    private boolean isStillTrackingShardInternal(StreamPartition streamPartition) {
        return !this.partitionsToRemove.contains((Object)streamPartition) && this.checkpoints.containsKey((Object)streamPartition);
    }

    private void markPartitionForRemoval(StreamPartition streamPartition) {
        if (!this.partitionsToRemove.contains((Object)streamPartition)) {
            this.partitionsToRemove.add(streamPartition);
            this.partitionsToGiveUp.add(streamPartition);
        }
    }
}

