/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.mongo.export;

import io.micrometer.core.instrument.Counter;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.source.coordinator.PartitionIdentifier;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourcePartition;
import org.opensearch.dataprepper.plugins.mongo.coordination.partition.DataQueryPartition;
import org.opensearch.dataprepper.plugins.mongo.coordination.partition.ExportPartition;
import org.opensearch.dataprepper.plugins.mongo.coordination.partition.GlobalState;
import org.opensearch.dataprepper.plugins.mongo.coordination.state.DataQueryProgressState;
import org.opensearch.dataprepper.plugins.mongo.coordination.state.ExportProgressState;
import org.opensearch.dataprepper.plugins.mongo.export.MongoDBExportPartitionSupplier;
import org.opensearch.dataprepper.plugins.mongo.model.ExportLoadStatus;
import org.opensearch.dataprepper.plugins.mongo.model.PartitionIdentifierBatch;
import org.opensearch.dataprepper.plugins.mongo.model.StreamLoadStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ExportScheduler
implements Runnable {
    public static final String EXPORT_PREFIX = "EXPORT-";
    private static final Logger LOG = LoggerFactory.getLogger(ExportScheduler.class);
    private static final int DEFAULT_TAKE_LEASE_INTERVAL_MILLIS = 60000;
    private static final Duration DEFAULT_CLOSE_DURATION = Duration.ofMinutes(10L);
    private static final int DEFAULT_MAX_CLOSE_COUNT = 36;
    private static final String COMPLETED_STATUS = "Completed";
    private static final String FAILED_STATUS = "Failed";
    static final int DEFAULT_GET_PARTITION_BACKOFF_MILLIS = 3000;
    static final String EXPORT_JOB_SUCCESS_COUNT = "exportJobSuccess";
    static final String EXPORT_JOB_FAILURE_COUNT = "exportJobFailure";
    static final String EXPORT_PARTITION_QUERY_TOTAL_COUNT = "exportPartitionQueryTotal";
    static final String EXPORT_RECORDS_TOTAL_COUNT = "exportRecordsTotal";
    private final PluginMetrics pluginMetrics;
    private final EnhancedSourceCoordinator enhancedSourceCoordinator;
    private final MongoDBExportPartitionSupplier mongoDBExportPartitionSupplier;
    private final Counter exportJobSuccessCounter;
    private final Counter exportJobFailureCounter;
    private final Counter exportPartitionTotalCounter;
    private final Counter exportRecordsTotalCounter;

    public ExportScheduler(EnhancedSourceCoordinator enhancedSourceCoordinator, MongoDBExportPartitionSupplier mongoDBExportPartitionSupplier, PluginMetrics pluginMetrics) {
        this.enhancedSourceCoordinator = enhancedSourceCoordinator;
        this.mongoDBExportPartitionSupplier = mongoDBExportPartitionSupplier;
        this.pluginMetrics = pluginMetrics;
        this.exportJobSuccessCounter = pluginMetrics.counter(EXPORT_JOB_SUCCESS_COUNT);
        this.exportJobFailureCounter = pluginMetrics.counter(EXPORT_JOB_FAILURE_COUNT);
        this.exportPartitionTotalCounter = pluginMetrics.counter(EXPORT_PARTITION_QUERY_TOTAL_COUNT);
        this.exportRecordsTotalCounter = pluginMetrics.counter(EXPORT_RECORDS_TOTAL_COUNT);
    }

    @Override
    public void run() {
        LOG.info("Start running Export Scheduler");
        while (!Thread.currentThread().isInterrupted()) {
            try {
                Optional sourcePartition = this.enhancedSourceCoordinator.acquireAvailablePartition("EXPORT");
                if (sourcePartition.isPresent()) {
                    ExportPartition exportPartition = (ExportPartition)((Object)sourcePartition.get());
                    LOG.info("Acquired an export partition: {}", (Object)exportPartition.getPartitionKey());
                    String exportPartitionKey = EXPORT_PREFIX + exportPartition.getCollection();
                    Optional globalPartition = this.enhancedSourceCoordinator.getPartition(exportPartitionKey);
                    while (globalPartition.isEmpty()) {
                        LOG.warn("Wait for global partition to be created.");
                        Thread.sleep(3000L);
                        globalPartition = this.enhancedSourceCoordinator.getPartition(exportPartitionKey);
                    }
                    PartitionIdentifierBatch partitionIdentifierBatch = this.mongoDBExportPartitionSupplier.apply(exportPartition);
                    this.createDataQueryPartitions(Instant.now(), partitionIdentifierBatch.getPartitionIdentifiers(), (GlobalState)((Object)globalPartition.get()));
                    this.updateExportPartition(exportPartition, partitionIdentifierBatch);
                    if (partitionIdentifierBatch.isLastBatch()) {
                        this.completeExportPartition(exportPartition, partitionIdentifierBatch.getPartitionIdentifiers().isEmpty());
                        this.markTotalPartitionsAsComplete(exportPartition.getCollection());
                    }
                }
                try {
                    Thread.sleep(60000L);
                    continue;
                }
                catch (InterruptedException e) {
                    LOG.info("The ExportScheduler was interrupted while waiting to retry, stopping processing");
                }
            }
            catch (Exception e) {
                LOG.error("Received an exception during export from DocumentDB, backing off and retrying", (Throwable)e);
                try {
                    Thread.sleep(60000L);
                    continue;
                }
                catch (InterruptedException ex) {
                    LOG.info("The ExportScheduler was interrupted while waiting to retry, stopping processing");
                }
            }
            break;
        }
        LOG.warn("Export scheduler interrupted, looks like shutdown has triggered");
    }

    private boolean createDataQueryPartitions(Instant exportTime, List<PartitionIdentifier> partitionIdentifiers, GlobalState globalState) {
        AtomicLong totalQueries = new AtomicLong();
        partitionIdentifiers.forEach(partitionIdentifier -> {
            DataQueryProgressState progressState = new DataQueryProgressState();
            progressState.setExecutedQueries(0L);
            progressState.setLoadedRecords(0L);
            progressState.setStartTime(exportTime.toEpochMilli());
            totalQueries.getAndIncrement();
            DataQueryPartition partition = new DataQueryPartition(partitionIdentifier.getPartitionKey(), progressState);
            this.enhancedSourceCoordinator.createPartition((EnhancedSourcePartition)partition);
        });
        if (totalQueries.get() > 0L) {
            this.exportPartitionTotalCounter.increment((double)totalQueries.get());
            ExportLoadStatus exportLoadStatus = ExportLoadStatus.fromMap(globalState.getProgressState().get());
            totalQueries.getAndAdd(exportLoadStatus.getTotalPartitions());
            exportLoadStatus.setTotalPartitions(totalQueries.get());
            exportLoadStatus.setLastUpdateTimestamp(Instant.now().toEpochMilli());
            globalState.setProgressState(exportLoadStatus.toMap());
            this.enhancedSourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)globalState, null);
            return true;
        }
        return false;
    }

    private void updateExportPartition(ExportPartition exportPartition, PartitionIdentifierBatch partitionIdentifierBatch) {
        ExportProgressState state = exportPartition.getProgressState().get();
        if (partitionIdentifierBatch.getEndDocId() != null) {
            state.setLastEndDocId(partitionIdentifierBatch.getEndDocId());
            this.enhancedSourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)exportPartition, null);
        }
    }

    private void completeExportPartition(ExportPartition exportPartition, boolean emptyPartition) {
        this.exportJobSuccessCounter.increment();
        ExportProgressState state = exportPartition.getProgressState().get();
        state.setStatus(COMPLETED_STATUS);
        this.enhancedSourceCoordinator.completePartition((EnhancedSourcePartition)exportPartition);
        if (emptyPartition) {
            LOG.info("There are no records to export. Streaming can continue...");
            StreamLoadStatus streamLoadStatus = new StreamLoadStatus(Instant.now().toEpochMilli());
            this.enhancedSourceCoordinator.createPartition((EnhancedSourcePartition)new GlobalState("STREAM-" + exportPartition.getCollection(), streamLoadStatus.toMap()));
        }
    }

    private void markTotalPartitionsAsComplete(String collection) {
        String exportPartitionKey = EXPORT_PREFIX + collection;
        Optional globalPartition = this.enhancedSourceCoordinator.getPartition(exportPartitionKey);
        if (globalPartition.isEmpty()) {
            throw new RuntimeException("Wait for global partition to be created.");
        }
        GlobalState globalState = (GlobalState)((Object)globalPartition.get());
        ExportLoadStatus exportLoadStatus = ExportLoadStatus.fromMap(globalState.getProgressState().get());
        exportLoadStatus.setTotalParitionsComplete(true);
        exportLoadStatus.setLastUpdateTimestamp(Instant.now().toEpochMilli());
        globalState.setProgressState(exportLoadStatus.toMap());
        this.enhancedSourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)globalState, null);
    }

    private void closeExportPartitionWithError(ExportPartition exportPartition) {
        LOG.error("The export from DocumentDB failed, it will be retried");
        this.exportJobFailureCounter.increment();
        ExportProgressState exportProgressState = exportPartition.getProgressState().get();
        exportProgressState.setStatus(FAILED_STATUS);
        this.enhancedSourceCoordinator.closePartition((EnhancedSourcePartition)exportPartition, DEFAULT_CLOSE_DURATION, 36);
    }
}

