/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.source.rds.export;

import io.micrometer.core.instrument.Counter;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.stream.Collectors;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourceCoordinator;
import org.opensearch.dataprepper.model.source.coordinator.enhanced.EnhancedSourcePartition;
import org.opensearch.dataprepper.plugins.source.rds.configuration.EngineType;
import org.opensearch.dataprepper.plugins.source.rds.coordination.partition.DataFilePartition;
import org.opensearch.dataprepper.plugins.source.rds.coordination.partition.ExportPartition;
import org.opensearch.dataprepper.plugins.source.rds.coordination.partition.GlobalState;
import org.opensearch.dataprepper.plugins.source.rds.coordination.state.DataFileProgressState;
import org.opensearch.dataprepper.plugins.source.rds.coordination.state.ExportProgressState;
import org.opensearch.dataprepper.plugins.source.rds.export.ExportTaskManager;
import org.opensearch.dataprepper.plugins.source.rds.export.SnapshotManager;
import org.opensearch.dataprepper.plugins.source.rds.model.ExportObjectKey;
import org.opensearch.dataprepper.plugins.source.rds.model.ExportStatus;
import org.opensearch.dataprepper.plugins.source.rds.model.LoadStatus;
import org.opensearch.dataprepper.plugins.source.rds.model.SnapshotInfo;
import org.opensearch.dataprepper.plugins.source.rds.model.SnapshotStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Request;
import software.amazon.awssdk.services.s3.model.ListObjectsV2Response;
import software.amazon.awssdk.services.s3.model.S3Object;

public class ExportScheduler
implements Runnable {
    private static final Logger LOG = LoggerFactory.getLogger(ExportScheduler.class);
    private static final int DEFAULT_TAKE_LEASE_INTERVAL_MILLIS = 60000;
    static final Duration DEFAULT_CLOSE_DURATION = Duration.ofMinutes(10L);
    static final int DEFAULT_MAX_CLOSE_COUNT = 36;
    private static final int DEFAULT_CHECKPOINT_INTERVAL_MILLS = 300000;
    private static final int DEFAULT_CHECK_STATUS_INTERVAL_MILLS = 30000;
    private static final Duration DEFAULT_SNAPSHOT_STATUS_CHECK_TIMEOUT = Duration.ofMinutes(60L);
    static final String PARQUET_SUFFIX = ".parquet";
    static final String EXPORT_JOB_SUCCESS_COUNT = "exportJobSuccess";
    static final String EXPORT_JOB_FAILURE_COUNT = "exportJobFailure";
    static final String EXPORT_S3_OBJECTS_TOTAL_COUNT = "exportS3ObjectsTotal";
    static final String DOT_DELIMITER = ".";
    private final S3Client s3Client;
    private final PluginMetrics pluginMetrics;
    private final EnhancedSourceCoordinator sourceCoordinator;
    private final ExecutorService executor;
    private final ExportTaskManager exportTaskManager;
    private final SnapshotManager snapshotManager;
    private final Counter exportJobSuccessCounter;
    private final Counter exportJobFailureCounter;
    private final Counter exportS3ObjectsTotalCounter;
    private EngineType engineType;
    private volatile boolean shutdownRequested = false;

    public ExportScheduler(EnhancedSourceCoordinator sourceCoordinator, SnapshotManager snapshotManager, ExportTaskManager exportTaskManager, S3Client s3Client, PluginMetrics pluginMetrics) {
        this.pluginMetrics = pluginMetrics;
        this.sourceCoordinator = sourceCoordinator;
        this.s3Client = s3Client;
        this.executor = Executors.newCachedThreadPool();
        this.snapshotManager = snapshotManager;
        this.exportTaskManager = exportTaskManager;
        this.exportJobSuccessCounter = pluginMetrics.counter(EXPORT_JOB_SUCCESS_COUNT);
        this.exportJobFailureCounter = pluginMetrics.counter(EXPORT_JOB_FAILURE_COUNT);
        this.exportS3ObjectsTotalCounter = pluginMetrics.counter(EXPORT_S3_OBJECTS_TOTAL_COUNT);
    }

    @Override
    public void run() {
        LOG.debug("Start running Export Scheduler");
        while (!this.shutdownRequested && !Thread.currentThread().isInterrupted()) {
            try {
                Optional sourcePartition = this.sourceCoordinator.acquireAvailablePartition("EXPORT");
                if (sourcePartition.isPresent()) {
                    ExportPartition exportPartition = (ExportPartition)((Object)sourcePartition.get());
                    LOG.debug("Acquired an export partition: {}", (Object)exportPartition.getPartitionKey());
                    String exportTaskId = this.getOrCreateExportTaskId(exportPartition);
                    if (exportTaskId == null) {
                        LOG.error("The export to S3 failed, it will be retried");
                        this.closeExportPartitionWithError(exportPartition);
                    } else {
                        CheckExportStatusRunner checkExportStatusRunner = new CheckExportStatusRunner(this.sourceCoordinator, this.exportTaskManager, exportPartition);
                        CompletableFuture<String> checkStatus = CompletableFuture.supplyAsync(checkExportStatusRunner::call, this.executor);
                        checkStatus.whenComplete((BiConsumer)this.completeExport(exportPartition));
                    }
                }
                try {
                    Thread.sleep(60000L);
                    continue;
                }
                catch (InterruptedException e) {
                    LOG.info("The ExportScheduler was interrupted while waiting to retry, stopping processing");
                }
            }
            catch (Exception e) {
                LOG.error("Received an exception during export, backing off and retrying", (Throwable)e);
                try {
                    Thread.sleep(60000L);
                    continue;
                }
                catch (InterruptedException ex) {
                    LOG.info("The ExportScheduler was interrupted while waiting to retry, stopping processing");
                }
            }
            break;
        }
        LOG.warn("Export scheduler interrupted, looks like shutdown has triggered");
        this.executor.shutdownNow();
    }

    public void shutdown() {
        this.shutdownRequested = true;
    }

    private String getOrCreateExportTaskId(ExportPartition exportPartition) {
        ExportProgressState progressState = exportPartition.getProgressState().get();
        this.engineType = EngineType.fromString(progressState.getEngineType());
        if (progressState.getExportTaskId() != null) {
            LOG.info("Export task has already created for db {}", (Object)exportPartition.getDbIdentifier());
            return progressState.getExportTaskId();
        }
        LOG.info("Creating a new snapshot for db {}", (Object)exportPartition.getDbIdentifier());
        SnapshotInfo snapshotInfo = this.snapshotManager.createSnapshot(exportPartition.getDbIdentifier());
        if (snapshotInfo == null) {
            LOG.error("The snapshot failed to create. The export will be retried");
            return null;
        }
        LOG.info("Snapshot id is {}", (Object)snapshotInfo.getSnapshotId());
        progressState.setSnapshotId(snapshotInfo.getSnapshotId());
        this.sourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)exportPartition, null);
        String snapshotId = snapshotInfo.getSnapshotId();
        try {
            snapshotInfo = this.checkSnapshotStatus(snapshotId, DEFAULT_SNAPSHOT_STATUS_CHECK_TIMEOUT);
        }
        catch (Exception e) {
            LOG.warn("Check snapshot status for {} failed. The export will be retried", (Object)snapshotId, (Object)e);
            return null;
        }
        progressState.setSnapshotTime(snapshotInfo.getCreateTime().toEpochMilli());
        LOG.info("Creating an export task for db {} from snapshot {}", (Object)exportPartition.getDbIdentifier(), (Object)snapshotId);
        String exportTaskId = this.exportTaskManager.startExportTask(snapshotInfo.getSnapshotArn(), progressState.getIamRoleArn(), progressState.getBucket(), progressState.getPrefix(), progressState.getKmsKeyId(), progressState.getTables());
        if (exportTaskId == null) {
            LOG.error("The export task failed to create, it will be retried");
            return null;
        }
        LOG.info("Export task id is {}", (Object)exportTaskId);
        progressState.setExportTaskId(exportTaskId);
        this.sourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)exportPartition, null);
        return exportTaskId;
    }

    private void closeExportPartitionWithError(ExportPartition exportPartition) {
        this.exportJobFailureCounter.increment();
        ExportProgressState exportProgressState = exportPartition.getProgressState().get();
        exportProgressState.setExportTaskId(null);
        this.sourceCoordinator.closePartition((EnhancedSourcePartition)exportPartition, DEFAULT_CLOSE_DURATION, 36);
    }

    private SnapshotInfo checkSnapshotStatus(String snapshotId, Duration timeout) {
        Instant endTime = Instant.now().plus(timeout);
        LOG.debug("Start checking status of snapshot {}", (Object)snapshotId);
        while (Instant.now().isBefore(endTime)) {
            SnapshotInfo snapshotInfo = this.snapshotManager.checkSnapshotStatus(snapshotId);
            if (snapshotInfo != null) {
                String status = snapshotInfo.getStatus();
                if (SnapshotStatus.AVAILABLE.getStatusName().equals(status)) {
                    LOG.info("Snapshot {} is available.", (Object)snapshotId);
                    return snapshotInfo;
                }
            }
            LOG.debug("Snapshot {} is still creating. Wait and check later", (Object)snapshotId);
            try {
                Thread.sleep(30000L);
            }
            catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        throw new RuntimeException("Snapshot status check timed out.");
    }

    private BiConsumer<String, Throwable> completeExport(ExportPartition exportPartition) {
        return (status, ex) -> {
            if (ex != null) {
                LOG.warn("Check export status for {} failed", (Object)exportPartition.getPartitionKey(), ex);
                this.sourceCoordinator.giveUpPartition((EnhancedSourcePartition)exportPartition);
            } else {
                if (!ExportStatus.COMPLETE.name().equals(status)) {
                    LOG.error("Export failed with status {}", status);
                    this.closeExportPartitionWithError(exportPartition);
                    return;
                }
                LOG.info("Export for {} completed successfully", (Object)exportPartition.getPartitionKey());
                ExportProgressState state = exportPartition.getProgressState().get();
                String bucket = state.getBucket();
                String prefix = state.getPrefix();
                String exportTaskId = state.getExportTaskId();
                long snapshotTime = state.getSnapshotTime();
                List<String> dataFileObjectKeys = this.getDataFileObjectKeys(bucket, prefix, exportTaskId);
                this.createDataFilePartitions(bucket, exportTaskId, dataFileObjectKeys, snapshotTime, state.getPrimaryKeyMap());
                this.completeExportPartition(exportPartition);
            }
        };
    }

    private List<String> getDataFileObjectKeys(String bucket, String prefix, String exportTaskId) {
        LOG.debug("Fetching object keys for export data files.");
        ListObjectsV2Request.Builder requestBuilder = ListObjectsV2Request.builder().bucket(bucket).prefix(prefix + "/" + exportTaskId);
        ArrayList<String> objectKeys = new ArrayList<String>();
        ListObjectsV2Response response = null;
        do {
            String nextToken = response == null ? null : response.nextContinuationToken();
            response = this.s3Client.listObjectsV2((ListObjectsV2Request)requestBuilder.continuationToken(nextToken).build());
            objectKeys.addAll(response.contents().stream().map(S3Object::key).filter(key -> key.endsWith(PARQUET_SUFFIX)).collect(Collectors.toList()));
        } while (response.isTruncated().booleanValue());
        return objectKeys;
    }

    private void createDataFilePartitions(String bucket, String exportTaskId, List<String> dataFileObjectKeys, long snapshotTime, Map<String, List<String>> primaryKeyMap) {
        LOG.info("Total of {} data files generated for export {}", (Object)dataFileObjectKeys.size(), (Object)exportTaskId);
        AtomicInteger totalFiles = new AtomicInteger();
        for (String objectKey : dataFileObjectKeys) {
            DataFileProgressState progressState = new DataFileProgressState();
            ExportObjectKey exportObjectKey = ExportObjectKey.fromString(objectKey);
            progressState.setEngineType(this.engineType.toString());
            progressState.setSourceDatabase(exportObjectKey.getDatabaseName());
            progressState.setSourceSchema(exportObjectKey.getSchemaName());
            progressState.setSourceTable(exportObjectKey.getTableName());
            progressState.setSnapshotTime(snapshotTime);
            progressState.setPrimaryKeyMap(primaryKeyMap);
            DataFilePartition dataFilePartition = new DataFilePartition(exportTaskId, bucket, objectKey, Optional.of(progressState));
            this.sourceCoordinator.createPartition((EnhancedSourcePartition)dataFilePartition);
            totalFiles.getAndIncrement();
        }
        this.exportS3ObjectsTotalCounter.increment((double)totalFiles.get());
        LoadStatus loadStatus = new LoadStatus(totalFiles.get(), 0);
        this.sourceCoordinator.createPartition((EnhancedSourcePartition)new GlobalState(exportTaskId, loadStatus.toMap()));
    }

    private void completeExportPartition(ExportPartition exportPartition) {
        this.exportJobSuccessCounter.increment();
        ExportProgressState progressState = exportPartition.getProgressState().get();
        progressState.setStatus("Completed");
        this.sourceCoordinator.completePartition((EnhancedSourcePartition)exportPartition);
    }

    static class CheckExportStatusRunner
    implements Callable<String> {
        private final EnhancedSourceCoordinator sourceCoordinator;
        private final ExportTaskManager exportTaskManager;
        private final ExportPartition exportPartition;

        CheckExportStatusRunner(EnhancedSourceCoordinator sourceCoordinator, ExportTaskManager exportTaskManager, ExportPartition exportPartition) {
            this.sourceCoordinator = sourceCoordinator;
            this.exportTaskManager = exportTaskManager;
            this.exportPartition = exportPartition;
        }

        @Override
        public String call() {
            return this.checkExportStatus(this.exportPartition);
        }

        private String checkExportStatus(ExportPartition exportPartition) {
            long lastCheckpointTime = System.currentTimeMillis();
            String exportTaskId = exportPartition.getProgressState().get().getExportTaskId();
            LOG.debug("Start checking the status of export {}", (Object)exportTaskId);
            while (true) {
                if (System.currentTimeMillis() - lastCheckpointTime > 300000L) {
                    this.sourceCoordinator.saveProgressStateForPartition((EnhancedSourcePartition)exportPartition, null);
                    lastCheckpointTime = System.currentTimeMillis();
                }
                String status = this.exportTaskManager.checkExportStatus(exportTaskId);
                LOG.debug("Current export status is {}.", (Object)status);
                if (ExportStatus.isTerminal(status)) {
                    LOG.info("Export {} is completed with final status {}", (Object)exportTaskId, (Object)status);
                    return status;
                }
                LOG.debug("Export {} is still running in progress. Wait and check later", (Object)exportTaskId);
                try {
                    Thread.sleep(30000L);
                }
                catch (InterruptedException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }
}

