/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.aws.sqs.common;

import com.linecorp.armeria.client.retry.Backoff;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSet;
import org.opensearch.dataprepper.model.acknowledgements.AcknowledgementSetManager;
import org.opensearch.dataprepper.plugins.aws.sqs.common.exception.SqsRetriesExhaustedException;
import org.opensearch.dataprepper.plugins.aws.sqs.common.metrics.SqsMetrics;
import org.opensearch.dataprepper.plugins.aws.sqs.common.model.SqsOptions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.sqs.SqsClient;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequest;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchRequestEntry;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchResponse;
import software.amazon.awssdk.services.sqs.model.DeleteMessageBatchResultEntry;
import software.amazon.awssdk.services.sqs.model.Message;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageRequest;
import software.amazon.awssdk.services.sqs.model.ReceiveMessageResponse;
import software.amazon.awssdk.services.sqs.model.SqsException;
import software.amazon.awssdk.services.sts.model.StsException;

public class SqsService {
    private static final Logger LOG = LoggerFactory.getLogger(SqsService.class);
    static final Duration END_TO_END_ACK_TIME_OUT = Duration.ofSeconds(10L);
    private final SqsMetrics sqsMetrics;
    private final SqsClient sqsClient;
    private final Backoff backoff;
    private int failedAttemptCount;

    public SqsService(SqsMetrics sqsMetrics, SqsClient sqsClient, Backoff backoff) {
        this.sqsMetrics = sqsMetrics;
        this.sqsClient = sqsClient;
        this.backoff = backoff;
    }

    public ReceiveMessageRequest createReceiveMessageRequest(SqsOptions sqsOptions) {
        return (ReceiveMessageRequest)ReceiveMessageRequest.builder().queueUrl(sqsOptions.getSqsUrl()).maxNumberOfMessages(sqsOptions.getMaximumMessages()).visibilityTimeout(SqsService.getTimeOutValueByDuration(sqsOptions.getVisibilityTimeout())).waitTimeSeconds(SqsService.getTimeOutValueByDuration(sqsOptions.getWaitTime())).messageAttributeNames(new String[]{"All"}).build();
    }

    private static Integer getTimeOutValueByDuration(Duration duration) {
        if (Objects.nonNull(duration)) {
            return (int)duration.toSeconds();
        }
        return null;
    }

    public List<Message> getMessagesFromSqs(SqsOptions sqsOptions) {
        try {
            ReceiveMessageRequest receiveMessageRequest = this.createReceiveMessageRequest(sqsOptions);
            ReceiveMessageResponse receiveMessageResponse = this.sqsClient.receiveMessage(receiveMessageRequest);
            return receiveMessageResponse.messages();
        }
        catch (SqsException | StsException e) {
            LOG.error("Error reading from SQS: {}. Retrying with exponential backoff.", (Object)e.getMessage());
            this.sqsMetrics.getSqsReceiveMessagesFailedCounter().increment();
            this.applyBackoff();
            return Collections.emptyList();
        }
    }

    public void applyBackoff() {
        long delayMillis;
        if ((delayMillis = this.backoff.nextDelayMillis(++this.failedAttemptCount)) < 0L) {
            Thread.currentThread().interrupt();
            throw new SqsRetriesExhaustedException("SQS retries exhausted. Make sure that SQS configuration is valid, SQS queue exists, and IAM role has required permissions.");
        }
        Duration delayDuration = Duration.ofMillis(delayMillis);
        LOG.info("Pausing SQS processing for {}.{} seconds due to an error in processing.", (Object)delayDuration.getSeconds(), (Object)delayDuration.toMillisPart());
        try {
            Thread.sleep(delayMillis);
        }
        catch (InterruptedException e) {
            LOG.error("Thread is interrupted while polling SQS with retry.", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    public void deleteMessagesFromQueue(List<DeleteMessageBatchRequestEntry> deleteMsgBatchReqList, String queueUrl) {
        block2: {
            try {
                DeleteMessageBatchResponse deleteMessageBatchResponse = this.sqsClient.deleteMessageBatch((DeleteMessageBatchRequest)DeleteMessageBatchRequest.builder().queueUrl(queueUrl).entries(deleteMsgBatchReqList).build());
                this.updateMetricsForDeletedMessages(deleteMessageBatchResponse);
                this.updateMetricsForUnDeletedMessages(deleteMessageBatchResponse);
            }
            catch (Exception e) {
                int failedMessageCount = deleteMsgBatchReqList.size();
                this.sqsMetrics.getSqsMessagesDeleteFailedCounter().increment((double)failedMessageCount);
                LOG.error("Failed to delete {} messages from SQS due to {}.", (Object)failedMessageCount, (Object)e.getMessage());
                if (!(e instanceof StsException)) break block2;
                this.applyBackoff();
            }
        }
    }

    private void updateMetricsForDeletedMessages(DeleteMessageBatchResponse deleteMessageBatchResponse) {
        int deletedMessagesCount;
        if (deleteMessageBatchResponse.hasSuccessful() && (deletedMessagesCount = deleteMessageBatchResponse.successful().size()) > 0) {
            String successfullyDeletedMessages = deleteMessageBatchResponse.successful().stream().map(DeleteMessageBatchResultEntry::id).collect(Collectors.joining(", "));
            LOG.info("Deleted {} messages from SQS. [{}]", (Object)deletedMessagesCount, (Object)successfullyDeletedMessages);
            this.sqsMetrics.getSqsMessagesDeletedCounter().increment((double)deletedMessagesCount);
        }
    }

    private void updateMetricsForUnDeletedMessages(DeleteMessageBatchResponse deleteMessageBatchResponse) {
        if (deleteMessageBatchResponse.hasFailed()) {
            int failedDeleteCount = deleteMessageBatchResponse.failed().size();
            this.sqsMetrics.getSqsMessagesDeleteFailedCounter().increment();
            if (LOG.isErrorEnabled()) {
                String failedMessages = deleteMessageBatchResponse.failed().stream().map(failed -> this.toString()).collect(Collectors.joining(", "));
                LOG.error("Failed to delete {} messages from SQS with errors: [{}].", (Object)failedDeleteCount, (Object)failedMessages);
            }
        }
    }

    public List<DeleteMessageBatchRequestEntry> getDeleteMessageBatchRequestEntryList(List<Message> messages) {
        ArrayList<DeleteMessageBatchRequestEntry> deleteMsgBatchReqList = new ArrayList<DeleteMessageBatchRequestEntry>(messages.size());
        messages.forEach(message -> deleteMsgBatchReqList.add((DeleteMessageBatchRequestEntry)DeleteMessageBatchRequestEntry.builder().id(message.messageId()).receiptHandle(message.receiptHandle()).build()));
        return deleteMsgBatchReqList;
    }

    public AcknowledgementSet createAcknowledgementSet(String queueUrl, AcknowledgementSetManager acknowledgementSetManager, List<DeleteMessageBatchRequestEntry> waitingForAcknowledgements) {
        AcknowledgementSet acknowledgementSet = null;
        acknowledgementSet = acknowledgementSetManager.create(result -> {
            this.sqsMetrics.getAcknowledgementSetCallbackCounter().increment();
            if (result.booleanValue()) {
                this.deleteMessagesFromQueue(waitingForAcknowledgements, queueUrl);
            }
        }, END_TO_END_ACK_TIME_OUT);
        return acknowledgementSet;
    }
}

