/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.dataprepper.plugins.sink.s3;

import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.DistributionSummary;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import org.opensearch.dataprepper.metrics.PluginMetrics;
import org.opensearch.dataprepper.model.codec.OutputCodec;
import org.opensearch.dataprepper.model.event.Event;
import org.opensearch.dataprepper.model.record.Record;
import org.opensearch.dataprepper.model.sink.OutputCodecContext;
import org.opensearch.dataprepper.model.types.ByteCount;
import org.opensearch.dataprepper.plugins.sink.s3.S3SinkConfig;
import org.opensearch.dataprepper.plugins.sink.s3.ThresholdCheck;
import org.opensearch.dataprepper.plugins.sink.s3.accumulator.Buffer;
import org.opensearch.dataprepper.plugins.sink.s3.grouping.S3Group;
import org.opensearch.dataprepper.plugins.sink.s3.grouping.S3GroupManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3SinkService {
    private static final Logger LOG = LoggerFactory.getLogger(S3SinkService.class);
    public static final String OBJECTS_SUCCEEDED = "s3SinkObjectsSucceeded";
    public static final String OBJECTS_FAILED = "s3SinkObjectsFailed";
    public static final String NUMBER_OF_RECORDS_FLUSHED_TO_S3_SUCCESS = "s3SinkObjectsEventsSucceeded";
    public static final String NUMBER_OF_RECORDS_FLUSHED_TO_S3_FAILED = "s3SinkObjectsEventsFailed";
    private static final String CURRENT_S3_GROUPS = "s3SinkNumberOfGroups";
    static final String NUMBER_OF_GROUPS_FORCE_FLUSHED = "s3SinkObjectsForceFlushed";
    static final String S3_OBJECTS_SIZE = "s3SinkObjectSizeBytes";
    private final S3SinkConfig s3SinkConfig;
    private final Lock reentrantLock;
    private final int maxEvents;
    private final ByteCount maxBytes;
    private final Duration maxCollectionDuration;
    private final int maxRetries;
    private final Counter objectsSucceededCounter;
    private final Counter objectsFailedCounter;
    private final Counter numberOfRecordsSuccessCounter;
    private final Counter numberOfRecordsFailedCounter;
    private final DistributionSummary s3ObjectSizeSummary;
    private final Counter numberOfObjectsForceFlushed;
    private final OutputCodecContext codecContext;
    private final Duration retrySleepTime;
    private final S3GroupManager s3GroupManager;

    public S3SinkService(S3SinkConfig s3SinkConfig, OutputCodecContext codecContext, Duration retrySleepTime, PluginMetrics pluginMetrics, S3GroupManager s3GroupManager) {
        this.s3SinkConfig = s3SinkConfig;
        this.codecContext = codecContext;
        this.retrySleepTime = retrySleepTime;
        this.reentrantLock = new ReentrantLock();
        this.maxEvents = s3SinkConfig.getThresholdOptions().getEventCount();
        this.maxBytes = s3SinkConfig.getThresholdOptions().getMaximumSize();
        this.maxCollectionDuration = s3SinkConfig.getThresholdOptions().getEventCollectTimeOut();
        this.maxRetries = s3SinkConfig.getMaxUploadRetries();
        this.objectsSucceededCounter = pluginMetrics.counter(OBJECTS_SUCCEEDED);
        this.objectsFailedCounter = pluginMetrics.counter(OBJECTS_FAILED);
        this.numberOfRecordsSuccessCounter = pluginMetrics.counter(NUMBER_OF_RECORDS_FLUSHED_TO_S3_SUCCESS);
        this.numberOfRecordsFailedCounter = pluginMetrics.counter(NUMBER_OF_RECORDS_FLUSHED_TO_S3_FAILED);
        this.s3ObjectSizeSummary = pluginMetrics.summary(S3_OBJECTS_SIZE);
        this.numberOfObjectsForceFlushed = pluginMetrics.counter(NUMBER_OF_GROUPS_FORCE_FLUSHED);
        pluginMetrics.gauge(CURRENT_S3_GROUPS, (Object)s3GroupManager, S3GroupManager::getNumberOfGroups);
        this.s3GroupManager = s3GroupManager;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void output(Collection<Record<Event>> records) {
        if (records.isEmpty() && this.s3GroupManager.hasNoGroups()) {
            return;
        }
        ArrayList<Event> failedEvents = new ArrayList<Event>();
        Exception sampleException = null;
        this.reentrantLock.lock();
        try {
            ArrayList completableFutures = new ArrayList();
            for (Record<Event> record : records) {
                Event event = (Event)record.getData();
                try {
                    S3Group s3Group = this.s3GroupManager.getOrCreateGroupForEvent(event);
                    Buffer currentBuffer = s3Group.getBuffer();
                    OutputCodec codec = s3Group.getOutputCodec();
                    if (currentBuffer.getEventCount() == 0) {
                        codec.start(currentBuffer.getOutputStream(), event, this.codecContext);
                    }
                    codec.writeEvent(event, currentBuffer.getOutputStream());
                    int count = currentBuffer.getEventCount() + 1;
                    currentBuffer.setEventCount(count);
                    s3Group.addEventHandle(event.getEventHandle());
                    this.flushToS3IfNeeded(completableFutures, s3Group, false);
                }
                catch (Exception ex) {
                    if (sampleException == null) {
                        sampleException = ex;
                    }
                    failedEvents.add(event);
                }
            }
            for (S3Group s3Group : this.s3GroupManager.getS3GroupEntries()) {
                this.flushToS3IfNeeded(completableFutures, s3Group, false);
            }
            if (this.s3SinkConfig.getAggregateThresholdOptions() != null) {
                this.checkAggregateThresholdsAndFlushIfNeeded(completableFutures);
            }
            if (!completableFutures.isEmpty()) {
                try {
                    ((CompletableFuture)CompletableFuture.allOf(completableFutures.toArray(new CompletableFuture[0])).thenRun(() -> LOG.debug("All {} requests to S3 have completed", (Object)completableFutures.size()))).join();
                }
                catch (Exception e) {
                    LOG.warn("There was an exception while waiting for all requests to complete", (Throwable)e);
                }
            }
        }
        finally {
            this.reentrantLock.unlock();
        }
        if (!failedEvents.isEmpty()) {
            failedEvents.stream().map(Event::getEventHandle).forEach(eventHandle -> eventHandle.release(false));
            LOG.error("Unable to add {} events to buffer. Dropping these events. Sample exception provided.", (Object)failedEvents.size(), (Object)sampleException);
        }
    }

    private boolean flushToS3IfNeeded(List<CompletableFuture<?>> completableFutures, S3Group s3Group, boolean forceFlush) {
        LOG.trace("Flush to S3 check: currentBuffer.size={}, currentBuffer.events={}, currentBuffer.duration={}", new Object[]{s3Group.getBuffer().getSize(), s3Group.getBuffer().getEventCount(), s3Group.getBuffer().getDuration()});
        if (forceFlush || ThresholdCheck.checkThresholdExceed(s3Group.getBuffer(), this.maxEvents, this.maxBytes, this.maxCollectionDuration)) {
            this.s3GroupManager.removeGroup(s3Group);
            try {
                s3Group.getOutputCodec().complete(s3Group.getBuffer().getOutputStream());
                String s3Key = s3Group.getBuffer().getKey();
                LOG.info("Writing {} to S3 with {} events and size of {} bytes.", new Object[]{s3Key, s3Group.getBuffer().getEventCount(), s3Group.getBuffer().getSize()});
                Consumer<Boolean> consumeOnGroupCompletion = success -> {
                    if (success.booleanValue()) {
                        LOG.info("Successfully saved {} to S3.", (Object)s3Key);
                        this.numberOfRecordsSuccessCounter.increment((double)s3Group.getBuffer().getEventCount());
                        this.objectsSucceededCounter.increment();
                        this.s3ObjectSizeSummary.record((double)s3Group.getBuffer().getSize());
                        s3Group.releaseEventHandles(true);
                    } else {
                        LOG.error("Failed to save {} to S3.", (Object)s3Key);
                        this.numberOfRecordsFailedCounter.increment((double)s3Group.getBuffer().getEventCount());
                        this.objectsFailedCounter.increment();
                        s3Group.releaseEventHandles(false);
                    }
                };
                Optional<CompletableFuture<?>> completableFuture = s3Group.getBuffer().flushToS3(consumeOnGroupCompletion, this::handleFailures);
                completableFuture.ifPresent(completableFutures::add);
                return true;
            }
            catch (IOException e) {
                LOG.error("Exception while completing codec", (Throwable)e);
            }
        }
        return false;
    }

    private void handleFailures(Throwable e) {
        LOG.error("Exception occurred while uploading records to s3 bucket: {}", (Object)e.getMessage());
    }

    private void checkAggregateThresholdsAndFlushIfNeeded(List<CompletableFuture<?>> completableFutures) {
        long currentTotalGroupSize = this.s3GroupManager.recalculateAndGetGroupSize();
        LOG.debug("Total groups size is {} bytes", (Object)currentTotalGroupSize);
        long aggregateThresholdBytes = this.s3SinkConfig.getAggregateThresholdOptions().getMaximumSize().getBytes();
        double aggregateThresholdFlushRatio = this.s3SinkConfig.getAggregateThresholdOptions().getFlushCapacityRatio();
        if (currentTotalGroupSize >= aggregateThresholdBytes) {
            LOG.info("aggregate_threshold reached, the largest groups will be flushed until {} percent of the maximum size {} is remaining", (Object)(aggregateThresholdFlushRatio * 100.0), (Object)aggregateThresholdBytes);
            for (S3Group s3Group : this.s3GroupManager.getS3GroupsSortedBySize()) {
                LOG.info("Forcing a flush of object with key {} due to aggregate_threshold of {} bytes being reached", (Object)s3Group.getBuffer().getKey(), (Object)aggregateThresholdBytes);
                boolean flushed = this.flushToS3IfNeeded(completableFutures, s3Group, true);
                this.numberOfObjectsForceFlushed.increment();
                if (flushed) {
                    currentTotalGroupSize -= s3Group.getBuffer().getSize();
                }
                if (!((double)currentTotalGroupSize <= (double)aggregateThresholdBytes * aggregateThresholdFlushRatio)) continue;
                break;
            }
        }
    }
}

